//////////////////////////////////////////////////////////////////////////
// Plants.inc  Copyright(c) 2003, K.E. Ayers Designs                    //
//////////////////////////////////////////////////////////////////////////
// This file provides colors, shapes and macros for generating basic    //
// plant shapes, such as plain and flowering shrubs.                    //
//////////////////////////////////////////////////////////////////////////

// This file should only be included once!
#ifndef (_Plants_Inc_)
#declare _Plants_Inc_ = true;

//#declare _DEBUG_        = true;
//#declare _DEBUG_SEL_    = true;


#include "SceneConstants.inc"
#include "Textures.inc"


//////////////////////////////////////////////////////////////////////////
// Some leaf colors and textures.                                       //
//////////////////////////////////////////////////////////////////////////

#declare LeafColor          = color rgb < 0.0,   0.5,    0.1   >;
#declare DyingLeafColor     = color rgb < 0.6,   0.8,    0.4   >;
#declare DeadLeafColor      = color rgb < 0.7,   0.6,    0.5   >;
#declare FallLeafColor      = color rgb < 1.0,   0.6,    0.0   >;

#declare SmoothLeafFinish =
finish {
    ambient     Ambient
    diffuse     Diffuse
    specular    0.25
    roughness   0.001
}

#declare BumpyLeafNormal = normal { bumps 0.5 scale 0.1 }

#declare SmoothLeafTexture =
texture {
    pigment { LeafColor }
    finish { SmoothLeafFinish }
}

#declare BumpyLeafTexture =
texture {
    pigment { LeafColor }
    finish { SmoothLeafFinish }
    normal { BumpyLeafNormal }
}


//////////////////////////////////////////////////////////////////////////
// Some flower colors and textures.                                     //
//////////////////////////////////////////////////////////////////////////

// A petal texture that blends from light pink, at the center,
// to bright red at the ends.
#declare RedPetalTexture =
texture {
    pigment {
        gradient z turbulence 0.15
        color_map {
            [ 0.0   color rgb < 1.0, 0.9, 0.9 > ]
            [ 0.1   color rgb < 1.0, 0.9, 0.9 > ]
            [ 0.5   color rgb < 1.0, 0.2, 0.3 > ]
            [ 1.1   color rgb < 1.0, 0.2, 0.3 > ]
        }
    }
    finish { ambient Ambient diffuse Diffuse }
    scale 1.2
}

// A texture for the center "button" of a flower, which is mostly
// brown with yellow flecks.
#local FlowerButtonTexture =
texture {
    pigment {
        granite turbulence 0.5
        color_map {
            [ 0.0   color rgb < 0.3, 0.1, 0.0 > ]
            [ 0.45  color rgb < 0.3, 0.1, 0.0 > ]
            [ 0.5   color rgb < 0.8, 0.8, 0.5 > ]
            [ 0.55  color rgb < 0.3, 0.1, 0.0 > ]
            [ 1.0   color rgb < 0.3, 0.1, 0.0 > ]
        }
    }
    scale 0.2
}



//////////////////////////////////////////////////////////////////////////
// Defines the shape for a roughly heart-shaped leaf.  The resulting    //
// leaf rests on the X-Y plane, oriented as follows.                    //
//                                                                      //
//                          +Z                                          //
//                          ^                                           //
//                          |                                           //
//                      ..  |  ..  <-- Z = ~1/8                         //
//                     .  . | .  .                                      //
//                    .    .|.    .                                     //
//            -X <----.-----V-----.----> +X                             //
//                    .     |     .\                                    //
//                     .    |    .  +-- X = ~1/2                        //
//                      .   |   .                                       //
//                       .  |  .                                        //
//                        . | .                                         //
//                         .|.  <-- Z = -1                              //
//                          |                                           //
//                          |                                           //
//                          v                                           //
//                          -Z                                          //
//////////////////////////////////////////////////////////////////////////

#local LeafLobe =
sphere {
    < 0, 0, 0 >, 1
    scale < 0.5, 0.25, 1 >
    rotate 10 * z
    translate < 0, 0, 1 >
    rotate 10 * y
}

#declare BasicLeafShape =
union {
    object { LeafLobe }
    object { LeafLobe scale < -1, 1, 1 > }
}

#macro LeafShape()
    difference {
        object { BasicLeafShape }
        object {
            BasicLeafShape
            scale < 1.1, 1, 1.1 >
            translate -0.01 * y
        }
        rotate 9 * x
        translate -1.225 * z
        scale < 0.82, 1, 0.82 >
    }
#end    // #macro LeafShape


//////////////////////////////////////////////////////////////////////////
// Uses the basic heart-shaped leaf shape to construct a daisy-like     //
// flower.  The leaf shape is first translated so that it extends along //
// the positive Z-axis, with the "pointed end" at the origin.  Then,    //
// the specified petal pattern is applied before the petal is rotated   //
// into position.  Finally, a flattened spherical button is placed in   //
// the center.                                                          //
//////////////////////////////////////////////////////////////////////////

#macro DaisyFlower
        (
            PetalWidth,         // The X-scale factor to be applied to each
                                // petal.
            PetalSpacing,       // The Y-rotation angle for spacing the
                                // individual petals.
            PetalSlant,         // The angle, relative to horizontal, that
                                // each petal is tilted.
            PetalDistance,      // The distance each petal is translated
                                // away from the center.
            ButtonRadius,       // The radius for the central button.
            ButtonThickness,    // The Y-scale factor to be applied to the
                                // flower's central button.
            PetalTexture,
            ButtonTexture
        )

    #local Petal =
    object {
        LeafShape()
        scale < PetalWidth, 1, 1 >
        translate 1 * z
        texture { PetalTexture }
        rotate -PetalSlant * x
        translate PetalDistance * z
    }

    #local A = 0;

    union {

        #while (A < 360)
            object { Petal rotate A * y }
            #local A = A + PetalSpacing;
        #end

        intersection {
            sphere {
                < 0, 0, 0 >, ButtonRadius
                scale < 1, ButtonThickness, 1 >
            }
            plane { < 0, -1, 0 >, 0 }
            texture { ButtonTexture }
        }
    }

#end    // #macro DaisyFlower


//////////////////////////////////////////////////////////////////////////
// Utility macros.                                                      //
//////////////////////////////////////////////////////////////////////////

// Creates a random scale factor in the range (1 - Slop) to (1 + Slop).
#declare RSlop = seed(0);
#macro SlopScale(S) (rand(RSlop) * 2 * S) - S + 1 #end


// Selects a random element from an one-dimensional array of objects,
// with the following probabilities:
//
//      Objects[0]: 50%
//      Objects[0]: 25%
//      Objects[0]: 12.5%
//      etc.
#declare RSel = seed(1);

#macro SelectObject(ObjArray)

    #local N    = dimension_size(ObjArray, 1);
    #local R    = rand(RSel) * pow(2, N);
    #local I    = 0;
    #local P    = pow(2, N - I - 1);

    #ifdef (_DEBUG_SEL_)
    #debug concat("\nN=",str(N,0,0),", R=",str(R,0,3),"\n")
    #debug concat("  I=",str(I,0,0),", P=",str(P,0,0),"\n")
    #end

    #while ((I < (N - 1)) & (P > R))
        #local I = I + 1;
        #local P    = pow(2, N - I - 1);

        #ifdef (_DEBUG_SEL_)
        #debug concat("  I=",str(I,0,0),", P=",str(P,0,0),"\n")
        #end

    #end

    #ifdef (_DEBUG_SEL_)
    #debug concat("Selected: I=",str(I,0,0),"\n")
    #end

    object { ObjArray[I] }

#end    // #macro SelectArrayElement

//////////////////////////////////////////////////////////////////////////
// Creates a roughly conical "leafy bush".                              //
//////////////////////////////////////////////////////////////////////////

#macro LeafyBush
        (
            StalkHeight,        // The height of the main stalk.
            StalkBaseRad,       // The stalk's radius at the ground.
            StalkEndRad,        // The stalk's radius at its tip.
            StemHeight,         // The height of the first ring of stems
                                // above the stalk's base.
            StemLength,         // The length of the lowest leaf stem.
            StemLengthScale,    // The scale factor applied to the
                                // stem length for each successive
                                // ring of stems ... should be < 1.0
            StemRad,            // The radius of the lowest leaf stem.
            StemSpacing,        // The angular spacing of stems around
                                // the central stalk.
            StemAngle,          // The angle for the lowest ring of stems.
                                // 90-deg is horizontal.
            StemAngleScale,     // The scale factor applied to each
                                // successive ring of stems.
            LeafAngle,          // The angle, relative to the stem angle,
                                // for the lowest ring of leaves.  90-deg
                                // is perpendicular.
            LeafAngleScale,     // The scale factor applied to the leaf
                                // angle for each successive ring of leaves.
            Rings,              // The number of rings of stems created
                                // evenly up the length of the main stalk.
            RingStagger,        // The radial angle applied to the first
                                // stem in each successive ring of stems.
            TheLeaves,          // An array of objects to be used as the
                                // leaves.  The objects will be selected at
                                // random, with the probabilities set at:
                                //   TheLeaves[0]: 50%
                                //   TheLeaves[1]: 25%
                                //   TheLeaves[2]: 12.5%
                                //   etc.
            TheFlowers,         // An array of objects to be used as the
                                // flowers.  The objects will be selected at
                                // random, with the probabilities set at:
                                //   TheFlowers[0]: 50%
                                //   TheFlowers[1]: 25%
                                //   TheFlowers[2]: 12.5%
                                //   etc.
            FlowerFrequency,    // The frequency with which flowers replace
                                // leaves.
            Slop,               // All stem lengths and angles will be
                                // scaled by a random value ranging form
                                // (1 - Slop) to (1 + Slop).
            StalkTexture
        )

    #local R = seed(11);

    #if (Rings > 1)
        #local RngYSpc  = (StalkHeight - StemHeight) / (Rings - 1);
    #else
        #local RngYSpc  = 0;
    #end

    #local StmLen   = StemLength;
    #local StmRad   = StemRad;
    #local StmAng   = StemAngle;
    #local LfAng    = LeafAngle;
    #local LfScl    = 1.0;
    #local RngNum   = 0;

    #ifdef (_DEBUG_)
    #debug "\n========\nLeafyBush\n"
    #end

    union {
        // The stalk
        cone {
            < 0, 0, 0 >, StalkBaseRad,
            < 0, StalkHeight, 0 >, StalkEndRad
            texture { StalkTexture }
        }

        // Loop for each ring of leaf stems.
        #while (RngNum < Rings)
            #local StmNum   = 0;

            #ifdef (_DEBUG_)
            #debug concat("\nRngNum=", str(RngNum,0,0), "\n")
            #debug concat("  StmLen=", str(StmLen,0,3), "\n")
            #debug concat("  StmAng=", str(StmAng,0,3), "\n")
            #debug concat("   LfAng=", str(LfAng,0,3),  "\n\n")
            #end

            // Position the stems around 
            #while ((StmNum * StemSpacing) < 360)


                #local SEnd = SlopScale(Slop) * StmLen;
                #local SAng = SlopScale(Slop) * StmAng;
                #local LAng = 90 - (SlopScale(Slop) * LfAng);
                #local A    = ((SlopScale(Slop) * StemSpacing * StmNum))
                                + (RngNum * RingStagger);
                #local Y    = ((SlopScale(Slop) * RngYSpc * RngNum)) + StemHeight;

                // Select either a leaf or a flower.
                #if (rand(R) <= FlowerFrequency)
                    #local LfObj =
                        object { SelectObject(TheFlowers) scale LfScl }
                    #local SEnd = SEnd * 1.125;
                #else
                    #local LfObj =
                        object { SelectObject(TheLeaves) scale LfScl }
                #end

                #ifdef (_DEBUG_)
                #debug concat("  StmNum=", str(StmNum,0,0), "\n")
                #debug concat("    SEnd=", str(SEnd,0,3), "\n")
                #debug concat("    SAng=", str(SAng,0,3), "\n")
                #debug concat("    LAng=", str(LAng,0,3), "\n")
                #debug concat("       A=", str(A,0,3),    "\n")
                #debug concat("       Y=", str(Y,0,3),    "\n")
                #end

                // Create the stem and leaf/flower as a single unit,
                // which will then be positioned together.
                union {
                    // Create the stem.
                    cone {
                        < 0, 0, 0 >, StmRad, < 0, SEnd, 0 >, StmRad * 0.25
                        texture { StalkTexture }
                    }

                    // Position the leaf/flower at the end of the stem.
                    object {
                        LfObj
                        rotate LAng * x
                        translate SEnd * y
                    }

                    rotate -SAng * x
                    rotate A * y
                    translate Y * y
                }

                #local StmNum = StmNum + 1;

            #end    // Current ring.

            #local StmLen = StmLen * StemLengthScale;
            #local StmAng = StmAng * StemAngleScale;
            #local LfAng  = LfAng  * LeafAngleScale;
            #local LfScl  = LfScl  * StemLengthScale;
            #local RngNum = RngNum + 1;

        #end    // Rings.
    }

    #ifdef (_DEBUG_)
    #debug "\n========\n\n"
    #end

#end    // #macro LeafyBush


//////////////////////////////////////////////////////////////////////////
// Variously colored leaves.                                            //
//////////////////////////////////////////////////////////////////////////

// A dark green leaf.
#declare GreenLeaf =
object {
    LeafShape()
    texture { BumpyLeafTexture }
}

// A lighter green leaf.
#declare LightGreenLeaf =
object {
    LeafShape()
    texture {
        pigment { LeafColor * 1.25 }
        finish { SmoothLeafFinish }
        normal { BumpyLeafNormal }
    }
}

// A pale yellow-green leaf.
#declare DyingLeaf =
object {
    LeafShape()
    texture {
        pigment { DyingLeafColor }
        finish { ambient Ambient diffuse Diffuse }
        normal { BumpyLeafNormal }
    }
}

// A brown leaf.
#declare DeadLeaf =
object {
    LeafShape()
    texture {
        pigment { DeadLeafColor }
        finish { ambient Ambient diffuse Diffuse }
        normal { BumpyLeafNormal }
    }
}

// An orange leaf.
#declare FallLeaf =
object {
    LeafShape()
    texture {
        pigment { FallLeafColor }
        finish { ambient Ambient diffuse Diffuse }
        normal { BumpyLeafNormal }
    }
}


//////////////////////////////////////////////////////////////////////////
// Some basic flower shapes.                                            //
//////////////////////////////////////////////////////////////////////////

// Looks something like a red Impatien.
#declare RedImpatienFlower =
object {
    DaisyFlower(1.0, 40, 30, 0.1, 0.3, 0.5,
                RedPetalTexture, FlowerButtonTexture)
   
}

// Looks something like a red Shasta Daisy.
#declare RedShastaDaisyFlower =
object {
    DaisyFlower(0.5, 30, 45, 0.3, 0.4, 0.2, 
                RedPetalTexture, FlowerButtonTexture)
    scale 0.4
}


#end    // _Plants_Inc_

