// Persistence of Vision Ray Tracer Scene Description File
// File: threshold.pov
// Vers: 3.5
// Desc: IRTC surrealism entry
// Date: 8/24/03
// Auth: James Moore
//------------------------------------------
//------------------------------------------
// includes

#include "metals.inc"
#include "finish.inc"
#include "functions.inc"
#include "shapes.inc"
#include "rand.inc"
#include "math.inc"
#include "transforms.inc"

//------------------------------------------
// global settings & macros

global_settings {max_trace_level 10}

#macro square (p1, p2, p3, p4)
	triangle {p1, p2, p3}
	triangle {p2, p3, p4}
#end

#macro sb (p1, p2, p3, p4, p5, p6, p7, p8)
	union {
		triangle {p1, p2, p3}
		triangle {p3, p4, p2}
		triangle {p1, p2, p6}
		triangle {p1, p5, p6}
		triangle {p1, p3, p7}
		triangle {p1, p5, p7}
		triangle {p2, p4, p8}
		triangle {p2, p6, p8}
		triangle {p3, p4, p8}
		triangle {p3, p7, p8}
		triangle {p5, p6, p7}
		triangle {p7, p8, p6}
	}
#end

//------------------------------------------
// grass

#macro grass (loc, wid, ht, crv, tr)
	#local yy=0;
	#local step=ht/15;
	#local prevloc=loc;
	#while (yy<ht)
		#local xx1=wid*(ht-yy)/ht;
		#local xx2=wid*(ht-yy-step)/ht;
		#local zz=pow(yy,2)*crv;
		#local p1=vtransform(prevloc+x*-xx1, tr);
		#local p2=vtransform(prevloc+x*xx1, tr);
		#local p3=vtransform(prevloc+<-xx2, yy, zz>, tr);
		#local p4=vtransform(prevloc+<xx2, yy, zz>, tr);
		square (p1, p2, p3, p4)
		#local prevloc=prevloc+<0, yy, zz>;
		#local yy=yy+step;
	#end
#end

#macro clump(sd)
mesh {
	#local c=0;
	#local maxc=10;
	#while (c<maxc)
		#local loc=RRand(0, 0.15, sd);
		#local sc=VRand_In_Box(0.75, 1.5, sd);
		#local ang=RRand(0, 360, sd);
		#local crv=RRand(1, 3, sd);
		#local wid=RRand(0.05, 0.15, sd);
		#local ht=RRand(0.5-loc, 0.5, sd);
		grass(<0, 0, 0>, wid, ht, crv, transform {scale sc translate z*loc rotate y*ang})
		#local c=c+1;
	#end
	pigment {
		wrinkles
		scale 0.05
		color_map {
			[0 blue 0.05]
			[1 green 0.25]
		}
	}
	finish {specular 0.6 roughness 0.3}
}
#end

//------------------------------------------
//trees

#declare t_moss=
texture {
	pigment {
		wrinkles
		turbulence 0.3 omega 0.8
		scale 0.01
		color_map {
			[0.3 rgb 0.95]
			[0.5 rgbf 0.5]
			[0.8 rgb 0.95]
		}
	}
	normal {wrinkles 1 turbulence 0.3 omega 0.8 scale 0.01}
	finish {specular 0.1 roughness 0.4}
}
#declare t_tree1=
texture {
	pigment {
		granite
		scale 0.3
		turbulence 0.2
		color_map {
			[0 rgb <0.5, 0.3, 0.2>]
			[1 rgb <0.8, 0.5, 0.3>]
		}
	}
	normal {granite turbulence 0.2 omega 0.8 scale 0.1}
	finish {specular 0.2 roughness 0.4}
}
#declare t_tree2=
texture {
	pigment {
		granite
		scale 0.3
		turbulence 0.2
		color_map {
			[0 rgb <0.5, 0.3, 0.2>*0.2]
			[1 rgb <0.8, 0.5, 0.3>*0.2]
		}
	}
	normal {granite turbulence 0.2 omega 0.8 scale 0.1}
	finish {specular 0.3 roughness 0.25}
}
#declare t_tree=
texture {
	planar
	texture_map {
		[0.95 t_tree1]
		[0.99 t_tree2]
	}
}

#declare tree=array[10];
#declare tree[0]=#include "tree0.inc"
#declare tree[1]=#include "tree1.inc"
#declare tree[2]=#include "tree2.inc"
#declare tree[3]=#include "tree3.inc"
#declare tree[4]=#include "tree4.inc"
#declare tree[5]=#include "tree5.inc"
#declare tree[6]=#include "tree6.inc"
#declare tree[7]=#include "tree7.inc"
#declare tree[8]=#include "tree8.inc"
#declare tree[9]=#include "tree9.inc"

//------------------------------------------
//keyboard

#declare t_kbm=
texture {
	pigment {rgb <0.99, 0.98, 0.92>}
	normal {granite 0.02 scale 0.05}
	finish {
		specular 0.3
		roughness 0.2
		metallic
		ambient 0.2
	}
}

#declare t_light=
texture {
	pigment {rgb <0.5, 1, 0.5>}
	finish {ambient 1 diffuse 0}
}

#declare kb=
union {
	union {
		height_field {
			png "kbhf.png" smooth
			water_level 0.01	
			texture {
				pigment {
					image_pattern {png "kblabel.png" once interpolate 2}
					color_map {
						[0 rgb 0]
						[1 rgb <0.99, 0.98, 0.92>*0.95]
					}
					rotate x*90
				}
				normal {granite 0.02 scale 0.05}
				finish {
					specular 0.3
					roughness 0.2
					metallic
					ambient 0.2
				}
			}
			translate <-0.5, 0, -0.5>
			scale <1.5, 0.085, 0.7>
		}
		difference {
			Round_Box_Merge (<-0.77, -0.03, -0.53>, <0.82, 0.07, 0.53>, 0.02)
			union {
				box {<-0.725, 0, -0.44>, <0.225, 1, 0.24>}
				box {<0.47, 0, -0.44>, <0.725, 1, 0.24>}
				box {<0.25, 0, -0.44>, <0.447, 1, -0.29>}
				box {<0.315, 0, -0.43>, <0.385, 1, -0.163>}
				box {<0.25, 0, -0.038>, <0.447, 1, 0.24>}
				box {<-0.725, 0, 0.315>, <-0.66, 1, 0.442>}
				box {<-0.62, 0, 0.315>, <-0.37, 1, 0.442>}
				box {<-0.33, 0, 0.315>, <-0.077, 1, 0.442>}
				box {<-0.032, 0, 0.315>, <0.22, 1, 0.442>}
				box {<0.251, 0, 0.315>, <0.442, 1, 0.442>}
				translate y*0.068
				pigment {rgb 0.02}
			}
			box {<0.47, 0.069, 0.315>, <0.725, 1, 0.442>}
			scale <1, 1, 0.7>
			texture {t_kbm}
		}
		box {<-0.012, 0, -0.005>, <0.012, 0.005, 0.005>
			texture {t_light}
			translate <0.52, 0.069, 0.34>
			scale <1, 1, 0.7>
		}
		box {<-0.012, 0, -0.005>, <0.012, 0.005, 0.005>
			texture {t_light}
			translate <0.6, 0.069, 0.34>
			scale <1, 1, 0.7>
		}
		box {<-0.012, 0, -0.005>, <0.012, 0.005, 0.005>
			texture {t_light}
			translate <0.68, 0.069, 0.34>
			scale <1, 1, 0.7>
		}
		text {
			ttf "crystal.ttf" "Keyboard", 2, 0
			scale 0.04
			rotate x*90
			translate <0.52, 0.07, 0.38>
			scale <1, 1, 0.7>
			pigment {rgb 0.1}
			finish {specular 0.3 roughness 0.3 ambient 0.3}
		}
		sphere_sweep {
			#local rad=0.01;
			b_spline
			16,
			<0, 0.02, 0.3>, rad,
			<0, 0.02, 0.3>, rad,
			<0.05, 0.04, 0.4>, rad,
			<0, 0.07, 0.42>, rad,
			<-0.1, 0.08, 0.4>, rad,
			<0, 0.1+rad, 0.15>, rad,
			<-0.1, 0.09+rad, -0.2>, rad,
			<-0.25, 0.09+rad, -0.2>, rad,
			<-0.4, 0.11+rad, 0.1>, rad,
			<-0.6, 0.1+rad, 0>, rad,
			<-0.8, 0.07+rad, -0.1>, rad,
			<-0.85, 0.02, -0.05>, rad,
			<-1.2, 0.1, 0>, rad,
			<-1.3, 0.1, 0>, rad,
			<-1.4, 0.08, 0>, rad,
			<-1.5, 0.05, 0>, rad
			texture {t_kbm}
		}
		rotate x*-5
	}
	sphere_sweep {
		#local rad=0.01;
		cubic_spline
		10,
		vtransform (<-1.3, 0.1, 0>, transform {rotate x*-5}), rad,
		vtransform (<-1.4, 0.08, 0>, transform {rotate x*-5}), rad,
		vtransform (<-1.5, 0.05, 0>, transform {rotate x*-5}), rad,
		<-1.7, 0.005, 0>, rad,
		<-2, 0.005, -0.2>, rad,
		<-2.1, 0.005, -0.5>, rad,
		<-2, 0.005, -0.75>, rad,
		<-1.8, 0, -0.5>, rad,
		<-1.7, -0.2, -0.6>, rad,
		<-1.7, -0.2, -0.6>, rad
		texture {t_kbm}
	}
	#declare o_bottle=
	lathe {
		cubic_spline
		16,
		<0, -0.9>,
		<0, -0.9>,
		<0.45, -0.95>,
		<0.5, -0.95>,
		<0.55, -0.95>,
		<0.62, -0.9>,
		<0.65, -0.7>,
		<0.62, 0>,
		<0.65, 0.75>,
		<0.23, 0.9>,
		<0.25, 1>,
		<0.2, 1.2>,
		<0.22, 1.25>,
		<0.2, 1.3>,
		<0, 1.3>,
		<0, 1.3>
		sturm
		rotate z*90
		scale <1, 0.7, 0.7>
		translate y*0.155
	}
	blob {
		threshold 0.1
		#local bsd=seed(100);
		#local dc=0;
		#local maxdc=50;
		#while (dc<maxdc)
			#local px=RRand(-0.9, 0.8, bsd);
			#local pang=RRand(10, 150, bsd);
			#local tr_bd=transform {translate z*-1 rotate x*pang};
			#local start=vtransform(x*px, tr_bd);
			#local dpos=trace(o_bottle, start, x*px-start);
			sphere {0, 0.015, 1 scale <1, 1, 0.5> translate dpos-y*0.008}
			sphere {0, 0.01, 1 scale <1, 2, 0.5> translate dpos}
			sphere {0, 0.006, 1 scale <1, 3, 0.5> translate dpos+y*0.004}
			#local dc=dc+1;
		#end
		clipped_by {object {o_bottle inverse scale 0.999}}
		pigment {rgbt 1}
		finish {specular 0.8 roughness 0.003}
		interior {ior 1.33}
	}
	object {
		o_bottle
		clipped_by {cylinder {<-2, 0.165, 0>, <0, 0.165, 0>, 0.1 inverse}}
		texture {
			pigment {rgbf <0.95, 0.95, 1, 0.99>}
			finish {
				specular 0.8
				roughness 0.001
				reflection 0.1
				ambient 0
				diffuse 0
			}
		}
		texture {
			pigment {
				image_map {
					png "botlabel.png"
					map_type 2 once
					interpolate 2
				}
				rotate z*-90
				translate x*-0.5
				rotate y*180
				rotate x*25
			}
			finish {specular 0.2 roughness 0.3 ambient 0.2}
		}
		interior {
			ior 1.5
			caustics 0.8
		}
	}
	#declare tr_kb=
	transform {
		scale 0.4
		rotate y*35
		translate <0.55, 0, -1.3>
	}
	transform {tr_kb}
}

//------------------------------------------
//mouse

#declare mouse=
union {
	difference {
		union {
			bicubic_patch {
				type 0
				flatness 0.0001
				u_steps 5 v_steps 5
				<-0.09, 0, 0.2>, <-0.03, 0, 0.22>, <0.03, 0, 0.22>, <0.1, 0, 0.2>,
				<-0.11, 0, 0.1>, <-0.03, 0.05, 0.12>, <0.03, 0.05, 0.12>, <0.12, 0, 0.1>,
				<-0.1, 0, -0.12>, <-0.04, 0.08, -0.07>, <0.04, 0.08, -0.07>, <0.13, 0, -0.12>,
				<-0.1, 0, -0.18>, <-0.08, 0, -0.25>, <0.08, 0, -0.25>, <0.1, 0, -0.18>
			}
			prism {
				linear_sweep
				bezier_spline
				-0.04, 0, 16,
				<-0.09, 0.2>, <-0.03, 0.22>, <0.03, 0.22>, <0.1, 0.2>,
				<0.1, 0.2>, <0.12, 0.1>, <0.13, -0.12>, <0.1, -0.18>,
				<0.1, -0.18>, <0.08, -0.25>, <-0.08, -0.25>, <-0.1, -0.18>,
				<-0.1, -0.18>, <-0.1, -0.12>, <-0.11, 0.1>, <-0.09, 0.2>
				sturm
			}
			bicubic_patch {
				type 0
				flatness 0.0001
				u_steps 5 v_steps 5
				<-0.09, 0, 0.2>, <-0.03, 0, 0.22>, <0.03, 0, 0.22>, <0.1, 0, 0.2>,
				<-0.11, 0, 0.1>, <-0.1, -0.02, 0.12>, <0.11, -0.02, 0.12>, <0.12, 0, 0.1>,
				<-0.1, 0, -0.12>, <-0.09, -0.02, -0.07>, <0.12, -0.02, -0.07>, <0.13, 0, -0.12>,
				<-0.1, 0, -0.18>, <-0.08, 0, -0.25>, <0.08, 0, -0.25>, <0.1, 0, -0.18>
				translate y*-0.04
			}
		}
		difference {
			Round_Box_Merge(<-0.075, -0.001, 0.1>, <0.075, 1, 1>, 0.01)
			Round_Box_Merge(<-0.07, -1, 0.102>, <-0.0025, 1, 1>, 0.01)
			Round_Box_Merge(<0.07, -1, 0.102>, <0.0025, 1, 1>, 0.01)
		}
		cylinder {y*-0.03, y*-1, 0.03}
	}
	texture {t_kbm}
	scale <0.45, 0.5, 0.45>
}

//------------------------------------------
// monitor

#macro mon(r)
union {
	//front of box
	difference {
		union {
			Round_Box_Merge (<-0.6, -0.6, 0>, <0.6, 0.6, 1>, 0.02)
			sphere {<0.58, -0.58, 0.98>, 0.02}
			sphere {<0.58, 0.58, 0.98>, 0.02}
			sphere {<-0.58, -0.58, 0.98>, 0.02}
			sphere {<-0.58, 0.58, 0.98>, 0.02}

			cylinder {<0.58, -0.58, 0.98>, <0.3, -0.58, 1.5>, 0.02}
			cylinder {<0.58, 0.58, 0.98>, <0.3, 0.4, 1.5>, 0.02}
			cylinder {<-0.58, -0.58, 0.98>, <-0.3, -0.58, 1.5>, 0.02}
			cylinder {<-0.58, 0.58, 0.98>, <-0.3, 0.4, 1.5>, 0.02}
			
			triangle {<0.6, 0.58, 0.98>, <0.6, -0.58, 0.98>, <0.32, 0.4, 1.5>}
			triangle {<0.6, -0.58, 0.98>, <0.32, -0.58, 1.5>, <0.32, 0.4, 1.5>}
			
			triangle {<-0.6, 0.58, 0.98>, <-0.6, -0.58, 0.98>, <-0.32, 0.4, 1.5>}
			triangle {<-0.6, -0.58, 0.98>, <-0.32, -0.58, 1.5>, <-0.32, 0.4, 1.5>}
			
			triangle {<-0.58, 0.6, 0.98>, <0.58, 0.6, 0.98>, <-0.3, 0.42, 1.5>}
			triangle {<0.58, 0.6, 0.98>, <-0.3, 0.42, 1.5>, <0.3, 0.42, 1.5>}
			
			triangle {<-0.58, -0.6, 0.98>, <0.58, -0.6, 0.98>, <-0.3, -0.6, 1.5>}
			triangle {<0.58, -0.6, 0.98>, <-0.3, -0.6, 0.98>, <0.3, -0.6, 1.5>}
			
			scale <1, 1, 0.4>
		}
		Round_Box_Merge (<-0.5, -0.45, -0.1>, <0.5, 0.45, 0.1>, 0.02)
	}

	//screen
	intersection {
		Round_Box_Merge (<-0.499, -0.449, -1>, <0.499, 0.449, 0.1>, 0.02)
		sphere {0, 3 scale <1, 1, 0.3> translate z*0.92}
		#if (r)
			pigment {
				image_map {png "mon.png" interpolate 2 once}
				translate <-0.5, -0.5, 0>
				scale <1, 0.9, 1>
			}
		#else
			pigment {rgb 0.02}
		#end
		finish {
			specular 0.8
			roughness 0.001
			reflection 0.2
			ambient 0.8
		}
	}
	
	//back of box
	cylinder {<-0.5, -0.652, 0.38>, <-0.4, -0.45, 1.2>, 0.02}
	cylinder {<0.5, -0.652, 0.38>, <0.4, -0.45, 1.2>, 0.02}
	cylinder {<-0.5, 0.5, 0.38>, <-0.4, 0.4, 1.2>, 0.02}
	cylinder {<0.5, 0.5, 0.38>, <0.4, 0.4, 1.2>, 0.02}	
	sphere {<-0.4, -0.45, 1.2>, 0.02}
	sphere {<0.4, -0.45, 1.2>, 0.02}
	sphere {<-0.4, 0.4, 1.2>, 0.02}
	sphere {<0.4, 0.4, 1.2>, 0.02}
	cylinder {<-0.4, -0.45, 1.2>, <0.4, -0.45, 1.2>, 0.02}
	cylinder {<-0.4, 0.4, 1.2>, <0.4, 0.4, 1.2>, 0.02}
	cylinder {<-0.4, -0.45, 1.2>, <-0.4, 0.4, 1.2>, 0.02}
	cylinder {<0.4, -0.45, 1.2>, <0.4, 0.4, 1.2>, 0.02}
	Round_Box_Merge (<-0.52, -0.672, 0.02>, <0.52, -0.55, 0.4>, 0.01)
	triangle {<-0.52, -0.652, 0.38>, <-0.52, 0.5, 0.38>, <-0.42, 0.4, 1.2>}
	triangle {<-0.52, -0.652, 0.38>, <-0.42, 0.4, 1.2>, <-0.42, -0.45, 1.2>}
	triangle {<0.52, -0.652, 0.38>, <0.52, 0.5, 0.38>, <0.42, 0.4, 1.2>}
	triangle {<0.52, -0.652, 0.38>, <0.42, 0.4, 1.2>, <0.42, -0.45, 1.2>}
	triangle {<-0.5, -0.672, 0.38>, <0.5, -0.672, 0.38>, <0.4, -0.47, 1.2>}
	triangle {<-0.5, -0.672, 0.38>, <0.4, -0.47, 1.2>, <-0.4, -0.47, 1.2>}
	triangle {<-0.5, 0.52, 0.38>, <0.5, 0.52, 0.38>, <0.4, 0.42, 1.2>}
	triangle {<-0.5, 0.52, 0.38>, <0.4, 0.42, 1.2>, <-0.4, 0.42, 1.2>}
	triangle {<-0.4, -0.45, 1.22>, <-0.4, 0.4, 1.22>, <0.4, 0.4, 1.22>}
	triangle {<-0.4, -0.45, 1.22>, <0.4, -0.45, 1.22>, <0.4, 0.4, 1.22>}
	#local yy=-0.4;
	#local my=0.38;
	#local step=0.02;
	#while (yy<my)
		sb(
			<-0.53, yy, 0.38>, <0.53, yy, 0.38>, <-0.43, yy, 1.2>, <0.43, yy, 1.2>
			<-0.53, yy+step, 0.38>, <0.53, yy+step, 0.38>, <-0.43, yy+step, 1.2>, <0.43, yy+step, 1.2>
		)
		#if (yy+step*2<my)
			object {
				sb(
					<-0.521, yy+step, 0.38>, <0.521, yy+step, 0.38>, <-0.422, yy+step, 1.19>, <0.422, yy+step, 1.19>
					<-0.521, yy+step*2, 0.38>, <0.521, yy+step*2, 0.38>, <-0.422, yy+step*2, 1.19>, <0.422, yy+step*2, 1.19>
				)
				pigment {rgb <0.99, 0.98, 0.92>*0.75}
				normal {granite 0.02 scale 0.05}
				finish {
					specular 0.3
					roughness 0.2
					metallic
					ambient 0.2
				}
			}
		#end
		#local yy=yy+step*2.5;
	#end
	//stand
	sphere {0, 1
		scale <0.3, 0.15, 0.3>
		translate <0, -0.6, 0.4>
	}
	cylinder {<0, -0.6, 0.4>, <0, -0.78, 0.4>, 0.2}
	Round_Box_Merge (<-0.4, -0.8, 0.1>, <0.4, -0.75, 0.8>, 0.02)
	
	
	//buttons
	difference {
		cylinder {<0, 0, 0>, <0, 0, 0.1>, 0.03}
		torus {0.015, 0.003 rotate x*90}
		translate <0.54, -0.51, -0.008>
	}
	box {<-0.01, -0.01, -0.001>, 0.01
		translate <0.47, -0.52, 0>
		texture {t_light}
	}
	sphere {0, 1 scale <0.02, 0.017, 0.004> translate <0, -0.51, 0>}
	sphere {0, 1 scale <0.02, 0.017, 0.004> translate <0.1, -0.51, 0>}
	sphere {0, 1 scale <0.02, 0.017, 0.004> translate <0.2, -0.51, 0>}
	sphere {0, 1 scale <0.02, 0.017, 0.004> translate <-0.1, -0.51, 0>}
	sphere {0, 1 scale <0.02, 0.017, 0.004> translate <-0.2, -0.51, 0>}
	
	//label
	text {
		ttf "crystal.ttf" "ViewMaster 2ooo", 1, 0
		scale 0.038
		translate <-0.52, 0.5, -0.003>
		pigment {rgb 0.1}
		finish {specular 0.3 roughness 0.3 ambient 0.3}
	}
	
	//rope
	#if (r)
		union {
			sphere {<0.61, 0.6, 0.2>, 0.012}
			sphere {<-0.61, 0.6, 0.2>, 0.012}
			sphere {<0.61, -0.6, 0.2>, 0.012}
			sphere {<-0.61, -0.6, 0.2>, 0.012}
			sphere {<0.4, -0.8, 0.2>, 0.012}
			sphere {<-0.4, -0.8, 0.2>, 0.012}
			
			cylinder {<-0.4, -0.8, 0.2>, <0.4, -0.8, 0.2>, 0.012}
			cylinder {<-0.4, -0.8, 0.2>, <-0.61, -0.6, 0.2>, 0.012}
			cylinder {<0.4, -0.8, 0.2>, <0.61, -0.6, 0.2>, 0.012}
			cylinder {<-0.61, -0.6, 0.2>, <-0.61, 0.6, 0.2>, 0.012}
			cylinder {<0.61, -0.6, 0.2>, <0.61, 0.6, 0.2>, 0.012}
			cylinder {<0.61, 0.6, 0.2>, <0, 1.2, 0.2>, 0.012}
			cylinder {<-0.61, 0.6, 0.2>, <0, 1.2, 0.2>, 0.012}
			
			sphere {<0, 0.6, -0.01>, 0.012}
			sphere {<0, -0.6, -0.01>, 0.012}
			sphere {<0, -0.8, 0.09>, 0.012}
			sphere {<0, -0.8, 0.81>, 0.012}
			sphere {<0, -0.46, 1.23>, 0.012}
			sphere {<0, 0.41, 1.23>, 0.012}
			
			cylinder {<0, 1.2, 0.2>, <0, 0.6, -0.01>, 0.012}
			cylinder {<0, 0.6, -0.01>, <0, -0.6, -0.01>, 0.012}
			cylinder {<0, -0.6, -0.01>, <0, -0.8, 0.09>, 0.012}
			cylinder {<0, -0.8, 0.09>, <0, -0.8, 0.81>, 0.012}
			cylinder {<0, -0.8, 0.81>, <0, -0.46, 1.23>, 0.012}
			cylinder {<0, -0.46, 1.23>, <0, 0.41, 1.23>, 0.012}
			cylinder {<0, 0.41, 1.23>, <0, 1.2, 0.2>, 0.012}
			
			sphere {0, 1
				scale <0.04, 0.02, 0.04>
				translate <0, 1.18, 0.2>
			}
			
			torus {0.02, 0.012 translate <0, 1.205, 0.2>}
			torus {0.02, 0.012 translate <0, 1.22, 0.2>}
			torus {0.02, 0.012 translate <0, 1.235, 0.2>}
			torus {0.02, 0.012 translate <0, 1.25, 0.2>}
			torus {0.02, 0.012 translate <0, 1.265, 0.2>}
			
			cylinder {<0, 1.2, 0.2>, <0, 10, 0.2>, 0.012}
			
			//cords
			sphere_sweep {
				#local rad=0.02;
				b_spline
				9,
				<0, -0.4, 1.1>, rad
				<0, -0.4, 1.1>, rad
				<0, -0.4, 1.3>, rad
				<0, -0.5, 1.35>, rad
				<0, -1, 1.35>, rad
				<0.1, -1.5, 1.5>, rad
				<-0.1, -2, 1.3>, rad
				<0, -2.5, 1.3>, rad
				<0, -2.5, 1.3>, rad
				texture {t_kbm}
			}
			
			sphere_sweep {
				#local rad=0.015;
				b_spline
				9,
				<0.3, -0.4, 1.1>, rad
				<0.3, -0.4, 1.1>, rad
				<0.3, -0.4, 1.3>, rad
				<0.3, -0.5, 1.35>, rad
				<0.3, -1, 1.35>, rad
				<0.25, -1.5, 1.5>, rad
				<0.35, -2, 1.3>, rad
				<0.2, -2.5, 1.3>, rad
				<0.2, -2.5, 1.3>, rad
				texture {t_kbm}
			}
			
			pigment {
				agate
				turbulence 0.4
				scale 0.02
				color_map {
					[0 rgb <0.9, 0.7, 0.6>]
					[1 rgb <0.6, 0.4, 0.3>]
				}
			}
			normal {granite 1 scale 0.02}
			finish {specular 0.1 roughness 0.5}
		}
	#end	
	
	texture {t_kbm}
	translate <0, 0.8, 0>
	scale <1.3, 1.1, 1.3>
}
#end

//------------------------------------------
// camera

#declare cam=
union {
	union {
		difference {
			Round_Box_Merge (<-0.25, -0.08, -0.12>, <0.25, 0.08, 0.12>, 0.02)
			box {<0.24, -0.055, -0.095>, <1, 0.055, 0.095>}
			box {<0.1, -0.05, -1>, <0.2, -0.02, -0.115>}
			texture {t_kbm}
		}
		union {
			cylinder {<-0.25, 0, 0>, <-0.35, 0, 0>, 0.05}
			torus {0.04, 0.01 rotate z*90 translate x*-0.25}
			box {<0.1, -0.05, -0.125>, <0.12, -0.02, 0>}
			box {<0.15, -0.05, -0.125>, <0.17, -0.02, 0>}
			pigment {rgb 0.2}
			normal {granite 0.02 scale 0.05}
			finish {
				specular 0.3
				roughness 0.2
				metallic
				ambient 0.2
			}
		}
		sphere {0, 0.035
			scale <0.25, 1, 1>
			translate x*-0.25
			pigment {rgb 0.1}
			finish {
				specular 0.8
				roughness 0.002
				reflection 0.3
			}
		}
		union {
			cylinder {<0.24, 0, 0.05>, <0.3, 0, 0.05>, 0.02}
			cylinder {<0, -0.08, 0>, <0, -0.095, 0>, 0.05}
			cylinder {<0, -0.08, 0>, <0, -0.1, 0>, 0.015}
			sphere {<0, -0.135, 0>, 0.035}
			texture {T_Silver_3E}
		}
		translate y*0.135
		rotate <-25, -55, 30>
	}
	union {
		cylinder {<0, -0.2, 0>, <0, 0.015, 0>, 0.04}
		sphere {<0, -0.2, 0>, 0.04}
		cylinder {<0, -0.2, 0>, <0, -0.2, 0.75>, 0.04}
		pigment {rgb 0.1}
		finish {
			specular 0.8
			roughness 0.002
			reflection 0.3
		}
	}
	sphere_sweep {
		#local rad=0.015;
		b_spline
		12,
		vtransform (<0.25, 0, 0.05>, transform {translate y*0.135 rotate <-25, -55. 30>}), rad,
		vtransform (<0.25, 0, 0.05>, transform {translate y*0.135 rotate <-25, -55. 30>}), rad,
		vtransform (<0.4, 0, 0.05>, transform {translate y*0.135 rotate <-25, -55. 30>}), rad,
		vtransform (<0.45, -0.1, -0.05>, transform {translate y*0.135 rotate <-25, -55. 30>}), rad,
		<0.15, -0.5, 0>, rad,
		<0.14, -0.75, -0.05>, rad,
		<0.13, -1, 0.1>, rad,
		<0.1, -1.5, 0>, rad,
		<0.13, -2, -0.1>, rad,
		<0.17, -2.5, 0>, rad,
		<0.15, -3, 0.1>, rad,
		<0.15, -5, -0.1>, rad
		pigment {rgb 0.1}
		finish {
			specular 0.8
			roughness 0.002
			reflection 0.3
		}
	}
}

//------------------------------------------
//clock

#declare clck=
union {
	torus {0.15, 0.01 texture {T_Chrome_5E}}
	cylinder {y*-0.01, y*-0.0009, 0.15 texture {T_Chrome_5E}}
	cylinder {y*-0.001, y*0, 0.14
		pigment {
			image_map {png "clock.png" interpolate 2}
			rotate x*90
			translate <-0.5, 0, -0.5>
			scale 0.28
		}
		finish {specular 0.2 roughness 0.1}
	}
	sphere {0, 1
		scale <0.15, 0.02, 0.15>
		pigment {rgbt 1}
		finish {
			specular 0.7
			roughness 0.002
			reflection {0, 0.2}
		}
		interior {ior 1.1}
	}
	union {
		box {<-0.0025, 0.001, -0.02>, <0.0025, 0.002, 0.12> rotate y*36}
		box {<-0.004, 0.001, -0.02>, <0.004, 0.002, 0.08> rotate y*126}
		pigment {rgb 0.05}
		finish {specular 0.2 roughness 0.1}
	}
}

//------------------------------------------
//bird

#declare bird=
union {
	blob {
		threshold 0.05
		
		//body
		sphere {0, 0.1, 1 scale <1, 1.5, 1> rotate z*-30}
		cylinder {0, <0.05, 0.15, 0>, 0.06, 1}
		sphere {<0.05, 0.15, 0>, 0.06, 1}
		cylinder {<0.05, 0.15, 0>, <0.08, 0.15, 0>, 0.04, 1}
		
		//tail
		cylinder {<-0.15, 0, 0>, <-0.2, -0.1, 0>, 0.1, 1 scale <1, 0.1, 0.7> rotate z*30}
		sphere {0, 0.06, 1 scale <1, 0.3, 0.7> rotate z*40 translate <-0.12, -0.06, 0>}
		
		//legs
		cylinder {<0, -0.05, -0.06>, <-0.05, -0.12, -0.05>, 0.015, 1}
		cylinder {<0, -0.05, 0.06>, <-0.05, -0.12, 0.05>, 0.015, 1}
		
		//beak
		sphere {0, 0.02, 1
			scale <1.5, 1, 1.25>
			translate <0.1, 0.15, 0>
			pigment {rgb <0.5, 0.3, 0.15>}
			finish {specular 0.3 roughness 0.05}
		}
		
		//eyes
		sphere {<0.07, 0.16, -0.039>, 0.015 1 pigment {rgb 0.1} finish {specular 0.5 roughness 0.001}}
		sphere {<0.07, 0.16, 0.039>, 0.015 1 pigment {rgb 0.1} finish {specular 0.5 roughness 0.001}}
		 
		pigment {rgb <0.9, 0.7, 0.4>}
		normal {wrinkles 0.3 scale <0.01, 0.04, 0.01> rotate z*-30}
		finish {specular 0.1 roughness 0.7 ambient 0.2}
	}
	
	//feet
	union {
		cone {<-0.055, -0.12, -0.05>, 0.006, <-0.02, -0.2, -0.05>, 0.004}
		cone {<-0.055, -0.12, 0.05>, 0.006, <-0.02, -0.2, 0.05>, 0.004}
		cone {<-0.02, -0.2, -0.05>, 0.003, <-0.04, -0.2, -0.05>, 0.001}
		cone {<-0.02, -0.2, 0.05>, 0.003, <-0.04, -0.2, 0.05>, 0.001}
		
		cone {<-0.02, -0.2, -0.05>, 0.003, <0.01, -0.2, -0.07>, 0.001}
		cone {<-0.02, -0.2, -0.05>, 0.003, <0.01, -0.2, -0.05>, 0.001}
		cone {<-0.02, -0.2, -0.05>, 0.003, <0.01, -0.2, -0.03>, 0.001}
		
		cone {<-0.02, -0.2, 0.05>, 0.003, <0.01, -0.2, 0.07>, 0.001}
		cone {<-0.02, -0.2, 0.05>, 0.003, <0.01, -0.2, 0.05>, 0.001}
		cone {<-0.02, -0.2, 0.05>, 0.003, <0.01, -0.2, 0.03>, 0.001}
		pigment {rgb 0.1}
		finish {specular 0.3 roughness 0.05}
	}
}

//------------------------------------------
// eye

#declare eye=
union {
	difference {
		sphere_sweep {
			b_spline
			14
			<0, 0.2, 0>, 0.05
			<0, 0.2, 0>, 0.05
			<0, 0.2, 0.1>, 0.05
			<0, 0.15, 0.15>, 0.05
			<0, 0.05, 0.2>, 0.05
			<0, 0, 0.2>, 0.05
			<0, -0.2, 0.25>, 0.05
			<0, -0.25, 0.4>, 0.05
			<0, -0.3, 0.5>, 0.05
			<0, -0.25, 0.6>, 0.05
			<0, -0.3, 0.7>, 0.05
			<0, -0.25, 0.9>, 0.03
			<0, -0.25, 0.9>, 0.03
			<0, -0.25, 1>, 0.02
		}
		sphere {0, 0.05
			scale <1.5, 0.8, 1>
			translate <0, 0.2, -0.05>
		}
		pigment {
			granite
			scale 0.1
			color_map {
				[0 green 0.4]
				[0.5 rgb 0.05]
				[1 green 0.4]
			}
		}
		normal {granite 0.2 scale 0.1}
		finish {specular 0.2 roughness 0.3}
	}
	sphere {<0, 0.2, 0.01>, 0.04
		pigment {
			cylindrical
			rotate x*90
			scale 0.05
			translate y*0.2
			color_map {
				[0 rgb 1]
				[0.6 rgb 1]
				[0.6 rgb <0.2, 0.4, 0.7>]
				[0.85 rgb <0.2, 0.4, 0.7>]
				[0.85 rgb 0.05]
				[1 rgb 0.05]
			}
		}
		finish {specular 0.3 roughness 0.003}
	}
}

//------------------------------------------
// sky

#declare sk=
sky_sphere {
	pigment {
		radial
		frequency 2
		rotate z*90
		color_map {
			[0 rgb <0.7, 0.8, 0.9>*0.5]
			[1 rgb <0.1, 0.2, 0.6>*0.5]
		}
	}
}

//------------------------------------------
//water

#declare water=
difference {
	plane {y, 0}
	object {o_bottle transform {tr_kb}}
	pigment {rgbf <0.2, 0.45, 0.4, 0.9>}
	normal {bozo 0.8 turbulence 0.2 omega 0.6 scale 0.3}
	finish {
		specular 0.8
		roughness 0.03
		brilliance 8
		reflection {0, 0.6}
	}
	interior {
		ior 1.2
		fade_distance 1
		fade_power 3
	}
}

//------------------------------------------
//fog

#declare f=
fog {
	rgbf <0.6, 0.7, 0.65, 0.3>
	distance 15
	turbulence 0.6 omega 0.7
	fog_type 2
	fog_offset 0.25
	fog_alt 0.25
}
#declare f2=
fog {
	rgbf <0.8, 0.85, 0.85, 0.99>
	distance 100
	turbulence 0.3 omega 0.2
	fog_type 2
	fog_offset 2
	fog_alt 4
}
//------------------------------------------
//------------------------------------------

camera {
	up y
	right image_width/image_height*x
	angle 50
	location <0, 0.7, -5>
	look_at y*0.5
}

light_source {
	<-8, 20, 6>
	rgb <1, 1, 0.9>*3.5
	parallel point_at 0
	area_light x*2, z*2, 5, 5 adaptive 1 circular orient
}

sky_sphere {sk}
object {water}
fog {f}
fog {f2}

light_group {
	light_source {<0, 0.4, -1> rgb <0.7, 1, 0.85>*0.5
		fade_distance 5
		fade_power 1
		shadowless
	}

	//trees
	#declare trees=yes;
	#declare sd=seed(11);
	#if (trees)
		#declare maxang=60;
		#declare angstep=15;
		#declare cstep=3;
		#declare rows=15;
		#declare c=3;
		#while (c<rows)
			#switch (mod(c,3))
				#case (0) #declare ang=-maxang; #break
				#case (1) #declare ang=-maxang+angstep/3; #break
				#case (2) #declare ang=-maxang+2*angstep/3; #break
			#end
			#while (ang<maxang)
				#declare tt=int(RRand(0, 9, sd));
				object {
					tree[tt]
					scale RRand(2.5, 5, sd)
					rotate y*RRand(0, 360, sd)
					translate VRand_In_Box (<-0.5, -0.01, -0.5>, <0.5, 0.01, 0.5>, sd)
					translate z*c*cstep
					rotate y*ang
				}
				#declare ang=ang+angstep;
			#end
			#declare c=c+1;
		#end
	#end
	object {tree[0] scale <2, 4, 2> translate <-2, 0, 0>}
	object {tree[4] rotate y*60 scale <1.8, 3, 1.8> translate <-1, 0, 2>}
	object {tree[2] scale <2, 4, 2> translate <2.5, 0, 2>}
	object {tree[1] scale <1.5, 3, 1.5> translate x*0.25}

	//grass
	#local c=0;
	#local maxc=100;
	#local s=seed(100);
	#while (c<maxc)
		#local scxz=RRand(0.2, 0.5, sd);
		#local scy=RRand(0.4, 0.6, sd);
		#local l=VRand_In_Box(<-10, -0.01, -10>, <10, 0, 20>, s);
		object {clump(seed(c)) scale <scxz, scy, scxz>*0.2 translate l}
		#local c=c+1;
	#end

	//objects
	object {kb}
	object {mon(yes) scale 0.4 rotate y*-40 translate <-0.99, 0.5, 1.5>}
	object {mon(no) scale 0.4 translate y*-0.8 rotate <87, 30, 2> translate <-0.6, 0.06, -1.5>}
	object {mon(no) scale 0.4 translate y*-0.8 rotate <87, -60, 2> translate <-0.7, 0.06, 2.5>}
	object {mon(no) scale 0.4 translate y*-0.8 rotate <88, -120, -2> translate <1.5, 0.06, 4.5>}
	object {mon(no) scale 0.4 translate y*-0.8 rotate <93, -170, 3> translate <-1.8, 0.06, 3.5>}
	object {cam scale 0.4 rotate y*35 translate <2.4, 1, 1.8>}
	object {clck rotate y*-20 translate <0.7, -0.011, -2.6>}
	object {bird scale 0.4 rotate y*20 translate vtransform (<0.2, 0.696, -0.01>, transform {rotate y*-40 translate <-0.99, 0.5, 1.5>})}
	object {eye rotate y*50 translate <2.35, 0, 2.5>}
	object {mouse rotate <15, -140, 0> translate <0.35, 0.27, -1.2>}
	object {mouse rotate <-3, 60, -2> translate <-0.5, 0.07, -1.9>}

	//mouse cords
	sphere_sweep {
		#local rad=0.004;
		b_spline
		15,
		vtransform(<0, -0.01, 0>, transform {rotate <15, -140, 0> translate <0.35, 0.27, -1.2>}), rad
		vtransform(<0, -0.01, 0.09>, transform {rotate <15, -140, 0> translate <0.35, 0.27, -1.2>}), rad
		vtransform(<0, -0.01, 0.1>, transform {rotate <15, -140, 0> translate <0.35, 0.27, -1.2>}), rad
		vtransform (<0, -0.01, 0.13>, transform {rotate <15, -140, 0> translate <0.35, 0.27, -1.2>}), rad
		vtransform (<0, -0.02, 0.15>, transform {rotate <15, -140, 0> translate <0.35, 0.27, -1.2>}), rad
		<0.2, 0.02, -1.5>, rad
		<0.1, 0.005, -1.5>, rad
		<0, 0.005, -1.4>, rad
		<-0.1, 0.003, -1.8>, rad
		<-0.3, 0.003, -1.9>, rad
		<-0.4, 0.005, -1.7>, rad
		<-0.3, 0.003, -1.6>, rad
		<-0.25, 0, -1.6>, rad
		<-0.25, -0.2, -1.6>, rad
		<-0.25, -1, -1.6>, rad
		texture {t_kbm}
	}
	sphere_sweep {
		#local rad=0.004;
		b_spline
		14
		vtransform(<0, -0.01, 0>, transform {rotate <-3, 60, -2> translate <-0.5, 0.07, -1.9>}), rad
		vtransform(<0, -0.01, 0.09>, transform {rotate <-3, 60, -2> translate <-0.5, 0.07, -1.9>}), rad
		vtransform(<0, -0.01, 0.1>, transform {rotate <-3, 60, -2> translate <-0.5, 0.07, -1.9>}), rad
		vtransform(<0, -0.01, 0.13>, transform {rotate <-3, 60, -2> translate <-0.5, 0.07, -1.9>}), rad
		vtransform(<0, -0.03, 0.15>, transform {rotate <-3, 60, -2> translate <-0.5, 0.07, -1.9>}), rad
		<-0.4, 0.005, -2>, rad
		<-0.2, 0.005, -2.1>, rad
		<-0.1, 0.003, -2>, rad
		<-0, 0.003, -1.7>, rad
		<0.1, 0.005, -1.5>, rad
		<0.15, 0, -1.6>, rad
		<0.15, -0.2, -1.5>, rad
		<0.25, -1, -1.6>, rad
		<0.25, -1, -1.6>, rad
		texture {t_kbm}
	}
	global_lights on
}

//sky clock
light_group {
	light_source {<0, -1, 0> rgb <2, 1.75, 1.5>}
	object {clck rotate x*180 scale 10}
	global_lights off
	rotate <30, 0, 20>
	translate <-1, 5.5, 22>
}

//signature
light_group {
	light_source {<0.52, 0.1, -3.15> rgb 1}
	text {
		ttf "Alakob.ttf" "J Moore, 2003", 5, 0
		texture {T_Chrome_5E}
		scale 0.06
		rotate x*90
		translate <0.52, -0.0001, -3.15>
	}
	global_lights on
}
