// Persistence of Vision Ray Tracer Scene Description File
// File: harpsichord.inc
// Vers: 3.5
// Desc: A XVII-century harpsichord
// Date: 2003/10/03
// Auth: Maurizio Tomasi
// Note: Terminology was taken from "Acoustic of the Harpsichord" by E. L. Kottick, 
//       K. D. Marshall and Thomas J. Hendrickson, published in the italian edition
//       of "Scientific American" ("Le Scienze" n. 272, April 1991)

#include "rand.inc"
#include "shapes.inc"
#include "colors.inc"
#include "woods.inc"

#include "harpsichord-leg.inc"

#local HarpsichordSeed = seed (21634);

#local BlackKeyTexture =
texture
{
    T_Wood4
}
texture
{
    pigment { color rgb <0.3, 0.3, 0.3> * 0.5 filter 0.6 }
}

#local WhiteKeyTexture =
texture
{
    T_Wood1
}
texture
{
    pigment { color rgb 1 transmit 0.7 }
}

#local WoodTexture =
texture
{
    T_Wood2
    
    finish
    {
        specular 1.0
        roughness 0.03
    }
    normal
    {
        bozo 0.3
        scale 0.5
    }
    
    scale 0.3
}
texture
{
    pigment { color rgbt <0.5, 0.5, 0.8, 0.93> }
}
texture
{
    // Dark spots
    
    pigment
    {
        bozo
        
        color_map
        {
            [0.00 color rgb 0]
            [0.15 color rgb 0]
            [0.15 color rgb 1 transmit 1]
        }
        
        scale 0.05
    }
}

#local BorderTexture =
texture
{
    T_Wood1
}


#local StdBorderSize = 0.075;
#local LegRadius = 2.5;
#local LegHeight = 1.5;

#macro BoxWithBorder (x1, y1, x2, y2, Thickness, BorderSize, InnerTexture)
    #local MinX = min (x1, x2);
    #local MaxX = max (x1, x2);
    #local MinY = min (y1, y2);
    #local MaxY = max (y1, y2);

    union
    {
        box
        {
            <x1, y1, -Thickness/2>, <x2, y2, Thickness/2>
            
            texture 
            { 
                InnerTexture
                translate VRand (HarpsichordSeed)
            }
        }

        object
        {
            Round_Box_Union (<x1 - BorderSize/2, y1 - BorderSize/2, -Thickness/2 - BorderSize/2>,
                <x2 + BorderSize/2, y1 + BorderSize/2,  Thickness/2 + BorderSize/2>, 0.015)
            
            texture 
            { 
                BorderTexture 
                translate VRand (HarpsichordSeed)
            }
        }

        object
        {
            Round_Box_Union (<x1 - BorderSize/2, y1 - BorderSize/2, -Thickness/2 - BorderSize/2>,
                <x1 + BorderSize/2, y2 + BorderSize/2,  Thickness/2 + BorderSize/2>, 0.015)

            texture 
            { 
                BorderTexture 
                rotate 91*x
                translate VRand (HarpsichordSeed)
            }
        }

        object
        {
            Round_Box_Union (<x2 + BorderSize/2, y2 + BorderSize/2, -Thickness/2 - BorderSize/2>,
                <x1 - BorderSize/2, y2 - BorderSize/2,  Thickness/2 + BorderSize/2>, 0.015)

            texture 
            { 
                BorderTexture 
                rotate 91*y
                translate VRand (HarpsichordSeed)
            }
        }

        object
        {
            Round_Box_Union (<x2 + BorderSize/2, y2 + BorderSize/2, -Thickness/2 - BorderSize/2>,
                <x2 - BorderSize/2, y1 - BorderSize/2,  Thickness/2 + BorderSize/2>, 0.015)

            texture 
            { 
                BorderTexture 
                rotate 91*x
                translate VRand (HarpsichordSeed)
            }
        }
    }
#end

/*
 * Top view of the harpsichord
 *
 *  <--D-->
 * ^ +-----+
 * | |     |
 * | |     |
 * | |     \
 * | |      \
 * B |       \
 * | |        \
 * | |         + ^
 * | |         | |
 * | |         | C
 * | |         | |
 * v +---------+ v
 *  .         .
 *  .<---A--->.
 */

#local A = 3.0;
#local B = 6.0;
#local C = 2.0;
#local D = 1.0;


#local Spline = array [8]
{
    <A + 1, C - 1>,
    <A, C>,
    <D, B>,
    <D, B + 3>,
    <A + 5, B + 3>,
    <A + 1, C - 1>,
    <A, C>
    <D, B>,
}

#local CurvedBeltShape =
prism
{
    cubic_spline
    0.0, 1.0
    
    dimension_size (Spline, 1),
    
    #local Count = 0;
    #while (Count < dimension_size (Spline, 1))
    
        Spline[Count]
        
        #local Count = Count + 1;
    #end
}


#macro CurvedBelt (Thickness, start_y, end_y)
intersection
{
    difference
    {
        object
        {
            CurvedBeltShape
            translate -Thickness / 2 *x
        }

        object
        {
            CurvedBeltShape
            translate Thickness / 2 * x
            scale <1, 10, 1>
            translate -5*y
        }
    }
    
    box
    {
        <0, -1, 0>, <A + StdBorderSize/2, 2, B + StdBorderSize/2>
    }
    
    scale <1, (end_y - start_y), 1>
    translate start_y * y
}
#end


#local BaseShape =
union
{
    box
    {
        <0.0, -StdBorderSize/2, 0.0>, <A, StdBorderSize/2, C>
    }
    
    difference
    {
        box
        {
            <0.0, -StdBorderSize/2, C>, <A, StdBorderSize/2, B>
        }

        object
        {
            CurvedBeltShape
            scale <1, 2, 1>
            translate -y
        }
    }
}

#local BaseBorder =
difference
{
    object 
    { 
        BaseShape 
        translate <-A/2, 0, -B/2>
        scale (A + StdBorderSize/2) / A 
        translate <A/2, 0, B/2>
    }
    
    object 
    { 
        BaseShape 
        translate <-A/2, 0, -B/2>
        scale <1, 0, 1> * (A - StdBorderSize/2) / A 
            + 10*y
        translate <A/2, 0, B/2>
    }

    texture { BorderTexture rotate 90*y }
}

#local Base =
union
{
    object { BaseBorder }
    object 
    { 
        BaseShape 
        texture 
        { 
            pigment
            {
                image_map
                {
                    jpeg "harpsichord-landscape.jpg"
                    interpolate 2
                }
                rotate -90*x
                rotate -90*y
                scale <A, 1, B>
            }
        
            normal
            {
                bozo 0.5
                
                scale 0.07
            }
        }
    }
}


#macro CreateKeyboard (Translation, Seed)
object
{
    #local Count = 0;
    #local CurKey = 3; // Typically, an harpsichord starts from F
    #local NumOfBlackKeys = 35;
    #local KeyboardWidth = A - 0.4;
    
    /* Note: in an harpsichord the meaning of "black key" and "white key" is usually 
             swapped with respect to the piano.  Here I follow the harpsichordists' 
             convention, i.e. C is a "black key", while C-sharp is a "white key". */
    union
    {
        #while (Count < NumOfBlackKeys)
            union
            {
                box
                {
                    <KeyboardWidth * Count / NumOfBlackKeys, 0.15, 0.1>,
                    <KeyboardWidth * (Count + 0.9) / NumOfBlackKeys, 0.30, 0.6>
                }
    
                box
                {
                    <KeyboardWidth * Count / NumOfBlackKeys, 0.30, 0.095>,
                    <KeyboardWidth * (Count + 0.9) / NumOfBlackKeys, 0.31, 0.6>
                }
                
                #if (RRand (0, 1, Seed) > 0.6)
                    translate RRand (-0.01, 0.003, Seed)*y
                #end
                
                texture 
                { 
                    BlackKeyTexture 
                    translate VRand (Seed) 
                    rotate VRand (Seed)*360 
                }
            }
            
            // Draw a white key
            
            #if (CurKey = 0 | CurKey = 1 | CurKey = 3 | CurKey = 4 | CurKey = 5)
                #if (Count < NumOfBlackKeys - 1)
                    object
                    {
                        Round_Box_Union (
                            <KeyboardWidth * (Count + 0.75) / NumOfBlackKeys, 0.15, 0.3>,
                            <KeyboardWidth * (Count + 1.25) / NumOfBlackKeys, 0.33, 0.6>, 0.005)
                            
                        texture 
                        { 
                            WhiteKeyTexture 
                            translate VRand (Seed) 
                            rotate VRand (Seed)*360 
                        }
                    }
                #end
            #end
            
            #local Count = Count + 1;
            #if (CurKey < 6)
                #local CurKey = CurKey + 1;
            #else
                #local CurKey = 0;
            #end
        #end
        translate 0.2*x
    }
    
    translate Translation
}
#end

#local SideBeltTexture =
texture
{
    pigment
    {
        image_map
        {
            png "harpsichord-side.png"
            interpolate 2
        }
        
        translate -0.5*y
        scale <399/238, 1, 1> * 0.9
        translate 0.5*y
        translate 0.6*x
    }
}

#local CoverAngle = 50;

#declare Harpsichord =
union
{
    union
    {
        difference
        {
            object
            {
                BoxWithBorder (0.0, 0.0, B, 1.0, 0.1, StdBorderSize, WoodTexture)
                rotate -90*y
            }

            box
            {
                <-1, StdBorderSize, StdBorderSize>,
                < 1, 1 - StdBorderSize, 1.2>
            }
            
            cutaway_textures
        }
        
        box
        {
            <0.0, 0.0, -0.05>, <1.2, 1.0, 0.05>

            texture
            {
                pigment
                {
                    image_map
                    {
                        png "harpsichord-side.png"
                        interpolate 2
                    }
                    
                    scale <399/238, 1, 1> * 0.6
                    translate <0.13, 0.4, 0.0>
                }
            }
            
            rotate -90*y
        }
    }
            
    
    object
    {
        BoxWithBorder (0.0, 0.0, C, 1.0, 0.1, StdBorderSize, SideBeltTexture)
        rotate -90*y
        translate A*x
    }
    
    // Name lath
    
    object
    {
        BoxWithBorder (0.0, 0.0, A, 0.2, 0.1, StdBorderSize/2, WoodTexture)
        translate StdBorderSize/2*z
    }
    
    // Name belt
    
    #local InnerBelt =
    object
    {
        BoxWithBorder (0.2, 0.55, A - 0.2, 0.85, 0.1, StdBorderSize/4, T_Wood6)
        
        translate 1.2*z
    }

    difference
    {
        object
        {
            BoxWithBorder (0.0, 0.0, A, 1.0, 0.1, StdBorderSize/2, WoodTexture)
            translate 1.2*z
        }
        
        object { InnerBelt }
        
        cutaway_textures
    }
    
    object { InnerBelt }

    // Tail
    
    object
    {
        BoxWithBorder (0.0, 0.0, D + StdBorderSize, 1.0, 0.1, StdBorderSize, WoodTexture)
        translate B*z
    }

    // Legs

    union
    {
        object
        {
            HarpsichordLeg
            translate -y
            scale <LegRadius, LegHeight, LegRadius>
            
            translate <0.25, 0, 0.25>
        }
        
        object
        {
            HarpsichordLeg
            translate -y
            scale <LegRadius, LegHeight, LegRadius>
            
            translate <D/2, 0, B - 0.25>
        }
    
        object
        {
            HarpsichordLeg
            translate -y
            scale <LegRadius, LegHeight, LegRadius>
            
            translate <A - 0.25, 0, 0.25>
        }
        
        texture
        {
            WoodTexture
        }
    }
    
    
    // Base

    object { Base }

    // Cover
    
    object 
    { 
        Base 
        
        rotate CoverAngle * z
        translate <0.0, 1.0 + StdBorderSize, 0.0>
    }

    // Curved lateral belt
    
    object { CurvedBelt (0.1, 0, 1) }

    union
    {
        object
        {
            CurvedBelt (0.1 + StdBorderSize, -StdBorderSize / 2, StdBorderSize / 2)
        }

        object
        {
            CurvedBelt (0.1 + StdBorderSize, 1.0 - StdBorderSize / 2, 
                1.0 + StdBorderSize / 2)
        }
    }

    // Strings
    
    union
    {
        #local XPos = 0.2;
        #local StringRadius = 0.005;
        #while (XPos < A - 0.2)
            cylinder
            {
                <XPos, 0.95, 1.3>, <XPos, 0.95, B + XPos * (C - B) / A>, StringRadius
            }
            
            #local XPos = XPos + StringRadius * 30;
        #end
        
        texture
        {
            pigment
            {
                color rgb <0.7, 0.7, 1.0>
            }
            
            finish
            {
                specular 0.3
                roughness 0.0005
                metallic
            }
        }
    }
    
    // Stick
    
    cone
    {
        <0, 0, 0>, 0.02, 
        2.05 * A * sin (radians (CoverAngle / 2)) * y, 0.01

        texture { WoodTexture }
        
        rotate z * (90 - CoverAngle + 10) / 2
        translate <A, 1.0, 1.1>
    }
    
    
    // Keys
    
    object { CreateKeyboard (0, seed (16735)) }
    object { CreateKeyboard (<0, 0.08, 0.6>, seed (41335)) }
        
    union
    {
        Round_Box_Union (<StdBorderSize/2, 0.1, 0.1>,
            <0.2 - 0.01, 0.4, 1.2>, 0.02)
        
        Round_Box_Union (<A - StdBorderSize/2, 0.1, 0.1>,
            <A - 0.2 + 0.01, 0.4, 1.2>, 0.02)
        
        texture { WoodTexture }
    }
    
    translate LegHeight*y
    
    pigment { color rgb 1 }
}
