;;; Parquet.scm - Maurizio Tomasi 2003
;;;
;;; This Scheme program will generate file "parquet.inc", a POV-Ray
;;; include file which contains the definition of a set of bricks in
;;; an herringbone pattern.  The overall object extends in the x,z
;;; direction.
;;;
;;; You must implement two macros named CreateParquetText1 and
;;; CreateParquetText2 before including "parquet.inc".  These macros
;;; must define a texture which extends along the z axis and the x
;;; axis respectively, and will be used for the bricks.  Mote that
;;; these are macros, not variables; this gives you a certain amount
;;; of flexibility (see "walls.inc" for an implementation).

;;; Output port
(define output-file #f)

;;; These two variables roughly fix the overall size of the floor.
;;; Divide by three to get the number of bricks along the x and
;;; z direction.
(define l1-limit 120)
(define l2-limit 160)

;;; Return the starting column
(define start-u
  (lambda (u)
    (- (remainder (+ u 2) 6) 2)))

;;; Create a new parquet brick.  If the specified (U,V) position is
;;; outside the parquet, return #f.
(define create-brick
  (lambda (u v up?)
    (if (and (< u l1-limit) (< v l2-limit))
	(begin
	  (display (format "box { <~a, ~a, ~a>, <~a, ~a, ~a> ~a () }\n"
                           (+ 0.01 u)                   0 (+ 0.01 (+ v (if up? -2 0)))
                           (+ -0.01 (+ u (if up? 1 3))) 1 (+ -0.01 (+ v 1))
                           ;; Distinguish between "up" and "right" bricks
                           (if up? "CreateParquetText1" "CreateParquetText2")) output-file)
	  #t)
	#f)))

;;; Create a line of bricks, either vertical or horizontal
;;; (according to the UP? parameter, which is #f if the row goes along
;;; the x axis, #t if along the z axis).
(define create-brick-line
  (lambda (u-start v-start up?)
    (letrec ((place-next-brick
	      (lambda (u v)
		(if (create-brick u v up?)
		    (place-next-brick (+ u 6) v)))))
      (place-next-brick u-start v-start))))

(define main
  (lambda (args)
    ;; Open the file
    (set! output-file (open-output-file "parquet.inc" 'replace))
    
    (letrec ((create-horizontal-line ; Implemented recursively
	      (lambda (u v count)
		(create-brick-line u v #f)
		(if (< v l2-limit)
		    (create-horizontal-line (start-u (+ u 1))
                                            (+ v 1) 
                                            (+ count 1)))))

	     (create-vertical-line ; Ditto
	      (lambda (u v count)
		(create-brick-line u v #t)
		(if (< v l2-limit)
		    (create-vertical-line (start-u (+ u 1))
                                          (+ v 1)
                                          (+ count 1))))))
      
      (create-horizontal-line 0 0 0)
      (create-vertical-line 1 -2 0))

    ;; Close the file
    (close-output-port output-file)
))

(main '())