// Persistence of Vision Ray Tracer Scene Description File
// File: walls.inc
// Vers: 3.5
// Desc: Walls, parquet floor, ceiling
// Date: 2003/09/10
// Auth: Maurizio Tomasi

#include "rand.inc"
#include "arrays.inc"
#include "shapes.inc"
#include "shapes2.inc"

#include "new-wood.inc"
#include "spiral.inc"

#declare WhiteTextures = array[3];
#declare GrayTextures = array[3]

#declare WhiteTextures[0] =
texture
{
    pigment { color rgb <0.8, 0.85, 0.85> }
    normal { bozo 0.6 turbulence 0.2 scale 0.09 translate <0.2, -0.6, 0.3> }
    finish { specular 0.8 roughness 0.01 }
}

#declare WhiteTextures[1] =
texture
{
    pigment { color rgb <0.8, 0.85, 0.9> * 0.9 }
    normal { bozo 0.6 turbulence 0.3 scale 0.4 }
    finish { specular 0.8 roughness 0.01 }
}

#declare WhiteTextures[2] =
texture
{
    pigment { color rgb <0.8, 0.85, 0.9> * 0.8 }
    normal { bozo 0.4 turbulence 0.3 scale 0.5 }
    finish { specular 0.8 roughness 0.03 }
}

// ---

#declare GrayTextures[0] =
texture
{
    pigment { color rgb <0.8, 0.9, 0.85> * 0.80 }
    normal { bozo 0.6 turbulence 0.2 scale 0.2 }
    finish { specular 0.8 roughness 0.01 }
}

#declare GrayTextures[1] =
texture
{
    pigment { color rgb <0.8, 0.9, 0.85> * 0.75 }
    normal { bozo 0.6 turbulence 0.2 scale 0.1 }
    finish { specular 0.8 roughness 0.01 }
}

#declare GrayTextures[2] =
texture
{
    pigment { color rgb <0.8, 0.9, 0.85> * 0.70 }
    normal { bozo 0.4 turbulence 0.1 scale 0.3 }
    finish { specular 0.8 roughness 0.03 }
}



//-------------------------------------------------------------
// Parquet textures


#local ParquetSeed = seed (15);

#macro CreateParquetText1 ()

    texture 
    { 
        Rand_Array_Item (WhiteTextures, ParquetSeed)
        //finish { WoodFinish1 }

        rotate RRand (-4, 4, ParquetSeed)*y
    }

#end

#macro CreateParquetText2 ()

    texture
    {
        Rand_Array_Item (GrayTextures, ParquetSeed)
        //finish { WoodFinish1 }

        rotate 90*y
        rotate RRand (-4, 4, ParquetSeed)*y
    }

#end


//-------------------------------------------------------------
// Floor, walls

/*
 *    ^
 * z2 +  +-----------G------------+
 *    |  |                        |
 *    |  |                        |
 *    |  F                        H
 *    |  |                        |
 *    |  |                        |
 * z1 +  +-------D------|  E  |---+
 *    |  |              .     .   |
 *    |  |              .     .   |
 *    |  B           .  .     .   C       O : main camera
 *    |  |           O  .     .   |
 *    |  |              .     .   |
 * z0 +  +------------A-----------+
 *    |                 .     .   .
 *    +--+--------------+-----+---+-->
 *       x0             x1    x2  x3
 */

#declare x0 = 0.0;
#declare x1 = 8.5;
#declare x2 = 12.5;
#declare x3 = 24.0; // 18.0
#declare z0 = 0.0;
#declare z1 = 12.0;
#declare z2 = 28.0;

#declare StepHeight = 0.35;
#declare StepEdge = 0.015;
#declare TopStepEdge = 0.03;
#declare DoorHeight = 8.0;
#declare WallHeight = 10.2;


//---------------------------------------------------------------
// Textures


#local WallTexture =
texture
{
    pigment 
    { 
        image_map 
        { 
            png "floral.png" 
            
            interpolate 2
        } 
        
        scale 4
        translate 2*y
        rotate 45*z
    }
    
    normal
    {
        granite 0.05
        scale <1, 2, 1> * 0.3
    }
    
    finish { diffuse 0.4 brilliance 0.2 }
}

#local StepTexture =
texture
{
    pigment { color rgb <0.3, 0.4, 0.3> }
    
    normal
    {
        marble 0.1
        turbulence 0.3
        rotate 87*z
        
        scale 0.07
    }
    
    finish { diffuse 0.4 brilliance 0.2 }
}
texture
{
    pigment
    {
        bozo
        color_map
        {
            [0.7 rgbt 1]
            [0.7 rgbt 0.5]
            [0.9 rgbt 0.1]
        }
        
        scale 0.03
    }
}

#local WhiteTexture =
texture
{
    pigment { color rgb 1 }
    
    finish { diffuse 0.5 }
}

#local GrainyWhiteTexture =
texture
{
    pigment { color rgb 1 }
    
    normal
    {
        granite 0.4
        turbulence 0.5
        scale 0.15
    }
    
    finish { diffuse 0.5 }
}


//---------------------------------------------------------------
// Macros


#macro CreateWall (start_x, start_z, end_x, end_z, x_dir, 
    use_decorations, deco_start, deco_end)
    
    union
    {
        box 
        {
            <start_x, 0, start_z>, <end_x, WallHeight, end_z>
            
            #if (x_dir)
                texture { WallTexture }
            #else
                texture { WallTexture rotate 90*y }
            #end
        }
        
        #local CurY = 0;
        
        #local CurIncrement = StepHeight;
        #local CurSize = StepEdge * 2;
        object
        {
            Round_Box_Union (<start_x - CurSize/2, CurY, start_z - CurSize/2>,
                <end_x + CurSize/2, CurY + CurIncrement, end_z + CurSize/2>, 
                StepEdge)
            
            texture { StepTexture }
        }
        #declare CurY = CurY + CurIncrement * 0.9;
        
        #local CurIncrement = 0.04;
        #local CurSize = StepEdge * 0.4;
        object
        {
            Round_Box_Union (<start_x - CurSize/2, CurY, start_z - CurSize/2>,
                <end_x + CurSize/2, CurY + CurIncrement, end_z + CurSize/2>, 
                StepEdge)
            
            texture { WhiteTexture }
        }
        #declare CurY = CurY + CurIncrement * 0.9;

        #local CurIncrement = 2.0;
        #local CurSize = StepEdge * 0.2;
        object
        {
            Round_Box_Union (<start_x - CurSize/2, CurY, start_z - CurSize/2>,
                <end_x + CurSize/2, CurY + CurIncrement, end_z + CurSize/2>, 
                StepEdge)
            
            texture { GrainyWhiteTexture }
        }
        #declare CurY = CurY + CurIncrement * 0.9;

        #local CurIncrement = 0.06;
        #local CurSize = StepEdge * 0.6;
        object
        {
            Round_Box_Union (<start_x - CurSize/2, CurY, start_z - CurSize/2>,
                <end_x + CurSize/2, CurY + CurIncrement, end_z + CurSize/2>, 
                StepEdge)
            
            texture { WhiteTexture }
        }
        #declare CurY = CurY + CurIncrement * 0.9;
        
        #local CurIncrement = 0.08;
        #local CurSize = StepEdge * 1.2;
        object
        {
            Round_Box_Union (<start_x - CurSize/2, CurY, start_z - CurSize/2>,
                <end_x + CurSize/2, CurY + CurIncrement, end_z + CurSize/2>, 
                StepEdge)
            
            texture { WhiteTexture }
        }
        #declare CurY = CurY + CurIncrement * 0.9;

        #local CurIncrement = 0.06;
        #local CurSize = StepEdge * 0.6;
        object
        {
            Round_Box_Union (<start_x - CurSize/2, CurY, start_z - CurSize/2>,
                <end_x + CurSize/2, CurY + CurIncrement, end_z + CurSize/2>, 
                StepEdge)
            
            texture { WhiteTexture }
        }
        
        // Top

        #if (use_decorations)

            #local CurY = WallHeight;
            #local CurIncrement = 0.15;
            #local CurSize = StepEdge * 1.1;
            object
            {
                Round_Box_Union (<start_x - CurSize/2, CurY, start_z - CurSize/2>,
                    <end_x + CurSize/2, CurY - CurIncrement, end_z + CurSize/2>, 
                    TopStepEdge)
                
                texture { WhiteTexture }
            }
            #declare CurY = CurY - CurIncrement;
    
            #local CurIncrement = 0.5;
            #local CurSize = StepEdge * 0.5;
            object
            {
                Round_Box_Union (<start_x - CurSize/2, CurY - CurIncrement, start_z - CurSize/2>,
                    <end_x + CurSize/2, CurY, end_z + CurSize/2>, 
                    TopStepEdge)
                
                texture { GrainyWhiteTexture }
            }
            #declare CurY = CurY - CurIncrement * 0.9;
    
            #local SpiralBase = CurY;
    
            #local CurIncrement = 0.15;
            #local CurSize = StepEdge * 10;
            object
            {
                Round_Box_Union (<start_x - CurSize/2, CurY - CurIncrement, start_z - CurSize/2>,
                    <end_x + CurSize/2, CurY, end_z + CurSize/2>, 
                    TopStepEdge)
                
                texture { WhiteTexture }
            }
            #declare CurY = CurY - CurIncrement;
            
            #local cur_x = start_x + deco_start * (end_x - start_x);
            #local spiral_size = 0.5;
            #local end_deco_x = start_x + deco_end * (end_x - start_x);
            #while (cur_x < end_deco_x)
                object
                {
                    Spiral (0.5, 0.02, 0.05, 5)
                    
                    scale spiral_size
                    translate <cur_x, SpiralBase, min (start_z, end_z)>
                    
                    texture 
                    { 
                        pigment { color rgb <0.3, 0.4, 0.3> * 1.5 }
                        finish { diffuse 0.5 }
                    }
                }
                
                #local cur_x = cur_x + spiral_size;
            #end
            
        #end
    }
#end


//---------------------------------------------------------------
// Objects


// Ground

#if (USE_PARQUET)
    union
    {
        #include "parquet.inc"
        
        translate <x0, 0, 0>
        scale <0.15, 0.1, 0.15>
    }
#else
    plane
    {
        y, 0
        pigment { color rgb 0.1 }
    }
#end


// Walls

CreateWall (x0, z0, x3, z0 - 0.5, true, false, 0, 0)  // A
CreateWall (x0 - 0.5, z0, x0, z1, false, false, 0, 0) // B
CreateWall (x3, z0, x3 + 0.5, z1, false, false, 0, 0) // C

difference // D
{
    CreateWall (x0, z1, x3, z1 + 0.5, true, 
        USE_WALL_DECORATIONS, 0.35, 0.70)

    // Door    
    box
    {
        <x1, -1, z1 - 1>, <x2, DoorHeight, z1 + 1>
    }
    
    cutaway_textures
}

CreateWall (x0 - 0.5, z1, x0, z2, false, false, 0, 0) // F
CreateWall (x0, z2, x3, z2 + 0.5, true, false, 0, 0)  // G
CreateWall (x3, z1, x3 + 0.5, z2, false, false, 0, 0) // H


// Ceiling

box
{
    <x0, WallHeight, z0>, <x3, WallHeight + 0.1, z2>
    
    pigment { color rgb 0.1 }
}


//---------------------------------------------------------------
// Door

#local DoorTexture =
texture
{
    pigment 
    { 
        wood
        turbulence 0.1
        
        color_map
        {
            [0.0 color rgb 1]
            [0.5 color rgb 1]
            [0.5 color rgb <0.9, 0.95, 0.9>]
        }
        
        scale 0.1
        rotate 90*z
    }
    
    finish { diffuse 0.5 }
}


#local Tip =
union
{
    cylinder 
    {
        <0, 0.05, 0>, <0, 0.5, 0>, 0.2
    }
    
    cylinder 
    {
        <0, -0.5, 0>, <0, -0.05, 0>, 0.2
    }
    
    cylinder
    {
        <0, -0.1, 0>, <0, 0.1, 0>, 0.15
    }

    sphere
    {
        <0, 0.58, 0>, 0.2
    }

    sphere
    {
        <0, -0.58, 0>, 0.2
    }

    texture
    {
        pigment { color rgb <0.7, 0.6, 0.4> * 0.5 }
        finish { specular 0.7 roughness 0.01 metallic }
    }
}


union
{
    union
    {
        union
        {
            Round_Box_Union (<x1 - 0.5, -0.05, z1 - 0.05>, 
                <x1, DoorHeight, z1 + 0.6>, 0.05)
        
            Round_Box_Union (<x2, -0.05, z1 - 0.05>, 
                <x2 + 0.5, DoorHeight, z1 + 0.6>, 0.05)

            texture { DoorTexture }
        }
    
        object
        {    
            Round_Box_Union (<x1 - 0.5, DoorHeight - 0.05, z1 - 0.05>, 
                <x2 + 0.5, DoorHeight + 0.5, z1 + 0.6>, 0.05)

            texture { DoorTexture rotate 89*y }
        }
    }
    
    // Door
    
    union
    {
        difference
        {
            object
            {
                Round_Box_Union (<- (x2 - x1), 0.05, -0.25>, <0, DoorHeight - 0.10, 0.25>, 0.05)
                texture { DoorTexture rotate 90*z }
            }
            
            union
            {
                #local BoxHoles =
                union
                {
                    object 
                    { 
                        Pyramid2 
                        scale <(x2 - x1) * 0.4, 5.0, 2.5>
                        rotate 90 * x 
                        
                        translate <-(x2 - x1) / 4, 1.75, -0.1>
                    }
        
                    object 
                    { 
                        Pyramid2
                        scale <(x2 - x1) * 0.4, 5.0, 2.5>
                        rotate 90 * x
                        
                        translate <-(x2 - x1) / 4, 4.75, -0.1>
                    }
                    
                    object 
                    { 
                        Pyramid2
                        scale <(x2 - x1) * 0.4, 5.0, 1.0>
                        rotate 90 * x
                        
                        translate <-(x2 - x1) / 4, 6.75, -0.1>
                    }
                }
                
                object { BoxHoles }
                object { BoxHoles translate -(x2 - x1) / 2 * x }

                pigment { color rgb <0.3, 0.4, 0.3> * 3 }
            }
        }

        #local Boxes =
        union
        {
            box
            { 
                <-0.5, 0, -0.5>, <0.5, 1, 0.5>
                
                texture { DoorTexture rotate 90*z }
                texture
                {
                    pigment
                    {
                        image_map
                        {
                            png "door-motive02.png"
                            interpolate 2
                            once
                        }
                        
                        translate <-0.5, -0.5, 0>
                        scale 0.35
                        rotate -90 * x
                    }
                }

                scale <(x2 - x1) * 0.4, 5.0, 2.5>
                rotate 90 * x 
                translate <0, 1.75, -0.18>
            }

            box
            { 
                <-0.5, 0, -0.5>, <0.5, 1, 0.5>

                texture { DoorTexture rotate 90*z }
                texture
                {
                    pigment
                    {
                        image_map
                        {
                            png "door-motive02.png"
                            interpolate 2
                            once
                        }
                        
                        translate <-0.5, -0.5, 0>
                        scale <1, -1, 1> * 0.35
                        rotate -90 * x
                    }
                }
                
                scale <(x2 - x1) * 0.4, 5.0, 2.5>
                rotate 90 * x
                translate <0, 4.75, -0.18>
            }
            
            box
            { 
                <-0.5, 0, -0.5>, <0.5, 1, 0.5>

                texture { DoorTexture }
                texture
                {
                    pigment
                    {
                        image_map
                        {
                            png "door-motive01.png"
                            interpolate 2
                            once
                        }
                        
                        translate <-0.5, -0.5, 0>
                        scale <0.4, 1, 1> * 0.5
                        rotate -90 * x
                    }
                }
                scale <(x2 - x1) * 0.4, 5.0, 1.0>
                rotate 90 * x
                
                translate <0, 6.75, -0.18>
            }
        }
        
        object { Boxes translate -(x2 - x1) / 4 * x }
        object { Boxes scale <-1, 1, 1> translate -(x2 - x1) * 3 / 4 * x }

        // Handle
        
        union
        {
            height_field
            {
                png "door-handle.png"
                water_level 0.01
                smooth
                
                scale <1, 0.1, 253/808> * 1.1
                rotate -90 * x
                translate -0.3*z
            }
            
            cylinder
            {
                <0, 0.2, 0>, <0, 0.2, -0.3>, 0.08
            }
            
            texture
            {
                pigment 
                { 
                    color rgb <0.6, 0.58, 0.55> * 0.5 
                }
                
                finish
                {
                    specular 0.2
                    roughness 0.01
                    metallic
                }
            }
            
            scale 0.75
            translate <- (x2 - x1) + 0.35, 3.25, -0.25>
        }
        

        scale <0.9, 1, 1>
        rotate 50*y
        translate <x2, 0, z1 + 0.05>
    }
    
    // Tips
    
    object { Tip scale 0.4 translate <x2, DoorHeight/3, z1 + 0.01 + 0.035> }
    object { Tip scale 0.4 translate <x2, DoorHeight*2/3, z1 + 0.01 + 0.035> }
    
}

