// Persistence of Vision Ray Tracer Scene Description File
// File: remembrance.pov
// Vers: 3.5
// Desc: Scene file for "In Remembrance is Hope" - IRTC Decay round, September-October 2003
// Date: October 22, 2003
// Auth: Philip Chan

#include "shapes.inc"
#include "functions.inc"
#include "tank/tank.inc"
#include "poppy/poppy.inc"

//----- Rendering Parameters -----\\
#declare use_quality = true; // disables all quality settings

#declare Use_Concrete = (true & use_quality);  // use concrete texture
#declare hf_smooth = (true & use_quality);  // smooth height field
#declare Use_Area_Light = ( true & use_quality);  // use area light for smooth shadows

#declare brightness_factor = 0;  // added onto radiosity brightness value
#declare rad_quality = (false & use_quality);  // enable radiosity quality factor variables
#declare rad_up_factor = 1.5*rad_quality;  // percentage increase for "higher is better" values
#declare rad_down_factor = 0.33*rad_quality;  // percentage decrease for "lower is better" values
//----- End of Rendering Parameters -----\\

//----- Lighting -----\\
// radiosity
global_settings {
	ambient_light 0
	radiosity {
		pretrace_start 0.08*( 1 - rad_down_factor )
		pretrace_end   0.04*( 1 - rad_down_factor )
		count floor( 35*( 1 + rad_up_factor ) )
		
		nearest_count floor( 5*( 1 + rad_up_factor ) )
		error_bound 1.3
		recursion_limit 3
		
		low_error_factor 0.5*( 1 - rad_down_factor )
		gray_threshold 0.5
		minimum_reuse 0.015*( 1 - rad_down_factor )
		brightness 2 + brightness_factor
		
		adc_bailout 0.01/2
	}
}
// light_source
light_source {
	<400, 1000, 500>
	colour rgb <1, 1, 1>
	#if( Use_Area_Light )
		area_light <150, 0, 0>, <0, 375, 0>, 10, 25
		adaptive 1
		jitter
	#end
	parallel
	point_at <-8, 0, 50>
}
//----- End Lighting -----\\

//----- Camera -----\\
camera {
	location <9, 10, 0>
	look_at <-9, 0, 50>
	/*
	// Unfortunately, even these ridiculous settings blur out the terrain texture
	focal_point <10, 0, 18.25>
	aperture 0.001
	blur_samples 1000
	confidence 0.999
	variance 1/1000000
	*/
}
//----- End Camera -----\\

//----- Fog -----\\
fog {
	distance 250
	turbulence 0.5
	colour rgb <0.50, 0.30, 0.14>*0.35
	fog_type 1
}
fog {
	distance 30
	turbulence 0.25
	colour rgb <0.50, 0.30, 0.14>*0.65
	fog_type 2
	fog_alt 10
	fog_offset 5
}
//----- End Fog -----\\

//----- Terrain -----\\
// Terrain object (declare for use in trace function)
#declare Terrain = union {
	difference {
		height_field {
			tga "terrain\terrain_hf.tga"
			#if ( hf_smooth )
				smooth
			#end
			scale <1000, 0.75, 1000>
			translate <-500, 0, -500 + 3>
		}
		box { <0, 0, -20>, <15, 2, 0.5>
			translate <7, 0, 25>
		}
	}
	plane { y, 0 }
}
// Place terrain 
object { Terrain
	pigment {
		granite
		colour_map {
			[0.00 colour rgb <0.40, 0.25, 0.13> ]
			[0.07 colour rgb <0.40, 0.25, 0.13> ]
			[0.07 colour rgb <0.60, 0.37, 0.20> ]
			[0.93 colour rgb <0.60, 0.37, 0.20> ]
			[0.93 colour rgb <0.80, 0.49, 0.31> ]
			[1.00 colour rgb <0.80, 0.49, 0.31> ]
		}
		frequency 15
	}
}

//----- Declarations for Rocks and Walls -----\\
// Pigment noise function for rocks and walls
#declare fn_Pigm = function {
	pigment {
		agate
		color_map {
			[0 color rgb 0]
			[1 color rgb 1]
		}
	}
}

// Concrete texture for rocks and walls
#declare Dark_Concrete = texture {
   pigment {
      agate
      turbulence 2.0
      colour_map {
         [0.00 colour rgb <0.25, 0.25, 0.25> ]
         [0.10 colour rgb <0.25, 0.25, 0.25> ]
         [0.20 colour rgb <0.40, 0.40, 0.40> ]
         [0.60 colour rgb <0.40, 0.40, 0.40> ]
         [0.70 colour rgb <0.55, 0.55, 0.55> ]
         [0.75 colour rgb <0.55, 0.55, 0.55> ]
         [0.80 colour rgb <0.40, 0.40, 0.40> ]
         [1.00 colour rgb <0.40, 0.40, 0.40> ]
      }
   }
   normal {
      granite
      turbulence 2.0
   }
   scale 1.0
   finish { 
      reflection 0.0
      diffuse 1.0
   }
}
//----- End Declarations -----\\

//----- Rocks -----\\
#declare Rrand = seed(734);  // random seed for rock placement and sizing
// near rocks (use isosurface and concrete texture)
#declare i = 0;
#while ( i < 50 )
	// generate random x-z plane locaton and radius
	#declare xpos = 140*( rand(Rrand) - 5/6 );
	#declare zpos = 75*( rand(Rrand) );
	#declare rad = 0.2 + 0.3*rand(Rrand);
	// calculate y position using trace function, and move up a bit (based on radius)
	#declare ypos = trace( Terrain, <xpos, 5, zpos>, -y ).y + 0.5*rad*rand(Rrand);
	// calculate "scale factors" for axes in sphere function
	#declare xscale = 0.5 + rand(Rrand);
	#declare yscale = 0.5 + rand(Rrand);
	#declare zscale = 0.5 + rand(Rrand);
	// generate random noise strength
	#declare noise = 0.3 + 0.1*rand(Rrand);
	// rock object
	isosurface {
		function {
			// scaled sphere function, centred at random location
			f_sphere( xscale*(x-xpos), yscale*(y-ypos), zscale*(z-zpos), rad )
			// noise function to affect shape
			+ noise*rad*fn_Pigm( 5*x, 5*y, 5*z ).gray
			// finer noise function for more detailed surface
			+ (noise/2)*rad*fn_Pigm( 8*(x + 168), 8*(y + 882), 8*(z + 308) ).gray
		}
		contained_by { sphere { <xpos, ypos, zpos>, rad }	}
	  accuracy 0.001
	  max_gradient 10
	  #if ( Use_Concrete )
	  	texture { Dark_Concrete }
	  #else
	  	pigment { colour rgb <0.5, 0.5, 0.5> }
	  #end
	}
	#declare i = i + 1;
#end
// far rocks (grey sphere objects)
#declare i = 0;
#while ( i < 400 )
	// generate random x-z plane locaton and radius
	#declare xpos = 600*( rand(Rrand) - 5/6 );
	#declare zpos = 75 + 500*( rand(Rrand) );
	#declare rad = 0.4 + (0.75 + 2.5*(zpos - 150)/450)*rand(Rrand);
	// calculate y position using trace function, and move up a bit (based on radius)
	#declare ypos = trace( Terrain, <xpos, 5, zpos>, -y ).y + 0.25*rad*rand(Rrand);
	// calculate "scale factors" for axes in sphere function
	#declare xscale = 0.5 + rand(Rrand);
	#declare yscale = 0.5 + rand(Rrand);
	#declare zscale = 0.5 + rand(Rrand);
	sphere { <0, 0, 0>, rad
		scale <xscale, yscale, zscale>
		translate <xpos, ypos, zpos>
  	pigment { colour rgb <0.5, 0.5, 0.5> }
	}
	#declare i = i + 1;
#end
//----- End Rocks -----\\

//----- Background -----\\
// walls along x-axis, from near to far
difference {
	union {
		isosurface {
		  function {
		  	y - 15  // height 15
		  	+ 18*( fn_Pigm(0.015*x, 478, 11).gray )  // major noise, cuts out sections of wall
		  	+ 3*( fn_Pigm(0.1*x, 273, 279).gray )  // noise to make more jagged
		  	+ 1*( fn_Pigm(0.4*x, 11, 261).gray )  // finer noise to make even more jagged
		  }
		  contained_by { box { <-525, 0, 350>, <100, 15, 351> } }
		  accuracy 0.001
		  max_gradient 4
		}
		isosurface {
		  function {
		  	y - 10
		  	+ 13*( fn_Pigm(0.02*x, 348, 14).gray )
		  	+ 3*( fn_Pigm(0.1*x, 324, 461).gray )
		  	+ 1*( fn_Pigm(0.4*x, 610, 45).gray )
		  }
		  contained_by { box { <-525, 0, 475>, <100, 10, 476> } }
		  accuracy 0.001
		  max_gradient 4
		}
		isosurface {
		  function {
		  	y - 15
		  	+ 18*( fn_Pigm(0.015*x, 437, 59).gray )
		  	+ 3*( fn_Pigm(0.1*x, 375, 634).gray )
		  	+ 1*( fn_Pigm(0.4*x, 6, 150).gray )
		  }
		  contained_by { box { <-525, 0, 600>, <100, 15, 601> } }
		  accuracy 0.001
		  max_gradient 4
		}
		isosurface {
		  function {
		  	y - 15
		  	+ 18*( fn_Pigm(0.015*x, 201, 636).gray )
		  	+ 3*( fn_Pigm(0.1*x, 647, 887).gray )
		  	+ 1*( fn_Pigm(0.4*x, 463, 652).gray )
		  }
		  contained_by { box { <-525, 0, 625>, <100, 15, 626> } }
		  accuracy 0.001
		  max_gradient 4
		}
	}
	// cut out "alleys"
	box { <-299, -1, 349>, <-275, 16, 977> }
	box { <-49, -1, 349>, <-25, 16, 977> }
	pigment { colour rgb <0.5, 0.5, 0.5> }
}
// walls along z-axis, left to right
difference {
	union {
		isosurface {
		  function {
		  	y - 15
		  	+ 18*( fn_Pigm(839, 299, 0.015*z).gray )
		  	+ 3*( fn_Pigm(891, 495, 0.1*z).gray )
		  	+ 1*( fn_Pigm(946, 766, 0.4*z).gray )
		  }
		  contained_by { box { <-550, 0, 350>, <-549, 15, 626> } }
		  accuracy 0.001
		  max_gradient 4
		}
		isosurface {
		  function {
		  	y - 15
		  	+ 18*( fn_Pigm(145, 357, 0.015*z).gray )
		  	+ 3*( fn_Pigm(211, 866, 0.1*z).gray )
		  	+ 1*( fn_Pigm(619, 238, 0.4*z).gray )
		  }
		  contained_by { box { <-525, 0, 350>, <-524, 15, 626> } }
		  accuracy 0.001
		  max_gradient 4
		}
		isosurface {
		  function {
		  	y - 15
		  	+ 18*( fn_Pigm(400, 205, 0.015*z).gray )
		  	+ 3*( fn_Pigm(736, 944, 0.1*z).gray )
		  	+ 1*( fn_Pigm(477, 978, 0.4*z).gray )
		  }
		  contained_by { box { <-300, 0, 350>, <-299, 15, 626> } }
		  accuracy 0.001
		  max_gradient 4
		}
		isosurface {
		  function {
		  	y - 15
		  	+ 18*( fn_Pigm(932, 437, 0.015*z).gray )
		  	+ 3*( fn_Pigm(61, 403, 0.1*z).gray )
		  	+ 1*( fn_Pigm(495, 399, 0.4*z).gray )
		  }
		  contained_by { box { <-275, 0, 350>, <-274, 15, 626> } }
		  accuracy 0.001
		  max_gradient 4
		}
		isosurface {
		  function {
		  	y - 10
		  	+ 13*( fn_Pigm(46, 67, 0.02*z).gray )
		  	+ 3*( fn_Pigm(547, 590, 0.1*z).gray )
		  	+ 1*( fn_Pigm(576, 266, 0.4*z).gray )
		  }
		  contained_by { box { <-150, 0, 350>, <-149, 10, 626> } }
		  accuracy 0.001
		  max_gradient 4
		}
		isosurface {
		  function {
		  	y - 15
		  	+ 18*( fn_Pigm(722, 611, 0.015*z).gray )
		  	+ 3*( fn_Pigm(62, 436, 0.1*z).gray )
		  	+ 1*( fn_Pigm(183, 210, 0.4*z).gray )
		  }
		  contained_by { box { <-50, 0, 350>, <-49, 15, 626> } }
		  accuracy 0.001
		  max_gradient 4
		}
		isosurface {
		  function {
		  	y - 15
		  	+ 18*( fn_Pigm(285, 564, 0.015*z).gray )
		  	+ 3*( fn_Pigm(976, 452, 0.1*z).gray )
		  	+ 1*( fn_Pigm(832, 157, 0.4*z).gray )
		  }
		  contained_by { box { <-25, 0, 350>, <-24, 15, 626> } }
		  accuracy 0.001
		  max_gradient 4
		}
		isosurface {
		  function {
		  	y - 10
		  	+ 13*( fn_Pigm(119, 743, 0.02*z).gray )
		  	+ 3*( fn_Pigm(94, 164, 0.1*z).gray )
		  	+ 1*( fn_Pigm(784, 43, 0.4*z).gray )
		  }
		  contained_by { box { <100, 0, 350>, <101, 10, 626> } }
		  accuracy 0.001
		  max_gradient 4
		}
	}
	// cut out "alleys"
	box { <-551, -1, 601>, <477, 16, 627> }
	pigment { colour rgb <0.5, 0.5, 0.5> }
}

/*
#declare Sun_Media = 350;
cylinder { <0, 0, 0>, <0, 0, 1>, 50
	hollow
	pigment { colour rgbt <1, 1, 1, 1> }
	interior {
		media {
			emission <1, 1, 1.00>
			intervals 20
			density {
				spherical
				turbulence 0.1
				density_map {
					[ 0.00 colour rgb <0.25, 0.15, 0.07>*Sun_Media ]
					[ 0.10 colour rgb <0.40, 0.40, 0.20>*Sun_Media ]
					[ 0.20 colour rgb <0.50, 0.50, 0.25>*Sun_Media ]
					[ 0.45 colour rgb <0.90, 0.90, 0.45>*Sun_Media ]
					[ 0.70 colour rgb <1.20, 1.20, 0.60>*Sun_Media ]
					[ 0.85 colour rgb <1.40, 1.40, 0.70>*Sun_Media ]
					[ 1.00 colour rgb <1.40, 1.40, 0.70>*Sun_Media ]
				}
				scale 50
			}
		}
	}
	translate <75, 400, 2000>
}
*/

//----- End Background -----\\
	
//----- Tank -----\\
object { Tank
	translate <-8, 0, 50>
}
//----- End Tank -----\\

//----- Road Sign -----\\
// text on sign, declared for use in object pattern
#declare Sign = Center_Object (
	text { ttf "verdanab.ttf"
		"MAIN ST", 0.1, 0
		translate -z*0.05
	},
	x + y  // centre about x and y axes
)

// Pigment for rusted scratches on sign
#declare Rust_Scratches = pigment {
  spotted
  turbulence 0.1
  colour_map {
		[0.00 colour rgbt <0.228, 0.075, 0.036, 0>]
    [0.15 colour rgbt <0.341, 0.112, 0.053, 0>] 
    [0.25 colour rgbt <0.455, 0.149, 0.071, 0.5>]
    [0.26 colour rgbt <0.455, 0.149, 0.071, 1>]
    [1.00 colour rgbt <0.455, 0.149, 0.071, 1>]
	}
}

// sign
union {
	// box with rounded corners
	box { <-3, 0, -0.5>, <3, 0.1, 0.5> }
	box { <-2.8, 0, -0.7>, <2.8, 0.1, 0.7> }
	cylinder { <-2.8, 0, -0.5>, <-2.8, 0.1, -0.5>, 0.2 }
	cylinder { <-2.8, 0, 0.5>, <-2.8, 0.1, 0.5>, 0.2 }
	cylinder { <2.8, 0, -0.5>, <2.8, 0.1, -0.5>, 0.2 }
	cylinder { <2.8, 0, 0.5>, <2.8, 0.1, 0.5>, 0.2 }
	// base texture, green sign with text
	texture {
		pigment {
			object { Sign
				rotate x*90
				translate y*0.1
				colour rgb <0.2, 0.75, 0.2>
				colour White
			}
		}
	}
	// fading texture, to grey out some areas
	texture {
		pigment {
			bozo
			turbulence 0.3
			colour_map {
				[0.00 colour rgbf <0.919, 0.890, 1.000, 0.480>]
				[0.50 colour rgbf <0.632, 0.612, 0.688, 0.698>]
				[1.00 colour rgbf <0.546, 0.597, 0.676, 0.953>]
			}
		}
		scale 0.6
	}
	// rusted scratches, "patterns" along three directions
	texture { pigment { Rust_Scratches }
		translate z*694  // used to "control" pattern
		scale <0.15, 0.75, 1>
		rotate y*20
	}
	texture { pigment { Rust_Scratches }
		translate z*966
		scale <0.15, 0.75, 1>
		rotate y*108
	}
	texture { pigment { Rust_Scratches }
		translate z*232
		scale <0.15, 0.75, 1>
		rotate y*141
	}
	scale 0.7
	rotate y*35
	rotate -z*10
	translate <-4, 0.35, 16.2>
}
//----- End Road Sign -----\\

//----- Near Walls -----\\
union {
	// left wall
	isosurface {
	  function {
  		(y - 5)	 // height of 5
  		- (z-0.5)/abs(z-0.5 + 0.001*(z = 0.5))*pow( abs(0.2*(z - 0.5)), 1.2 )  // shrink wall height in -z direction
  		+ 0.1*fn_Pigm(x, y, z).gray  // random noise
  		+ (fn_Pigm(1, 1, 0.05*z).gray - 0.5)  // height noise
	  	+ 20*(x > 0.5)*(  // falloff behaviour for x > 0.5
	  		(x - 0.5)  // linear falloff
	  		- 0.01*fn_Pigm(2*x, y, z).gray  // noise
	  	)
	  }
	  contained_by { box { <-0.5, 0, -14.5>, <0.75, 6, 0.5> } }
	  accuracy 0.001
	  max_gradient 4
	}
	// back wall
	isosurface {
		function {
			(y - 4.9)	 // height 5
			+ (x+0.5)/abs(x+0.5 + 0.001*(x = -0.5))*pow( abs(0.1*(x + 0.5)), 1.2)  // shrink wall height in x direction
			+ 0.1*fn_Pigm(x, y, z).gray  // random noise
			+ (fn_Pigm(1, 1, 0.05*x).gray - 0.5)  // height noise
			+ 20*(z < -0.5)*(  // falloff behaviour for z < -0.5
				(-z - 0.5)  // linear falloff
				- 0.01*fn_Pigm(x, y, 2*z).gray  // noise
			)
		}
		contained_by { box { <-0.5, 0, -0.35>, <9.5, 6, 0.5> } }
		accuracy 0.001
		max_gradient 2.720
		translate z*0.4
	}
	translate <7, 0, 25>
	texture {
		#if ( Use_Concrete )
			Dark_Concrete
		#else
			pigment { colour rgb <0.5, 0.5, 0.5> }
		#end
	}
}
//----- End Near Walls -----\\

//----- Pictures -----\\
// pigment for yellowed fading of the pictures
#declare Paper_Fade = pigment {
	bozo
	turbulence 0.3
	colour_map {
		[0.00 color rgbft<1.00, 0.92, 0.37, 0.5, 0.5>]
		[0.40 color rgbft<0.71, 0.65, 0.26, 0.8, 0.2>]
		[1.00 color rgbft<0.53, 0.49, 0.20, 1.0, 0.0>]
	}
	scale 1
}

// crinkled texture for pictures
#declare Crinkled = normal {
	crackle 5
	scale <0.2, 1, 0.2>
}

// pictures
union {
	// Hiroshima
	isosurface {
		function {
			(-z - 0.001)	 // height 0.001
			+ (x < 0.2)*(
				(-x + 0.2)	// falloff behaviour
				- 0.2*fn_Pigm(0.5*x, 0.5*y, 1).gray  // noise
			)
			+ (y < 0.2)*(
				(-y + 0.2)
				- 0.2*fn_Pigm(0.5*x, 0.5*y, 1).gray
			)
			+ (x > 2.8)*(
				(x - 2.8)
				- 0.2*fn_Pigm(0.5*x, 0.5*y, 1).gray
			)
			+ (y > 3.8)*(
				(y - 3.8)
				- 0.2*fn_Pigm(0.5*x, 0.5*y, 1).gray
			)
		}
		contained_by { box { <0, 0, 0>, <3, 4, -0.001> } }
		accuracy 0.001
		max_gradient 2.3
		texture {
			pigment {
				image_map {
					gif "Hiroshima.gif"
					once
				}
				scale <3, 4, 1>
			}
		}
		texture { pigment { Paper_Fade } }
		normal { Crinkled }
		rotate x*90
		translate <-1.5, 0, -2>
		scale 0.75
		rotate -y*10
		translate <8.5, 0, 17>
	}
	// Lusitania
	isosurface {
		function {
			(-z - 0.0005)	 // height 0.0005
			+ (x < 0.15)*(
				(-x + 0.15)
				- 0.2*fn_Pigm(0.7*x, 0.9*y, 0).gray
			)
			+ (y < 0.15)*(
				(-y + 0.15)
				- 0.2*fn_Pigm(0.7*x, 0.9*y, 0).gray
			)
			+ (x > 2.85)*(
				(x - 2.85)
				- 0.2*fn_Pigm(0.7*x, 0.9*y, 0).gray
			)
			+ (y > 1.85)*(
				(y - 1.85)
				- 0.2*fn_Pigm(0.7*x, 0.9*y, 0).gray
			)
		}
		contained_by { box { <0, 0, 0>, <3, 4, -0.0006> } }
		accuracy 0.001
		max_gradient 2.3
		texture {
			pigment {
				image_map {
					jpeg "Lusitania.jpg"
					once
				}
				scale <3, 2, 1>
			}
		}
		texture { pigment { Paper_Fade } }
		normal { Crinkled }
		rotate x*90
		translate <-1.5, 0, -1>
		scale 0.9
		rotate -y*20
		translate <7.5, 0, 19>
	}
	// World Trade Center
	isosurface {
		function {
			(-z - 0.0002)	 // height 0.0005
			+ (x < 0.075)*(
				(-x + 0.075)
				- 0.1*fn_Pigm(1.5*x, 1.5*y, 2).gray
			)
			+ (y < 0.075)*(
				(-y + 0.075)
				- 0.1*fn_Pigm(1.5*x, 1.5*y, 2).gray
			)
			+ (x > 0.925)*(
				(x - 0.925)
				- 0.1*fn_Pigm(1.5*x, 1.5*y, 2).gray
			)
			+ (y > 0.925)*(
				(y - 0.925)
				- 0.1*fn_Pigm(1.5*x, 1.5*y, 2).gray
			)
		}
		contained_by { box { <0, 0, 0>, <1, 1, -0.0002> } }
		accuracy 0.001
		max_gradient 2.3
		texture {
			pigment {
				image_map {
					jpeg "WTC.jpg"
					once
				}
			}
		}
		texture { pigment { Paper_Fade } }
		normal { Crinkled
			scale 1/2.5
		}
		rotate x*90
		translate <-0.5, 0, -0.5>
		scale 2.5
		rotate y*5
		translate <10.2, 0, 18.5>
	}
	translate <2, 0, 1>
}
//----- End Pictures -----\\

//----- Poppy -----\\
object { Poppy
	translate y*0.03
	rotate y*45
	scale <0.12, 1, 0.12>
	translate <11, 0.001, 19.5>
}
//----- End Poppy -----\\

//----- Plant -----\\
union {
	// right stem
	sphere_sweep {
		b_spline
		6,
		<0, -0.5, 0>, 0.10
		<0, 0, 0>, 0.10
		<0, 0.3, 0>, 0.06
		<0.2, 0.75, 0>, 0.04
		<0.4, 0.8, 0>, 0.02
		<0.6, 0.85, 0>, 0.02
	}
	// right leaf
	prism {
		cubic_spline
		0, 0.02, 11,
		<0.000, -0.250>,
		<-0.177, -0.177>,
		<-0.250, 0.000>,
		<-0.177 + 0.02, 0.177 + 0.02>,
		<0.000, 0.250 + 0.15>,
		<0.177 + 0.03, 0.177 + 0.03>,
		<0.250, 0.000>,
		<0.177, -0.177>,
		<0.000, -0.250>,
		<-0.177, -0.177>,
		<-0.250, 0.000>
		scale 0.75
		rotate y*90
		rotate z*10
		translate <0.5, 0.85, 0>
	}
	// left stem
	sphere_sweep {
		b_spline
		6,
		<0, -0.5, 0>, 0.10
		<0, 0, 0>, 0.10
		<0, 0.5, 0>, 0.06
		<-0.1, 0.75, 0>, 0.04
		<-0.2, 0.8, 0>, 0.02
		<-0.3, 0.85, 0>, 0.02
	}
	// left leaf
	prism {
		cubic_spline
		0, 0.02, 11,
		<0.000, -0.250>,
		<-0.177, -0.177>,
		<-0.250, 0.000>,
		<-0.177 + 0.02, 0.177 + 0.02>,
		<0.000, 0.250 + 0.15>,
		<0.177 + 0.03, 0.177 + 0.03>,
		<0.250, 0.000>,
		<0.177, -0.177>,
		<0.000, -0.250>,
		<-0.177, -0.177>,
		<-0.250, 0.000>
		scale 0.75
		rotate -y*90
		rotate -z*15
		translate <-0.25, 0.85, 0>
	}		
	pigment { colour Green }
	scale 0.3
	translate <9, 0, 17>
}
//----- End Plant -----\\