// Persistence of Vision Ray Tracer Scene Description File
// File: exodus.pov
// Vers: 3.5
// Desc: Scene file for "Exodus" - IRTC Future round, November-December 2003
// Date: December 29th, 2003
// Auth: Philip Chan

// +W1024 +H576 +A0.1 +AM2

#include "pc_stars.inc"
#include "colors.inc"

//----- Rendering Parameters -----\\
#declare Use_Quality = true; // disables all quality settings
#declare Camera_Select = 0;  // selects between render view and test views

#declare HF_Smooth = (true & Use_Quality);  // smooth height field
#declare Use_Planet_Media = (true & Use_Quality);  // scattering media around Earth
//----- End of Rendering Parameters -----\\

//----- Lighting -----\\
light_source {
	<400, 1000, -500>
	colour rgb <1, 1, 1>
	parallel
	point_at <-8, 0, 50>
}
//----- End Lighting -----\\

//----- Camera -----\\
#switch ( Camera_Select )
	// top down on Hub
	#case ( 1 )
		camera {
			location <-52.5, 100, 500/3>
			look_at <-52.5, 0, 500/3>
		}
	#break
	// top down zoomed on hydroponics
	#case ( 2 )
		camera {
			location <-65, 30, 470/3>
			look_at <-65, 0, 470/3>
		}
	#break
	// top down on mine
	#case ( 3 )
		camera {
			location <0, 100, 500>
			look_at <0, 0, 500>
		}
	#break
#else
	// scene view
	camera {
		up 0.6*y
		right 16/15*x
		location <-15, 20, 0>
		look_at <-15, 20, 50>
	}
#end
//----- End Camera -----\\

//----- Earth -----\\
union {
	// image map for surface
	sphere { <0, 0, 0>, 5000
		pigment {
			image_map {
				png "earth/earthmap1k.png"
				map_type 1
			}
		}
	}
	// clouds
	sphere { <0, 0, 0>, 5010
		pigment { 
			bozo
			lambda 3
			omega 0.7
			turbulence 0.5
			colour_map {
				[ 0.00 colour rgbt <1.00, 1.00, 0.60, 0> ]
				[ 0.10 colour rgbt <1.00, 1.00, 0.60, 0> ]
				[ 0.65 colour rgbt <1.00, 1.00, 0.60, 1> ]
				[ 1.00 colour rgbt <1.00, 1.00, 0.60, 1> ]
			}
			scale <400, 300, 300>
		}
	}
	// media to give yellowish glow
	#if ( Use_Planet_Media )
		sphere { <0, 0, 0>, 5100
			hollow
			pigment { colour rgbt <1, 1, 1, 1> }
			interior {
				media {
					scattering { 1, 1 }
					intervals 5
					density {
						spherical
						colour_map {
							[ 0.000 colour rgb <0, 0, 0> ]
							[ 0.025 colour rgb <0.7, 0.7, 0.5>*0.00075 ]
							[ 1.000 colour rgb <0.7, 0.7, 0.5>*0.00075 ]
						}
						scale 5100
					}
				}
			}
		}
	#end
	rotate y*140
	rotate -x*20
	rotate z*20
	translate <-25000, 13500, 70000>
	finish {
		diffuse 1.0
		brilliance 7
	}
}
//----- End Earth -----\\

Stars( 0.75, <1, 1, 0> )  // starfield

//----- Terrain -----\\
#declare T_rotate = 5; // -x rotation of the terrain, used for other objects to set them on the terrain properly

// Terrain object (declare for use in trace function)
#declare Terrain = height_field {
	tga "terrain\terrain_hf.tga"
	#if ( HF_Smooth )
		smooth
	#end
	scale <700, 20, 1000>
	translate <-350, -10, 0>
}

// Place terrain 
object { Terrain
	pigment {
		granite
		colour_map {
			[ 0.000 colour rgb <0.50, 0.50, 0.50> ]
			[ 0.090 colour rgb <0.50, 0.50, 0.50> ]
			[ 0.090 colour rgb <0.60, 0.60, 0.60> ]
			[ 0.180 colour rgb <0.60, 0.60, 0.60> ]
			[ 0.180 colour rgb <0.70, 0.70, 0.70> ]
			[ 0.270 colour rgb <0.70, 0.70, 0.70> ]
			[ 0.270 colour rgb <0.60, 0.60, 0.60> ]
			[ 0.360 colour rgb <0.60, 0.60, 0.60> ]
			[ 0.360 colour rgb <0.50, 0.50, 0.50> ]
			[ 0.450 colour rgb <0.50, 0.50, 0.50> ]
			[ 0.450 colour rgb <0.60, 0.60, 0.60> ]
			[ 0.550 colour rgb <0.60, 0.60, 0.60> ]
			[ 0.550 colour rgb <0.70, 0.70, 0.70> ]
			[ 0.640 colour rgb <0.70, 0.70, 0.70> ]
			[ 0.640 colour rgb <0.60, 0.60, 0.60> ]
			[ 0.730 colour rgb <0.60, 0.60, 0.60> ]
			[ 0.730 colour rgb <0.50, 0.50, 0.50> ]
			[ 0.820 colour rgb <0.50, 0.50, 0.50> ]
			[ 0.820 colour rgb <0.60, 0.60, 0.60> ]
			[ 0.910 colour rgb <0.60, 0.60, 0.60> ]
			[ 0.910 colour rgb <0.70, 0.70, 0.70> ]
			[ 1.000 colour rgb <0.70, 0.70, 0.70> ]
		}
	}
	rotate -x*T_rotate
}
//----- End Terrain -----\\

//----- Plant -----\\
// basic plant shape for use in hydroponics
#declare Plant = union {
	// right stem
	sphere_sweep {
		b_spline
		6,
		<0, -0.5, 0>, 0.10
		<0, 0, 0>, 0.10
		<0, 0.3, 0>, 0.06
		<0.2, 0.75, 0>, 0.04
		<0.4, 0.8, 0>, 0.02
		<0.6, 0.85, 0>, 0.02
	}
	// right leaf
	prism {
		cubic_spline
		0, 0.02, 11,
		<0.000, -0.250>,
		<-0.177, -0.177>,
		<-0.250, 0.000>,
		<-0.177 + 0.02, 0.177 + 0.02>,
		<0.000, 0.250 + 0.15>,
		<0.177 + 0.03, 0.177 + 0.03>,
		<0.250, 0.000>,
		<0.177, -0.177>,
		<0.000, -0.250>,
		<-0.177, -0.177>,
		<-0.250, 0.000>
		scale 0.75
		rotate y*90
		rotate z*10
		translate <0.5, 0.85, 0>
	}
	// left stem
	sphere_sweep {
		b_spline
		6,
		<0, -0.5, 0>, 0.10
		<0, 0, 0>, 0.10
		<0, 0.5, 0>, 0.06
		<-0.1, 0.75, 0>, 0.04
		<-0.2, 0.8, 0>, 0.02
		<-0.3, 0.85, 0>, 0.02
	}
	// left leaf
	prism {
		cubic_spline
		0, 0.02, 11,
		<0.000, -0.250>,
		<-0.177, -0.177>,
		<-0.250, 0.000>,
		<-0.177 + 0.02, 0.177 + 0.02>,
		<0.000, 0.250 + 0.15>,
		<0.177 + 0.03, 0.177 + 0.03>,
		<0.250, 0.000>,
		<0.177, -0.177>,
		<0.000, -0.250>,
		<-0.177, -0.177>,
		<-0.250, 0.000>
		scale 0.75
		rotate -y*90
		rotate -z*15
		translate <-0.25, 0.85, 0>
	}		
	pigment { colour Green }
}

// Stalk macro to stack sections of the plant object with random rotations
#declare Stalk_Rand = seed(64713);
#macro Stalk ( sections, scaling, rotation, translation )
	union {
		#local i = 0;
		#while ( i < sections )
			object { Plant
				rotate y*rand(Stalk_Rand)*360
				translate y*i*0.85
			}
			#local i = i + 1;
		#end
		scale scaling
		rotate rotation
		translate translation
	}
#end
//----- End Plant -----\\

//----- Hub and Mine Textures -----\\
// Metal covered by semi-transparent fading layer
#declare Dull_Metal = texture {
	pigment { colour rgb <0.85, 0.85, 0.85> }
	finish {
		brilliance 1.5
		specular 0.9
		roughness 0.02
		reflection 0.05
		metallic
	}
}
texture {
	pigment {
		bozo
		turbulence 0.3
		colour_map {
			[0.00 color rgbf<1.000, 1.000, 1.000, 1.000>]
			[0.50 color rgbf<0.632, 0.612, 0.688, 0.698>]
			[1.00 color rgbf<0.546, 0.597, 0.676, 0.953>]
		}
	}
	scale 0.3
}

// dark tinted glass
#declare Tinted_Glass = material {
	texture {
		pigment { colour rgbft <0.4, 0.4, 0.2, 0.5, 0.5> }
		finish {
			specular 0.1
			roughness 0.03
			reflection 0.05
		}
	}
	interior { ior 1.4 }
}
//----- End Hub and Mine Textures -----\\

//----- Hub -----\\
#declare Hub_Centre = trace( Terrain, <-52.5, 50, 500/3>, -y ) - y*0.1;  // last term is because the height field is not completely smooth

// Tunnel coming in from left
union {
	// body
	difference {
		box { <-40, 0, -1>, <0, 1.5, 1> }
		box { <-41, 0.1, -0.8>, <1, 2, 0.8> }
	}
	// glass
	difference {
		cylinder { <-130, 1.5, 0>, <0, 1.5, 0>, 1.00 }
		cylinder { <-130, 1.5, 0>, <0, 1.5, 0>, 0.90 }
		pigment { colour rgbft <0.4, 0.4, 0.2, 0.5, 0.5> }
	}
	// metal sections over glass
	#declare i = -5;
	#while ( i > -40 )
		difference {
			cylinder { <i - 0.25, 1.5, 0>, <i + 0.25, 1.5, 0>, 1.1 }
			cylinder { <i - 0.3, 1.5, 0>, <i + 0.3, 1.5, 0>, 0.8 }
		}
		box { <i - 0.25, 0, -1.1>, <i + 0.25, 1.5, -1> }
		box { <i - 0.25, 0, 1>, <i + 0.25, 1.5, 1.1> } 
		#declare i = i - 5;
	#end
	texture { Dull_Metal }
	translate Hub_Centre - x*18
	rotate -x*T_rotate
}

// communications satellite for buildings
#declare Satellite = union {
	// post
	cylinder { <0, 0, 0>, <0, 0.5, 0>, 0.25 }
	sphere { <0, 0.5, 0>, 0.25 }
	union {
		// mount
		box { <0, -0.05, 0.25>, <0.48, 0.05, 0.3> }
		box { <0, -0.05, -0.3>, <0.48, 0.05, -0.25> }
		// dish
		difference {
			sphere { <1.2, 0, 0>, 0.8
				texture { Dull_Metal }
			}
			sphere { <1.2, 0, 0>, 0.75
				pigment { colour White }
			}
			plane { -x, -0.6
				pigment { colour White }
			}
		}
		// point coming out of dish
		cone { <0.4, 0, 0>, 0.1, <0.8, 0, 0>, 0.025 }
		sphere { <0.8, 0, 0>, 0.075 }
		rotate z*30
		translate y*0.4
	}
}

#declare prism_rad = 3/cos( radians(30) );  // radius of hexagonal prism with sides of length 3
// connector hexagon
#declare C_hex = prism { 0, 3,
	7,
	#declare i = 0;
	#while ( i < 6 )
		<prism_rad*sin(i*pi/3), prism_rad*cos(i*pi/3)>,
		#declare i = i + 1;
	#end
	<prism_rad*sin(i*pi/3), prism_rad*cos(i*pi/3)>
}

#declare Door_Sections = 7;  // number of sections in garage door
#declare Garage = union {
	object { C_hex }
	// body
	difference {
		union {
			box { <0, 0, -prism_rad>, <8, 3, prism_rad> }
			box { <0, 0, -3.6>, <1, 3.15, 3.6> }
			box { <3.5, 0, -3.6>, <4.5, 3.15, 3.6> }
			box { <7, 0, -3.6>, <8, 3.15, 3.6> }
		}
		box { <0, 0, -2.5>, <8.1, 2.5, 2.5> }
	}
	// markings
	box { <8, 0, -3>, <8.001, 2.5, -2.5>
		pigment {
			gradient y
			colour_map {
				[ 0.00 colour rgb <1, 1, 0> ]
				[ 0.50 colour rgb <1, 1, 0> ]
				[ 0.50 colour rgb <0, 0, 0> ]
				[ 1.00 colour rgb <0, 0, 0> ]
			}
			scale 0.3
			rotate x*45
		}
	}
	box { <8, 0, 2.5>, <8.001, 2.5, 3>
		pigment {
			gradient y
			colour_map {
				[ 0.00 colour rgb <1, 1, 0> ]
				[ 0.50 colour rgb <1, 1, 0> ]
				[ 0.50 colour rgb <0, 0, 0> ]
				[ 1.00 colour rgb <0, 0, 0> ]
			}
			scale 0.3
			rotate -x*45
		}
	}
	// ladder
	cylinder { <5.5, 0, -3.6>, <5.5, 3.2, -3.6>, 0.05 }
	difference {
		torus { 0.2, 0.05
			rotate z*90
		}
		plane { y, 0 }
		translate <5.5, 3.2, -3.4>
	}
	cylinder { <5.5, 3, -3.2>, <5.5, 3.2, -3.2>, 0.05 }
	cylinder { <6, 0, -3.6>, <6, 3.2, -3.6>, 0.05 }
	difference {
		torus { 0.2, 0.05
			rotate z*90
		}
		plane { y, 0 }
		translate <6, 3.2, -3.4>
	}
	cylinder { <6, 3, -3.2>, <6, 3.2, -3.2>, 0.05 }
	#declare i = 0.4;
	#while ( i <= 3.2 )
		cylinder { <5.5, i, -3.6>, <6, i, -3.6>, 0.05 }
		#declare i = i + 0.4;
	#end
	// satellite
	object { Satellite
		translate <6, 3, 0>
	}
	// door
	#declare i = 2.5 / (2*Door_Sections);
	#while ( i < 2.5 )
		cylinder { <0, 0, -2.5>, <0, 0, 2.5>, 1
			scale <2.5 / (4*Door_Sections), 2.5 / (1.5*Door_Sections), 1>
			translate <8 - 2.5/(3*Door_Sections), i, 0>
		}
		#declare i = i + 2.5/Door_Sections;
	#end
}

union {
	// left connector hexagon
	object { C_hex }
	// right connector hexagon
	object { C_hex
		translate x*5
	}
	// central box
	box { <0, 0, -7>, <5, 3, 7> }
	// living quarters
	object {
		Satellite
		scale 2
		rotate y*180
		translate <2.5, 10, 10.5>
	}
	difference {
		box { <-1, 0, 7>, <6, 10, 14> }
		// front rooms
		#declare i = -0.88;
		#while ( i < 6 )
			#declare j = 3.1;
			#while ( j < 10 )
				box { <i, j, 7.12>, <i + 1.6, j + 1.05, 8.72> }
				box { <i + 0.6, j + 0.3, 6.9>, <i + 1, j + 0.7, 7.2> }
				#declare j = j + 1.15;
			#end
			#declare i = i + 1.72;
		#end
		// right rooms
		#declare i = 7.12;
		#while ( i < 14 )
			#declare j = 3.1;
			#while ( j < 10 )
				box { <4.28, j, i>, <5.88, j + 1.05, i + 1.6> }
				box { <5.8, j + 0.3, i + 0.6>, <6.1, j + 0.7, i + 1> }
				#declare j = j + 1.15;
			#end
			#declare i = i + 1.72;
		#end
	}
	// front windows
	#declare i = -0.88;
	#while ( i < 6 )
		#declare j = 3.1;
		#while ( j < 10 )
			box { <i + 0.5, j + 0.3, 7>, <i + 1.1, j + 0.7, 7.12>
				material { Tinted_Glass }
			}
			#declare j = j + 1.15;
		#end
		#declare i = i + 1.72;
	#end
	// right windows
	#declare i = 7.12;
	#while ( i < 14 )
		#declare j = 3.1;
		#while ( j < 10 )
			box { <5.88, j + 0.3, i + 0.5>, <6, j + 0.7, i + 1.1>
				material { Tinted_Glass }
			}
			#declare j = j + 1.15;
		#end
		#declare i = i + 1.72;
	#end
	// hydroponics
	difference {
		union {
			cylinder { <0, 0, 0>, <0, 3, 0>, 5 }
			#declare i = 0;
			#while ( i < 360 )
				cylinder { <0, 0, 4.8>, <0, 3, 4.8>, 0.4
					rotate y*i
				}
				#declare i = i + 45;
			#end
		}
		cylinder { <0, 2, 0>, <0, 3, 0>, 4.6 }
		translate <2.5, 0, -10>
	}
	difference {
		sphere { <0, 0, 0>, 1
			scale <4.7, 2.5, 4.7>
		}
		sphere { <0, 0, 0>, 1
			scale <4.6, 2.4, 4.6>
		}
		material { Tinted_Glass }
		translate <2.5, 3, -10>
	}
	// plants
	// centre
	Stalk ( 2, 1, 0, <2.5, 2, -6.7> )
	Stalk ( 3, 1, 0, <2.5, 2, -7.8> )
	Stalk ( 3, 1, 0, <2.5, 2, -8.9> )
	Stalk ( 3, 1, 0, <2.5, 2, -10> )
	Stalk ( 3, 1, 0, <2.5, 2, -11.1> )
	Stalk ( 3, 1, 0, <2.5, 2, -12.2> )
	Stalk ( 2, 1, 0, <2.5, 2, -13.3> )
	// right once
	Stalk ( 2, 1, 0, <3.6, 2, -6.7> )
	Stalk ( 3, 1, 0, <3.6, 2, -7.8> )
	Stalk ( 3, 1, 0, <3.6, 2, -8.9> )
	Stalk ( 3, 1, 0, <3.6, 2, -10> )
	Stalk ( 3, 1, 0, <3.6, 2, -11.1> )
	Stalk ( 3, 1, 0, <3.6, 2, -12.2> )
	Stalk ( 2, 1, 0, <3.6, 2, -13.3> )
	// right twice
	Stalk ( 2, 1, 0, <4.7, 2, -7.8> )
	Stalk ( 3, 1, 0, <4.7, 2, -8.9> )
	Stalk ( 3, 1, 0, <4.7, 2, -10> )
	Stalk ( 3, 1, 0, <4.7, 2, -11.1> )
	Stalk ( 2, 1, 0, <4.7, 2, -12.2> )
	// right thrice
	Stalk ( 2, 1, 0, <5.8, 2, -8.9> )
	Stalk ( 2, 1, 0, <5.8, 2, -10> )
	Stalk ( 2, 1, 0, <5.8, 2, -11.1> )
	// left once
	Stalk ( 2, 1, 0, <1.4, 2, -6.7> )
	Stalk ( 3, 1, 0, <1.4, 2, -7.8> )
	Stalk ( 3, 1, 0, <1.4, 2, -8.9> )
	Stalk ( 3, 1, 0, <1.4, 2, -10> )
	Stalk ( 3, 1, 0, <1.4, 2, -11.1> )
	Stalk ( 3, 1, 0, <1.4, 2, -12.2> )
	Stalk ( 2, 1, 0, <1.4, 2, -13.3> )
	// left twice
	Stalk ( 2, 1, 0, <0.3, 2, -7.8> )
	Stalk ( 3, 1, 0, <0.3, 2, -8.9> )
	Stalk ( 3, 1, 0, <0.3, 2, -10> )
	Stalk ( 3, 1, 0, <0.3, 2, -11.1> )
	Stalk ( 2, 1, 0, <0.3, 2, -12.2> )
	// left thrice
	Stalk ( 2, 1, 0, <-0.8, 2, -8.9> )
	Stalk ( 2, 1, 0, <-0.8, 2, -10> )
	Stalk ( 2, 1, 0, <-0.8, 2, -11.1> )
	// connector to right garage
	// body
	difference {
		box { <8, 0, -1>, <25, 1.5, 1> }
		box { <7, 0.1, -0.8>, <26, 2, 0.8> }
	}
	// glass
	difference {
		cylinder { <8, 1.5, 0>, <25, 1.5, 0>, 1.00 }
		cylinder { <8, 1.5, 0>, <25, 1.5, 0>, 0.90 }
		material { Tinted_Glass }
	}
	// sections over glass
	#declare i = 13;
	#while ( i < 25 )
		difference {
			cylinder { <i - 0.25, 1.5, 0>, <i + 0.25, 1.5, 0>, 1.1 }
			cylinder { <i - 0.3, 1.5, 0>, <i + 0.3, 1.5, 0>, 0.8 }
		}
		box { <i - 0.25, 0, -1.1>, <i + 0.25, 1.5, -1> }
		box { <i - 0.25, 0, 1>, <i + 0.25, 1.5, 1.1> } 
		#declare i = i + 5;
	#end	
	// right garage
	object { Garage
		translate x*15 + <13, 0, 0>
	}
	// connector to near garage
	// body
	difference {
		box { <0, 0, -1>, <10.5, 1.5, 1> }
		box { <-1, 0.1, -0.8>, <11, 2, 0.8> }
		rotate y*60
		translate x*5
	}
	difference {
		box { <-9.5, 0, -1>, <0, 1.5, 1> }
		box { <-10, 0.1, -0.8>, <1, 2, 0.8> }
		rotate -y*60
		rotate y*degrees( atan2(5, 2) )
		translate x*15 + <4, 0, -10>
	}	
	// glass
	difference {
		union {
			cylinder { <0, 1.5, 0>, <10.5, 1.5, 0>, 1.00
				rotate y*60
				translate x*5
			}
			cylinder { <-9.5, 1.5, 0>, <0, 1.5, 0>, 1.00
				rotate -y*60
				rotate y*degrees( atan2(5, 2) )
				translate x*15 + <4, 0, -10>
			}
		}
		cylinder { <0, 1.5, 0>, <10.5, 1.5, 0>, 0.90
			rotate y*60
			translate x*5
		}
		cylinder { <-9.5, 1.5, 0>, <0, 1.5, 0>, 0.90
			rotate -y*60
			rotate y*degrees( atan2(5, 2) )
			translate x*15 + <4, 0, -10>
		}
		material { Tinted_Glass }
	}
	// sections over glass
	difference {
		cylinder { <4.75, 1.5, 0>, <5.25, 1.5, 0>, 1.1 }
		cylinder { <4.7, 1.5, 0>, <5.3, 1.5, 0>, 0.8 }
		rotate y*60
		translate x*5
	}
	union {
		box { <4.75, 0, -1.1>, <5.25, 1.5, -1> }
		box { <4.75, 0, 1>, <5.25, 1.5, 1.1> } 
		rotate y*60
		translate x*5
	}
	difference {
		cylinder { <9.75, 1.5, 0>, <10.25, 1.5, 0>, 1.1 }
		cylinder { <9.7, 1.5, 0>, <10.3, 1.5, 0>, 0.8 }
		rotate y*60
		translate x*5
	}
	union {
		box { <9.75, 0, -1.1>, <10.25, 1.5, -1> }
		box { <9.75, 0, 1>, <10.25, 1.5, 1.1> } 
		rotate y*60
		translate x*5
	}
	difference {
		cylinder { <-5.25, 1.5, 0>, <-4.75, 1.5, 0>, 1.1 }
		cylinder { <-5.3, 1.5, 0>, <-4.7, 1.5, 0>, 0.8 }
		rotate -y*60
		rotate y*degrees( atan2(5, 2) )
		translate x*15 + <4, 0, -10>
	}
	union {
		box { <-5.25, 0, -1.1>, <-4.75, 1.5, -1> }
		box { <-5.25, 0, 1>, <-4.75, 1.5, 1.1> } 
		rotate -y*60
		rotate y*degrees( atan2(5, 2) )
		translate x*15 + <4, 0, -10>
	}
	// near garage
	object { Garage
		rotate y*degrees( atan2(5, 2) )
		translate x*15 + <4, 0, -10>
	}
	// connector to far garage
	// body
	union {
		difference {
			box { <0, 0, -1>, <11.2, 1.5, 1> }
			box { <-1, 0.1, -0.8>, <12, 2, 0.8> }
			rotate -y*60
			translate x*5
		}
		difference {
			box { <-7, 0, -1>, <0, 1.5, 1> }
			box { <-8, 0.1, -0.8>, <1, 2, 0.8> }
			rotate y*60
			rotate -y*degrees( atan2(10, 1.5) )
			translate x*15 + <1.8, 0, 12>
		}
	}
	// glass
	difference {
		union {
			cylinder { <0, 1.5, 0>, <11.2, 1.5, 0>, 1.00
				rotate -y*60
				translate x*5
			}
			cylinder { <-7, 1.5, 0>, <0, 1.5, 0>, 1.00
				rotate y*60
				rotate -y*degrees( atan2(10, 1.5) )
				translate x*15 + <1.8, 0, 12>
			}
		}
		cylinder { <0, 1.5, 0>, <11.2, 1.5, 0>, 0.90
			rotate -y*60
			translate x*5
		}
		cylinder { <-7, 1.5, 0>, <0, 1.5, 0>, 0.90
			rotate y*60
			rotate -y*degrees( atan2(10, 1.5) )
			translate x*15 + <1.8, 0, 12>
		}
		material { Tinted_Glass }
	}
	// sections over glass
	difference {
		cylinder { <4.75, 1.5, 0>, <5.25, 1.5, 0>, 1.1 }
		cylinder { <4.7, 1.5, 0>, <5.3, 1.5, 0>, 0.8 }
		rotate -y*60
		translate x*5
	}
	union {
		box { <4.75, 0, -1.1>, <5.25, 1.5, -1> }
		box { <4.75, 0, 1>, <5.25, 1.5, 1.1> } 
		rotate -y*60
		translate x*5
	}
	difference {
		cylinder { <9.75, 1.5, 0>, <10.25, 1.5, 0>, 1.1 }
		cylinder { <9.7, 1.5, 0>, <10.3, 1.5, 0>, 0.8 }
		rotate -y*60
		translate x*5
	}
	union {
		box { <9.75, 0, -1.1>, <10.25, 1.5, -1> }
		box { <9.75, 0, 1>, <10.25, 1.5, 1.1> } 
		rotate -y*60
		translate x*5
	}
	difference {
		cylinder { <-5.25, 1.5, 0>, <-4.75, 1.5, 0>, 1.1 }
		cylinder { <-5.3, 1.5, 0>, <-4.7, 1.5, 0>, 0.8 }
		rotate y*60
		rotate -y*degrees( atan2(10, 1.5) )
		translate x*15 + <1.8, 0, 12>
	}
	union {
		box { <-5.25, 0, -1.1>, <-4.75, 1.5, -1> }
		box { <-5.25, 0, 1>, <-4.75, 1.5, 1.1> } 
		rotate y*60
		rotate -y*degrees( atan2(5, 2) )
		translate x*15 + <1.8, 0, 12>
	}
	// far garage
	object { Garage
		rotate -y*degrees( atan2(10, 1.5) )
		translate x*15 + <1.8, 0, 12>
	} 
	texture { Dull_Metal }
	translate Hub_Centre - x*15
	rotate -x*T_rotate
}
//----- End Hub -----\\

//----- Road Markers -----\\
#declare Road_Marker = union {
	// post
	cylinder { <0, 0, 0>, <0, 1, 0>, 0.1
		pigment { colour rgb <0, 0, 0> }
	}
	// reflective sections
	cylinder { <0, 0.70, 0>, <0, 0.75, 0>, 0.11
		pigment { colour rgb <1, 1, 1> }
		finish {
			specular 1
			roughness 1
		}
	}
	cylinder { <0, 0.90, 0>, <0, 0.95, 0>, 0.11
		pigment { colour rgb <1, 1, 1> }
		finish {
			specular 1
			roughness 1
		}
	}
}

// from near garage
#declare i = 20;
#while ( i < 140 )
	#declare Pos = vrotate( <i, 50, -1.75>, y*degrees( atan2(5, 2) ) - 1 ) + <4, 0, -10> + Hub_Centre;
	#declare translation = trace( Terrain, Pos, -y ) - y*0.1;  // last term is because the height field is not completely smooth
	object { Road_Marker
		translate translation
		rotate -x*T_rotate
	}
	#declare Pos = vrotate( <i, 50, 1.75>, y*degrees( atan2(5, 2) ) - 1 ) + <4, 0, -10> + Hub_Centre;
	#declare translation = trace( Terrain, Pos, -y ) - y*0.1;  // last term is because the height field is not completely smooth
	object { Road_Marker
		translate translation
		rotate -x*T_rotate
	}
	#declare i = i + 10;
#end
// off to right
#declare i = 20;
#while ( i < 140 )
	object { Road_Marker
		translate <i, 0, -1.75> + x*13 + Hub_Centre
		rotate -x*T_rotate
	}
	object { Road_Marker
		translate <i, 0, 1.75> + x*13 + Hub_Centre
		rotate -x*T_rotate
	}
	#declare i = i + 10;
#end
// from far garage
#declare i = 20;
#while ( i < 300 )
	#declare Pos = vrotate( <i, 50, -1.75>, -y*degrees( atan2(10, 1.5) ) ) + <1.8, 0, 12> + Hub_Centre;
	#declare translation = trace( Terrain, Pos, -y ) - y*0.1;  // last term is because the height field is not completely smooth
	object { Road_Marker
		translate translation
		rotate -x*T_rotate
	}
	#declare Pos = vrotate( <i, 50, 1.75>, -y*degrees( atan2(10, 1.5) ) ) + <1.8, 0, 12> + Hub_Centre;
	#declare translation = trace( Terrain, Pos, -y ) - y*0.1;  // last term is because the height field is not completely smooth
	object { Road_Marker
		translate translation
		rotate -x*T_rotate
	}
	#declare i = i + 10;
#end
//----- End Road Markers -----\\

//----- Rocks -----\\
// No Rock Zone (this is more efficient than a difference to remove rocks)
#declare NRZ = union {
	box { <20, 0, -2.5>, <140, 1, 2.5>
		rotate y*degrees( atan2(5, 2) ) - 1
		translate <4, 0, -10> + Hub_Centre
	}
	cylinder { <0, 0, 0>, <0, 1, 0>, 25
		translate Hub_Centre
	}
	box { <-500, 0, -2.5>, <500, 1, 2.5> }
}

#declare Rock_Rand = seed(33389);
#declare N_Rocks = 1500;
#declare i = 0;
#while ( i < N_Rocks )
	// generate random position
	#declare Pos = <150*rand(Rock_Rand) - 75, 50, 300*rand(Rock_Rand)>;
	// check if it would lie in the No Rock Zone
	#declare Normal = <1, 0, 0>;
	#declare translation = trace( NRZ, Pos, -y, Normal );
	// if it didn't intersect with the NRZ
	#if ( vlength( Normal ) = 0 )
		// find location on the height field and place a rock
		#declare translation = trace( Terrain, Pos, -y );
		sphere { <0, 0, 0>, 1
			pigment { colour rgb ( 0.4 + 0.4*rand(Rock_Rand) ) }
			scale <0.1 + 0.1*rand(Rock_Rand), 0.1 + 0.1*rand(Rock_Rand), 0.1 + 0.1*rand(Rock_Rand)>
			translate translation -y*0.03
			rotate -x*T_rotate
		}
		#declare i = i + 1;
	#end
#end
//----- End Rocks -----\\

//----- Mine -----\\
#declare Mine_location = trace( Terrain, <0, 50, 500>, -y );

union {
	// "legs"
	intersection {
		difference {
			sphere { <0, 0, 0>, 1
				scale <1, 2, 1>
			}
			sphere { <0, 0, 0>, 1
				scale <0.90, 1.8, 0.90>
			}
		}
		union {
			box { <-1, 0, -0.07>, <1, 2, 0.07> }
			box { <-0.07, 0, -1>, <0.07, 2, 1> }
			rotate y*30
		}
	}
	// top section
	sphere { <0, 1.8, 0>, 0.2 }
	cylinder { <0, 1.8, 0>, <0, 2.1, 0>, 0.2 }
	sphere { <0, 2.1, 0>, 0.2 }
	// communications satellite
	object { Satellite
		scale 0.15
		rotate y*( 180 - degrees( atan2(10, 1.5) ) )
		translate <0, 2.25, 0>
	}
	// tube
	difference {
		cylinder { <0, -0.1, 0>, <0, 2.1, 0>, 0.15 }
		cylinder { <0, -0.2, 0>, <0, 2.2, 0>, 0.14 }
		material { Tinted_Glass }
	}
	// sections over tube
	#declare i = 0.5;
	#while ( i < 2.1 )
		difference {
			cylinder { <0, i - 0.025, 0>, <0, i + 0.025, 0>, 0.16 }
			cylinder { <0, i - 0.03, 0>, <0, i + 0.03, 0>, 0.13 }
		}
		#declare i = i + 0.5;
	#end
	scale 11
	texture { Dull_Metal }
	translate Mine_location
	rotate -x*T_rotate
}
//----- End Mine -----\\

//----- Ships -----\\

#declare Ship_Metal = texture {
	pigment { colour rgb <0.85, 0.85, 0.85> }
	finish {
		brilliance 2
		specular 0.9
		roughness 0.01
		reflection 0.1
		metallic
	}
}
texture {
	pigment {
		bozo
		turbulence 0.3
		colour_map {
			[0.00 color rgbf<1.000, 1.000, 1.000, 1.000>]
			[0.50 color rgbf<0.632, 0.612, 0.688, 0.698>]
			[1.00 color rgbf<0.546, 0.597, 0.676, 0.953>]
		}
	}
	scale 0.3
}

#declare Engine = union {
	sphere { <0, 0, -20>, 5 }
	cylinder { <0, 0, -30>, <0, 0, -20>, 5 }
	//cone { <0, 0, -40>, 2, <0, 0, -30>, 5
	difference {
		sphere { <0, 0, 0>, 1 }
		plane { -z, 0 }
		scale <5, 5, 10>
		translate -z*30
		pigment { colour rgb <0.6, 0.6, 1.2> }
		finish { ambient 1 }
	}
} 

#declare Lander = prism { 0, 1.5
	7,
	<0, 5>, <3, 2>, <3, 0>, <7, -4>, <-7, -4>, <-3, 0>, <-3, 2>
	scale 1.5
}	

#declare Colony_Ship = union {
	// body cylinder
	cylinder { <0, 0, -20>, <0, 0, 30>, 10*(sqrt(2) - 1) }
	// engines
	object { Engine
		translate <5, 5, 0>
	}
	object { Engine
		translate <5, -5, 0>
	}
	object { Engine
		translate <-5, -5, 0>
	}
	object { Engine
		translate <-5, 5, 0>
	}
	// engine to ring
	sphere { <0, 0, -20>, 10 }
	cylinder { <0, 0, -20>, <0, 0, -10>, 10 }
	cone { <0, 0, -10>, 10, <0, 0, 0>, 6.5 }
	// ring
	cylinder { <0, 0, 0>, <0, 0, 10>, 6.5 }
	cylinder { <0, 0, 5>, <0, 19, 5>, 3
		rotate z*45
	}
	cylinder { <0, 0, 5>, <0, 19, 5>, 3
		rotate z*165
	}
	cylinder { <0, 0, 5>, <0, 19, 5>, 3
		rotate z*285
	}
	difference {
		cylinder { <0, 0, 0>, <0, 0, 10>, 22 }
		cylinder { <0, 0, -1>, <0, 0, 11>, 19 }
	}
	// ring to front
	cone { <0, 0, 10>, 6.5, <0, 0, 20>, 10 }
	cylinder { <0, 0, 20>, <0, 0, 40>, 10 }
	sphere { <0, 0, 40>, 10 }
	// landing vessels
	object { Lander
		translate <0, 12, 35>
	}
	object { Lander
		translate <0, 12, 35>
		rotate z*90
	}
	object { Lander
		translate <0, 12, 35>
		rotate z*180
	}
	object { Lander
		translate <0, 12, 35>
		rotate z*270
	}
}

// leftmost ship
object { Colony_Ship
	rotate -x*50
	rotate y*70
	translate <780, 500, 2500>
	texture { Ship_Metal }
}

// "lead" ship
object { Colony_Ship
	rotate -x*50
	rotate y*70
	translate <975, 625, 2650>
	texture { Ship_Metal }
}

// rightmost ship
object { Colony_Ship
	rotate -x*50
	rotate y*70
	translate <960, 450, 2400>
	texture { Ship_Metal }
}
//----- End Ships -----\\