// Persistence of Vision Ray Tracer Scene Description File
// File: key.inc
// Vers: 3.5
// Desc: Definition of a key (with all the stuff behind it...)
// Date: 2003/03/16
// Auth: Maurizio Tomasi
//

#ifndef (__KEY_INC__)
#declare __KEY_INC__ = 1;

#version 3.5;

#include "rand.inc"


// ----------------------------------------

/*                                   _____
 *                                  /     \                             
 *          . . . +       . . .+---< <-P-> |
 *          ^    /|       ^    |<N> \_____/
 *        Q v   | |     M |    |
 *          . . +-+       |    |
 *              |         v    |
 *    ^ . . . . |         . . .|. . . . .
 *    |        /              /|        ^
 *    |       /          L   / |        |
 *    |      |         <--->|  |        |
 *    |      |        |_____|  |        | K
 *    |      |        |     |  |        |
 *    |      |              |  |        |
 *    |      |               \ |        v
 *  H |      |            +---++. . . . .
 *    |      |            |   |         ^
 *    |      |  .         | +-+         |
 *    |      |<I>         | |           |
 *    |      |  .         | |           | J
 *    |      |            | |           |
 *    |       \ .    +----+ |           |
 *    |        \     |      |           v
 *    v         |    +--+-+-+ . . . . . .
 *    . . . . . |       | |
 *         ^ . +--------+-+-+
 *         |   |<-----F---->|
 *       G |   |____        |
 *         |        \   +-+ |^
 *         v . . . . .--| |/ | E
 *                      +-+  v
 *                      | |  ^
 *                      | |  |
 *                      | |  | D
 *                      | |  |
 *                      | |  v
 *  +----------------------+ ^
 *  |                   <C>| | B
 *  +----------------------+ v
 *  <---------------------->
 *             A
 */

#macro CreateKey (Pos, KeySeed, UseDamper, DamperAngle, HammerAngle, 
    BlackKey, Shear, KeyThickness, A, B, C, D, E, F, G, H, I, J, K, 
    L, M, N, P, Q)
union
{
    #local PusherAngle = 15;
    #local PusherHeight = 8;
    #local PusherRadius = 0.07;
    #local DamperArmRadius = 0.07;
    #local DamperNailRadius = 0.05;

    #local LightWood1Col1 = <175, 142, 96>/255;
    #local LightWood1Col2 = <125, 105, 76>/255;
    #local LightWood1Col3 = <129, 103, 64>/255;

    #local TexRand = RRand (80, 100, KeySeed);
    
    #local LightWoodTex1 =
    texture
    {
        pigment
        {
            agate
            color_map
            {
                [0.0 color rgb LightWood1Col1 * TexRand / 100.0]
                [0.6 color rgb LightWood1Col2 * TexRand / 100.0]
                [0.8 color rgb LightWood1Col3 * TexRand / 100.0]
            }
            
            scale <0.1, 1, 0.1>
        }
    }
    
    #local LightWood2Col1 = <145, 110, 62>/255;
    #local LightWood2Col2 = <164, 118, 67>/255;
    #local LightWood2Col3 = <134,  98, 61>/255;

    #local TexRand = RRand (70, 100, KeySeed);

    #local LightWoodTex2 =
    texture
    {
        pigment
        {
            agate
            color_map
            {
                [0.0 color rgb LightWood2Col1 * TexRand / 100.0]
                [0.6 color rgb LightWood2Col2 * TexRand / 100.0]
                [0.8 color rgb LightWood2Col3 * TexRand / 100.0]
            }
            
            scale <0.1, 1, 0.1>
        }
    }
    
    #local GreenTex =
    texture
    {
        pigment
        {
            color rgb <0.05, 0.3, 0.05>
        }
        
        finish { crand 0.3 }
    }
    
    #local RedTex =
    texture
    {
        pigment
        {
            color rgb <0.25, 0.02, 0.03>
        }
        
        normal
        {
            crackle 3
            scale 0.05
        }
        
    }
    
    
    #local MetalTex =
    texture
    {
        pigment
        {
            color rgb RRand (0.5, 0.8, KeySeed)
        }
        
        finish
        {
            specular RRand (0.5, 0.7, KeySeed)
            roughness 0.05
            metallic
            
            reflection { 0.6 metallic 0.4 }
        }
    }
    
    #local HammerOuterTex =
    texture
    {
        pigment { color rgb RRand (0.90, 1.10, KeySeed) }
        normal { wrinkles 0.55 scale 0.05 }
    }
    
    #local HammerInnerTex = texture { RedTex }
    
    // Base
    union
    {
        #local BaseNailTransform =
        transform
        {
            translate -11*A/12 * z
            #if (BlackKey)
                translate A/20 * z
            #end
        }
        
        // Wood support
        difference
        {
            union
            {
                box
                {
                    <-0.5, 0, C>, <0.5, B, -A>
                }
                
                box
                {
                    <-0.5, B, 0>, <0.5, B * 1.1, -A/4>
                }
                
                box
                {
                    <-0.5, B, -A/2>, <0.5, B * 1.1, -3*A/4>
                }
                
                scale <KeyThickness, 1, 1>
            }
            
            union
            {
                cylinder
                {
                    0, 2*B * y, KeyThickness/6
                }
                
                cone
                {
                    <0, B * 1.1, 0>, KeyThickness/3,
                    <0, B * 0.9, 0>, KeyThickness/6
                }

                transform { BaseNailTransform }                
            }
                
            texture { LightWoodTex1 rotate 90*x }
        }
        
        // Nail
        
        sphere_sweep
        {
            linear_spline
            
            3,
            
            <0, B*0.9, 0>, KeyThickness/6,
            <0, B*1.2, 0>, KeyThickness/6,
            <0, B*1.3, 0>, KeyThickness/20
            
            transform { BaseNailTransform }                
            
            texture { MetalTex }
        }
        
        
        // Shear transformation
        matrix <    1, 0, 0,
                    0, 1, 0,
                Shear, 0, 1,
                    0, 0, 0>
    }
    
    // Stick on the base
    object
    {
        cylinder
        {
            0, D*y, C/2 
        }

        texture { LightWoodTex2 }
        
        translate B*y
    }
    
    // Stick cap

    difference
    {    
        union
        {
            box
            {
                <-C/2, 0, -C/2>, <C/2, E, C/2>
            }
        
            cylinder
            {
                <-C/2, E, 0>, <C/2, E, 0>, C/2
            }

        }

        box
        {
            <-C/2, E/2, -C>, <C/2, 3/2*E, C> 
            scale <0.75, 1, 1>
        }
        
        scale <1.2, 1, 1.2>

        texture { LightWoodTex2 }
        
        translate (B + D) * y        
    }

    // Top structure (by far the most complicated stuff...)
    union
    {
        ////////////////////////////////////////////////////////////////////////////////
        
        // Base of the top structure (where dampers and hammers start)
        prism
        {
            linear_spline
            -C/2, C/2, 8,
            
            /*
             * +-----
             * |     \
             * |      \
             * |       \
             * |        |
             * |        |
             * |        |
             * |        |
             * |        |
             * |        |
             * |       /
             * |      /
             * |     /
             * |     |
             * |     |
             * |     |
             * |     |
             * +-----+
             *
             *
             */
            
            
            <-G/2, 0>, 
            <-G/2, F>, 
            <0, F>, 
            <G/2, 3/4*F>, 
            <G/2, F/2>, 
            <G/4, F/4>, 
            <-G/4, 0>, 
            <-G/2, 0>
            
            rotate -90*z
            scale <0.75, 1, 1>
            translate <0, G/6, -5/8*F>
            
            texture { LightWoodTex1 rotate 90*x }
        }
        
        ////////////////////////////////////////////////////////////////////////////////
        
        // Pusher
        union
        {
            union
            {
                cylinder
                {
                    0, PusherHeight*y, PusherRadius
    
                    texture { MetalTex }                
                }
                
                union
                {
                    prism
                    {
                        quadratic_spline
                        
                        -C/2, C/2, 7
                        
                        /*    ___
                         *   /   |
                         *  /    |
                         * |     |
                         * |     |
                         * |     |
                         * |     |
                         * +-----+
                         */
        
                        <C/2, 0>, <-C/2, 0>, <-C/2, 2/3*C>, <0, C>, <C/2, C>, <C/2, 0>, <-C/2, 0>
                        rotate -90*x
                        rotate -90*y
                        
                        texture { LightWoodTex1 }
                    }
                    
                    box
                    {
                        <-C/2, 0, 2/3*C>, <C/2, C, 2/3*C*1.1>
                        
                        texture { GreenTex }
                    }

                    scale <1, 2.5, 1>
                    translate PusherHeight*y
                }                

                rotate -PusherAngle * RRand (0.95, 1.05, KeySeed) * x
            }
            
            // Tongue support
            sphere_sweep
            {
                b_spline
                
                9,
                <0, -PusherHeight * 0.5, 0>, PusherRadius,
                0, PusherRadius,
                <0, PusherHeight * 0.5, 0>, PusherRadius,
                <- 7 * PusherRadius, PusherHeight * 0.6, 0>, PusherRadius,
                <0, PusherHeight * 0.7, 0>, PusherRadius,
                <7 * PusherRadius, PusherHeight * 0.6, 0>, PusherRadius,
                <0, PusherHeight * 0.6, 0>, PusherRadius,
                <0, PusherHeight * 0.65, -4 * PusherRadius>, PusherRadius * 0.1,
                <0, PusherHeight * 0.85, -6 * PusherRadius>, PusherRadius * 0.1
                
                texture { MetalTex }
                
                rotate -PusherAngle * RRand (0.95, 1.05, KeySeed) * x
                translate -0.1*F*z
            }
            
            // Prism used for the red tongue
            #local p =
            prism
            {
                cubic_spline
                
                /*
                 *       _____________
                 *      /             |
                 *     /              |
                 *    /               |
                 *   /                |
                 *  |                 |
                 *  |                 |
                 *  +-----------------+
                 */
                -2*PusherRadius, 2*PusherRadius, 8,
                <0, -PusherRadius>,
                <0, 0>, 
                <0, 2 * PusherRadius>, 
                <2 * PusherRadius, 4 * PusherRadius>, 
                <6 * PusherRadius, 6 * PusherRadius>,
                <6 * PusherRadius, 0>,
                <0, 0>,
                <0, 2 * PusherRadius>
                
                rotate -90*x
                rotate -90*y
            }
            
            // Tongue
            difference
            { 
                object { p }
                object { p scale <2, 1, 1> translate <0, -1, 1> * 0.25 * PusherRadius }

                scale 1.25
                
                translate <0, PusherHeight * 0.58, -4 * PusherRadius>
                rotate -PusherAngle * RRand (0.95, 1.05, KeySeed) * x
                translate -0.1*F*z
                
                texture { RedTex }
            }
            
            translate <0, G/2, -F/4>
        }
        
        ////////////////////////////////////////////////////////////////////////////////
        
        // Hammer's base
        
        union
        {
            cylinder
            {
                0, 2 * C * y, C/2
                
                translate -F/6 * z
                
                texture { LightWoodTex1 }
            }
            
            prism
            {             
                /*    +----+
                 *    |    |
                 *    |  +-+
                 *    |  |
                 * +--+  |
                 * |     |
                 * +-----+
                 */
                 
                linear_spline
                 
                -C/3, C/3, 9,
                 
                <0, 0>, 
                <0, C/2>, 
                <F/3 - 2*C, C/2>, 
                <F/3 - C, J>, 
                <1/3*F, J>, 
                <1/3*F, J - C>, 
                <1/3*F - C/2, J - C>, 
                <1/3*F - C, 0>,
                <0, 0>
                 
                texture { LightWoodTex2 rotate 90*x }
                 
                rotate -90*x
                rotate -90*y
                
                translate -4 * C * z
                translate 2 * C * y
            }
            
            union
            {
                torus
                {
                    C, PusherRadius
                    rotate 90*z
                    
                    translate K/12*z
                    texture { MetalTex }
                }

                /////////////////////////////////////////////////                
                // Hammer's pusher
                /////////////////////////////////////////////////                
                
                difference
                {
                    box
                    {
                        <-0.5, 0, -K/6>, <0.5, K, K/6>
                    }
    
                    union
                    {                
                        cylinder
                        {
                            <-1, 0, -K/6>, <1, 0, -K/6>, K/6
                            
                            scale <1, 2, 1>
                        }
    
                        cylinder
                        {
                            <-1, 0, -K/6>, <1, 0, -K/6>, K/6
                            
                            translate K*y
                        }
                    }
                    
                    scale <KeyThickness, 1, 1>
                    translate K/6/2 * z

                    texture { LightWoodTex2 rotate 90*x }
                }
                
                union
                {
                    cylinder
                    {
                        <0, 0, 0>, <0, 0, -L>, C/4
                        
                        texture { LightWoodTex2 rotate 90*x }
                    }
                    
                    box
                    {
                        <-C/2, -C, -C/2>, <C/2, C, C/2>
                        translate <0, 0, -L>
                        texture { LightWoodTex2 }
                    }
                    
                    box
                    {
                        <-C/2, -C, -C/2*1.1>, <C/2, C, -C/2>
                        translate <0, 0, -L>
                        texture { GreenTex }
                    }
                    translate 2*K/3 * y
                }
                
                /////////////////////////////////////////////////                
                // Hammer
                /////////////////////////////////////////////////
                
                cylinder
                {
                    0, <0, M, 0>, C/3
                    
                    translate <0, K, K/6>
                    
                    texture { LightWoodTex1 }
                }
                
                union
                {
                    difference
                    {
                        cylinder
                        {
                            0, <0, 0, N>, C/3
                            
                        }
                        
                        union
                        {
                            plane
                            {
                                -y, 0
                                rotate 5*x
                                translate N * z
                            }

                            plane
                            {
                                y, 0
                                rotate -5*x
                                translate N * z
                            }
                        }
                        
                        texture { LightWoodTex1 }
                    }
                    
                    cylinder
                    {
                        0, <0, 0, 3*C>, C/2

                        texture { LightWoodTex2 }
                    }
                    
                    // The *real* hammer!
                    #local InnerHammer =
                    cylinder
                    {
                        <-0.5, 0, 0>, <0.5, 0, 0>, P/2
                        
                        scale <KeyThickness * 0.9, 1/6, 1>
                        translate N*0.95*z
                        
                        texture { HammerInnerTex }
                    }

                    union
                    {
                        difference
                        {
                            cylinder
                            {
                                <-0.5, 0, 0>, <0.5, 0, 0>, P
                                
                                scale <KeyThickness * 0.9, 1/6, 1>
                                translate N*z
                                
                                texture { HammerOuterTex }
                            }
                            
                            object { InnerHammer scale <2, 1, 1> }
                        }
                        
                        object { InnerHammer }
                        
                        rotate HammerAngle * RRand (0.95, 1.05, KeySeed) * z
                    }
                    
                    translate (K + M) * y
                }
                
                
                
                /////////////////////////////////////////////////
                
                rotate -PusherAngle * RRand (0.95, 1.05, KeySeed) * x
                translate (J + C) * y
            }            
            
            
            translate <0, G/2, F/6>
        }
        
        ////////////////////////////////////////////////////////////////////////////////
        
        #if (UseDamper)        
            // Damper
            union
            {
                box
                {
                    <-C/3, 0, -C/3>, <C/3, G/10, C/3>
                    
                    translate G/10*y
                    texture { GreenTex }
                }
                
                // Nail
                sphere_sweep
                {
                    linear_spline
                    
                    3,
                    -G/4*y, DamperNailRadius/5,
                    -G/4*0.5*y, DamperNailRadius,
                    G/2*y, DamperNailRadius
                    
                    texture { MetalTex }
                }
                
                // Cap
                union
                {
                    cylinder
                    {
                        G/3*y, G/2*y, C/3
                        
                        texture { LightWoodTex2 }
                    }
                    
                    cylinder
                    {
                        G/3*0.9*y, G/3*y, C/3*1.05
                        
                        texture { GreenTex }
                    }
                }
                
                // Arm
                sphere_sweep
                {
                    linear_spline
                    
                    6,
                    <0,       0, 0>, DamperArmRadius,
                    <0,  1/25*H, 0>, DamperArmRadius,
                    <0,  2/25*H, -I>, DamperArmRadius,
                    <0, 20/25*H, -I>, DamperArmRadius,
                    <0, 22/25*H, 0>, DamperArmRadius,
                    <0,       H, 0>, DamperArmRadius
                    
                    translate G/2*y  
                    
                    texture { MetalTex }
                }
                
                union
                {
                    // Top
                    difference
                    {
                        box { <-0.5, 0, 0>, <0.5, Q, Q/2> }
                        
                        union
                        {
                            cylinder { <-1, 0, 0>, <1, 0, 0>, Q/4 translate Q*y }
                            cylinder { <-1, 0, 0>, <1, 0, 0>, Q/4 scale <1, 2, 1> }
                        }
                        
                        scale <KeyThickness, 1, 1>
                        texture { LightWoodTex1 }
                    }
                    
                    // Screw
                    difference
                    {
                        cylinder
                        {
                            <0, 0, -Q/20>, <0, 0, 0>, C/3
                        }
                        
                        box
                        {
                            <-C/2, -Q/60, -Q/10>, <C/2, Q/60, -Q/30>
                            rotate RRand (-45, 45, KeySeed) * z
                        }
                        
                        translate 2*Q/3 * RRand (0.98, 1.02, KeySeed) * y
                        
                        texture { MetalTex }
                    }

                    // Horizontal bar going against the string(s)
                    
                    union
                    {
                        box
                        {
                            <-0.5, Q/2, Q/2>, <0.5, 2*Q/3, F - Q/2>
                        }

                        box
                        {
                            <-0.5, 2*Q/3, 3*Q/4>, <0.5, Q, F - Q/2>
                        }
                        
                        cylinder
                        {
                            <-0.5, 2*Q/3, 0>, <0.5, 2*Q/3, 0>, Q/6
                            
                            translate 3*Q/4 * z
                        }
                        
                        scale <KeyThickness, 1, 1>
                        texture { LightWoodTex2 }
                    }
                    
                    translate <0, G/2 + H, -Q/3>
                }          
                
                rotate DamperAngle*z
                translate <0, G/2, -3/5*F>
            }
        #end // #if (UseDamper)
                
        translate (B + D + E) * y
    }
    
    translate Pos
    
}
#end


#ifndef (INCLUDED_FLAG)
    global_settings {
      assumed_gamma 1.0
    }
    
    // ----------------------------------------
    
    camera {
      location  <0.0, 0.5, -4.0>
      direction 1.5*z
      right     x*image_width/image_height
      look_at   <0.0, 0.0,  0.0>
    }
    
    light_source {
      <0, 0, 0>            // light's position (translated below)
      color rgb <1, 1, 1>  // light's color
      translate <-30, 30, -30>
    }

    #local Seed = seed (16742);
    
    #declare I = 1;
    
    #while (I < 30)
        object
        {
            CreateKey (<0, -18, 0>, Seed, true, 0, mod (I, 2), 0.0, 0.95
                25, 1.5, 0.5,  // A, B, C
                13, 2, 7,      // D, E, F
                2, 25, 2.5,    // G, H, I
                6, 5, 3.5,     // J, K, L
                10, 6, 2,      // M, N, P
                2)             // Q
        
            translate -10*y
            rotate 30*y    
            scale 0.05
            
            translate (I - 15)*0.08*x
        }
        #local I = I + 1;
    #end
#end // INCLUDED_FLAG

#end // __KEY_INC__
