// Persistence of Vision Ray Tracer Scene Description File
// File: keyboard.inc
// Vers: 3.5
// Desc: Definition of a key (with all the stuff behind it...)
// Date: 2003/03/26
// Auth: Maurizio Tomasi
//

#ifndef (__KEYBOARD_INC__)
#declare __KEYBOARD_INC__ = 1;

#version 3.5;

#include "rand.inc"
#include "shapes.inc"

#ifndef (USE_ROUNDED_CORNERS)
    #declare USE_ROUNDED_CORNERS = false;
#end

#include "new-wood.inc"

/////////////////////////////////////////////////////////////////////////////////////////////

#declare WhiteKeyTex =
texture
{
    pigment
    {
        bozo
        color_map
        {
            [0.20 color rgb <207, 206, 165>/256]
            [0.35 color rgb <209, 174, 86>/256]
            [0.40 color rgb <209, 174, 86>/256]
            [0.55 color rgb <214, 185, 100>/256]
            [0.60 color rgb <214, 185, 100>/256]
            [0.85 color rgb <218, 194, 115>/256]
            [0.90 color rgb <218, 194, 115>/256]
        }
        
        warp { turbulence 0.4 }
        scale <0.4, 0.4, 1.0> * 3
    }
}

#declare BlackKeyTex =
texture
{
    pigment { color rgb 0.03 }
    
    finish
    {
        specular 0.4 roughness 0.03
    }
    
    normal
    {
        wood 0.3
        scale 0.05
    }
}
                   


/////////////////////////////////////////////////////////////////////////////////////////////

/*
 * Up view:
 *
 *      | 0
 * +----+----+------------------------------------------------------+----+----+ ^
 * |    |    |                                                      |    |    | |
 * |    |    |                                                      |    |    | |
 * |    |    |                                                      |    |    | |
 * |    |    |                                                      |    |    | | D
 * |    |    |                                                      |    |    | |
 * |    |    |                                                      |    |    | |
 * |    |    |                                                      |    |    | |
 * +----+----+------------------------------------------------------+----+----+ v
 * |    |    |                                                      |    |    | ^
 * |    |    |                                                      |    |    | | I
 * +----+----+------------------------------------------------------+----+----+ v
 *  <--> <--> <----------------------------------------------------> <--> <-->
 *   A    B                             C                             B    A
 *
 * Front view:
 *
 * +----+                                                                +----+ . . . .
 * |    |                                                                |    |       ^
 * |    |                                                                |    |       | E
 * |    |                                                                |    |       v
 * |    +----+------------------------------------------------------+----+ . .| . . . .
 * |    |    |                                                      |    |    |       ^
 * |    |    |                                                      |    |    |       | F
 * |    |    |                                                      |    |    |       v
 * +----+----+------------------------------------------------------+----+----+ . . . .
 * |    |                              O                                 |    |       ^
 * |    |                              ^                                 |    |       | G
 * |    |                              V                                 |    |       v
 * +----+----------------------------------------------------------------+----+ . . . .
 * |    |                                                                |    |       | H
 * +----+----------------------------------------------------------------+----+ . . . .
 *
 */

// This array is expressed in relative units
#declare BlackKeyXPos = array[5] { 0.112, 0.373, 0.522, 0.795, 0.956 };

#macro CreateBlackKeys (KeyboardSeed, WidthFactor, DepthFactor, StartY, Height)
    union
    {
        #local BlackKeyCount = 0;
        #while (BlackKeyCount < 35)
            union
            {
                #if (USE_ROUNDED_CORNERS)
                    Round_Box_Union (
                        <-BlackKeyWidth/2, -Height, BlackKeyDepth * DepthFactor>,
                        <BlackKeyWidth/2, Height, -BlackKeyDepth * WidthFactor>
                        BlackKeyHeight/10
                    )
    
                    object
                    {                
                        Round_Box_Union (
                            <-BlackKeyWidth/2, -2 * Height, 0>,
                            <BlackKeyWidth/2, 0, BlackKeyDepth * 0.1 * WidthFactor>
                            BlackKeyHeight/10
                        )
                        
                        rotate 30*x
                        translate <0, Height, -BlackKeyDepth * WidthFactor>
                    }
                #else
                    box {
                        <-BlackKeyWidth/2, -Height, BlackKeyDepth * DepthFactor>,
                        <BlackKeyWidth/2, Height, -BlackKeyDepth * WidthFactor>
                    }
                    
                    box
                    {                
                        <-BlackKeyWidth/2, -2 * Height, 0>,
                        <BlackKeyWidth/2, 0, BlackKeyDepth * 0.1 * WidthFactor>
                        
                        rotate 30*x
                        translate <0, Height, -BlackKeyDepth * WidthFactor>
                    }
                #end                
    
                texture { BlackKeyTex translate VRand (KeyboardSeed) }
                
                scale <WidthFactor, 1, 1>
                translate <OctaveWidth*div (BlackKeyCount, 5) 
                    + OctaveWidth * BlackKeyXPos[mod (BlackKeyCount, 5)] 
                    + BlackKeyWidth/2,
                    StartY, 0>
            }
            
            #declare BlackKeyCount = BlackKeyCount + 1;
        #end
    }
#end


/* This works like a sphere_sweep, but it can be deformed more freely */
#macro BorderSpline (ScaleFactor, XPos, Radius)
    union
    {
        cylinder
        {
            <0, 0, 0>, <0, 0, -BorderDepth>, Radius
            
            scale <1, ScaleFactor, 1>
            translate <XPos, E + F, 0>
        }
        
        cylinder
        {
            <0, E + F, 0>, <0, -(G + H), 0>, Radius
            
            scale <1, 1, ScaleFactor>
            translate <XPos, 0, -BorderDepth>
        }
        
        sphere
        {
            0, Radius
            
            scale <1, ScaleFactor, ScaleFactor>
            translate <XPos, E + F, -BorderDepth>
        }

        sphere
        {
            0, Radius
            
            scale <1, ScaleFactor, ScaleFactor>
            translate <XPos, -(G + H), -BorderDepth>
        }
    }
#end

#macro Keyboard (Stream, A, B, C, D, E, F, G, H, I)

    #local BorderDepth = (D + I) * 1.05;
    #local KeyboardBorder =
    difference
    {
        union
        {
            box
            {
                //Round_Box_Union (<0, - (G + H), 0>, <-A, E + F, -(D + I) * 1.05>, G/20)
                <0, -(G + H), 0>, <-A, E + F, -BorderDepth>
            }

            BorderSpline (0.25, -A/2, A/8)

            BorderSpline (0.25, -A/12, A/12)
            BorderSpline (0.25, -A + A/12, A/12)
        }
        
        union
        {
            BorderSpline (0.5, -A/4, A/12)
            BorderSpline (0.5, -A + A/4, A/12)
        }
        
        texture { WoodTex rotate 90*z }
    }
    
    
    #local NumOfWhiteKeys = 50;

    #local WhiteKeyWidth = C / NumOfWhiteKeys;
    #local WhiteKeyHeight = F;
    #local WhiteKeyDepth = D * 0.99;
    
    #local BlackKeyWidth = WhiteKeyWidth / 2;
    #local BlackKeyHeight = WhiteKeyHeight/2;
    #local BlackKeyDepth = 2/3 * WhiteKeyDepth;
    
    #local OctaveWidth = WhiteKeyWidth * 7;
    
    union
    {
        // White keys
        difference
        {
            union
            {
                #local WhiteKeyCount = 0;
                #while (WhiteKeyCount < NumOfWhiteKeys)
                    union
                    {
                        box
                        {
                            <0, -WhiteKeyHeight, 0>, <WhiteKeyWidth, WhiteKeyHeight, -WhiteKeyDepth>
                        }
    
                        cylinder
                        {
                            <0, 0, -WhiteKeyDepth>, <WhiteKeyWidth, 0, -WhiteKeyDepth>,
                                WhiteKeyHeight/18
                            
                            translate WhiteKeyHeight*17/18*y
                        }
                        
                        texture { WhiteKeyTex translate VRand (Stream)*WhiteKeyWidth }
                        
                        scale <0.96, 1, 1>
                        translate WhiteKeyCount * WhiteKeyWidth * x
                    }
                    
                    #declare WhiteKeyCount = WhiteKeyCount + 1;
                #end
            }
            
            object
            {
                CreateBlackKeys (Stream, 1.0, 0.01, -WhiteKeyHeight * 0.01, WhiteKeyHeight * 1.1)
            }
            
            cutaway_textures        
        }
        
        // Black keys
        object
        {
            CreateBlackKeys (Stream, 0.9, 0, WhiteKeyHeight, BlackKeyHeight)
        }
    
        // Border
        union
        {
            object
            {
                Round_Box_Union (<0, -G, -WhiteKeyDepth*0.02>, 
                    <-B, WhiteKeyHeight * 1.05, -WhiteKeyDepth>, G/20)
            }
    
            object
            {
                Round_Box_Union (<0, -G, -WhiteKeyDepth*0.02>, 
                    <B, WhiteKeyHeight * 1.05, -WhiteKeyDepth>, G/20)
                translate C * x
            }
            
            texture { WoodTex translate VRand(Stream) }
        }
        
        union
        {
            box
            {
                <0, -0.5, -0.5>, <1, 0.5, 0.5>
                scale <1, WhiteKeyHeight, WhiteKeyHeight>
                rotate 45*x
                scale <1, 0.75, 1>
                translate <0, WhiteKeyHeight + BlackKeyHeight, WhiteKeyHeight>

                texture { WoodTex translate VRand(Stream) * C }
            }
            
            box
            {
                <0, 0, 0>, <1, WhiteKeyHeight + BlackKeyHeight, WhiteKeyHeight>
                
                texture { WoodTex translate VRand(Stream) * C }
            }
            
            scale <C + 2*B, 1, 1>
            translate -B*x
        }
        
        object
        {
            KeyboardBorder
            
            translate -A * x
        }
        
        object
        {
            KeyboardBorder
            
            translate (2*B + C) * x
        }
        
        // Base
        
        union
        {
            // Panel with the lock
            #declare LockBody =
            box
            {
                <-0.5, -1, -0.5>, <0.5, 1, 0.5>
                
                scale <4*I, G/15, 2/3*I>
            }
            
            union
            {
                difference
                {
                    object
                    {
                        Round_Box_Union (<-B, 0, -D>, <C + B, -G, -(D + I)>, G/20)
                        
                        texture { WoodTex translate VRand(Stream) * C rotate 90*z }
                    }
                    
                    union
                    {
                        // Lock body
                        object 
                        { 
                            LockBody

                            translate <C/2, 0, -(D + I/2)>
                        }
                        
                        // Keyhole
                        
                        union
                        {
                            cylinder
                            {
                                <0, I/5, I>, <0, I/5, -(D + I)*2>, I/5
                            }
                            
                            box
                            {
                                <-I/10, 0, I>, <I/10, -I/2, -(D + I)*2>
                            }
                            
                            translate <C/2, -G/2, 0>
                        }
                    }                    
                }
                
                
                // Screw
                
                #local Screw =
                cylinder
                {
                    <0, 0, 0>, <0, G/15, 0>, I/4
                    translate <C/2, 0, -(D + I/2)>
                }

                #local Screws =
                union
                {
                    object { Screw translate -1.5*I * x }
                    object { Screw translate +1.5*I * x }
                }
                
                #local Locker =
                box
                {
                    <-0.5, 0, -0.5>, <0.5, 1, 0.5>
                    
                    scale <2*I, G/15, 1/3*I>
                    translate <C/2, 0, -(D + I/2)>
                }
                
                // Lock
                
                union
                {
                    difference
                    {
                        object 
                        { 
                            LockBody
                            translate <C/2, 0, -(D + I/2)>
                        }
                        
                        union
                        {
                            object { Screws scale <1, 10, 1> }
                            object { Locker scale <1, 10, 1> }
                        }
                        
                        texture
                        {
                            pigment
                            {
                                color rgb <0.6, 0.6, 0.05>
                            }
                            
                            normal
                            {
                                bozo 0.8
                                scale 0.5
                            }
                            
                            finish
                            {
                                specular 0.1 roughness 0.03 metallic
                            }
                        }
                    }
    
                    object
                    {
                        Locker
                        
                        texture
                        {
                            pigment
                            {
                                color rgb <0.6, 0.6, 0.05> * 0.6
                            }
                            
                            normal
                            {
                                bozo 1.3
                                scale 0.5
                            }
                            
                            finish
                            {
                                specular 0.1 roughness 0.03 metallic
                            }
                        }
                    }
                    
                    object 
                    { 
                        Screws
                        scale <1, 0.3, 1>
                        translate G/15*y
    
                        texture
                        {
                            pigment
                            {
                                color rgb <0.6, 0.6, 0.05> * 0.6
                            }
                            
                            finish
                            {
                                specular 0.1 roughness 0.04 metallic
                            }
                        }
                    }
                    
                    translate -G/10*y
                }
            }
            
            // Decorated panel (under the previous one)
            union
            {
                box
                {
                    <-B, 0, 0>, <2*B + C, H, -(D + I)>
                }
                
                cylinder
                {
                    <-B, 0, 0>, <2*B + C, 0, 0>, H/10
                    translate <0, H/10, -(D + I)>
                }
                
                cylinder
                {
                    <-B, 0, 0>, <2*B + C, 0, 0>, H/10
                    translate <0, 9*H/10, -(D + I)>
                }

                cylinder
                {
                    <-B, 0, 0>, <2*B + C, 0, 0>, 4*H/10
                    
                    scale <1, 1, 0.4>
                    translate <0, H/2, -(D + I)>
                }

                translate -(G + H) * y
                texture { WoodTex translate VRand(Stream) * C rotate 90*z }
            }
        }
        

        translate B*x
    }
#end


#ifndef (INCLUDED_FLAG)
    global_settings {
      assumed_gamma 1.0
    }
    
    // ----------------------------------------
    
    camera {
      location  <0.0, 0.5, -4.0>
      direction 1.5*z
      right     x*image_width/image_height
      look_at   <0.0, 0.0,  0.0>
    }
    
    light_source {
      <0, 0, 0>            // light's position (translated below)
      color rgb <1, 1, 1>  // light's color
      translate <-30, 30, -30>
    }

    object
    {
        Keyboard (seed (176453), 6.5, 6.5, D - 6.5*2, 15.0, 5, 2, 4, 5, 2.5)
    
        rotate -30*x
        translate <-WhiteKeyWidth * 10, -WhiteKeyHeight * 5, 0>
        scale 0.06
    }
#end // INCLUDED_FLAG

#end // __KEYBOARD_INC__
