// POV-Ray 3.5 Scene File
// File: mtpiano.pov
// Vers: 1.0
// Desc: The Piano
// Date: 2004/03/14
// Auth: Maurizio Tomasi
//

#declare IMPORT_SPATCH = 1;
#declare CAMERA_NUM = 1;
#declare LIGHT_MODEL_NUM = 2;

#include "rand.inc"
#include "functions.inc"
#include "arrays.inc"

#declare USE_KEYBOARD = true;
#declare USE_INNER_MECHANICS = true;
#declare USE_REAR_PANEL = true;
#declare USE_NAILS = true;
#declare USE_RADIOSITY = true;
#declare USE_ROUNDED_CORNERS = false;

#declare INCLUDED_FLAG = true;

#include "keyboard.inc"
#include "key.inc"
#include "rear.inc"
#include "new-wood.inc"

#local OrthoMode = off;

#switch (CAMERA_NUM)

    #case(1) // Default view
        #declare Location = <-70.0, 180, -200.0>;
        #declare Direction = 2*z;
        #declare Right = x*image_width/image_height;
        #declare LookAt = <-5.0, 80, 0.0>;
    #break
    
    #case(2) // Right view
        #declare Location = <90.0, 180, -200.0>;
        #declare Direction = 2*z;
        #declare Right = x*image_width/image_height;
        #declare LookAt = <5.0, 80, 0.0>;
    #break

    #case(3) // Up view
        #declare Location = <-10.0, 280, -200.0>;
        #declare Direction = 2*z;
        #declare Right = x*image_width/image_height;
        #declare LookAt = <5.0, 80, 0.0>;
    #break

    #case(4) // The same view as photo #5
        #declare Location = <-40.0, 170, -200.0>;
        #declare Direction = 2*z;
        #declare Right = x*image_width/image_height;
        #declare LookAt = <-5.0, 80, 0.0>;
    #break

    #case(5) // Front view
        #declare Location = <0.0, 170, -200.0>;
        #declare Direction = 2*z;
        #declare Right = 120*x*image_width/image_height;
        #declare Up = 120*y;
        #declare LookAt = <0.0, 80, 0.0>;
        
        #declare OrthoMode = on;
    #break

#end

camera 
{
    #if (OrthoMode)
        orthographic
        up Up
    #end

    location  Location
    direction Direction
    right     Right
    look_at   LookAt
}


////////////////////////////////////////////////////////////////////////


// Wall
plane
{
    z, 40
    
    hollow
    
    texture
    {
        pigment { color rgb 0.2 }
        normal
        {
            bumps 0.3
        }
        
        scale 0.3
        
        finish { specular 0.1 roughness 0.0001 }
    }
}

// Wall

#if (USE_RADIOSITY)
    light_group
    {
        plane
        {
            z, -500
            
            hollow
            
            texture
            {
                pigment 
                { 
                    image_map
                    {
                        jpeg "incubus.jpg"
                        interpolate 2
                    }
                    
                    translate <-0.5, 0, 0>
                    scale 150
                }
            }
        }
        
        light_source
        {
            <0, 0, 0>, color rgb 3
            shadowless
        }
    }
#end

// Ground
plane
{
    y, 0
    
    hollow
    
    texture
    {
        pigment { color rgb 0.2 }
        finish { specular 0.1 roughness 0.0001 }
    }
}


////////////////////////////////////////////////////////////////////////

/* +-+------------------+-+ ^            +-------+
 * | |                  | | |            |       |
 * | |                  | | | A          |       |
 * | |                  | | |            |       |
 * +-+------------------+-+ + . . . . . .| . . . +----+
 * | |                  | | | B          |       |    |
 * +-+------------------+-+ + . . . . . .| . . . +----+
 * | |                  | | |            |       |
 * | |                  | | | C          |       |<--->
 * | |                  | | |            |       |  F
 * +-+------------------+-+ |            +-------+
 * | |                  | ||+- WheelSize          0
 * +-+                  +-+ v
 * <-><----------------><->              <------->
 *  E         D          E                   G
 */

#local A = 55;
#local B = 16;
#local C = 57;
#local D = 140;
#local E = 6.5;
#local F = 31;
#local G = 33;
#local WheelSize = 5;
#local ColumnRadius = 2;
 
// Piano
union
{
    // Side panels
    difference
    {
        box
        {
            <-D/2 - E, WheelSize, 0>, <D/2 + E, A + B + C, G>

            texture { WoodTex }
        }

        box
        {
            <-D/2 - E/2, C, -1>, <D/2 + E/2, 2 * (A + B + C), 2 * G>

            texture { LightWoodTex rotate 88*x }
        }
    }    

    box
    {
        <-D/2 - E/2, C, 2*G/5>, <D/2 + E/2, C + B/5, 3*G/5>
        
        texture { WoodTex }
    }
    
    box
    {
        <-D/2 + 10, C + B/5, 2*G/5>, <D/2 - 10, C + B/5 + 0.1, 3*G/5>

        texture
        {        
            pigment
            {
                color rgb <0.05, 0.3, 0.05>
            }
            
            finish { crand 0.3 }
        }
    }

    // Rear panel
    box
    {
        <-D/2 - E/2, C, G>, <D/2 + E/2, A + B + C, G + 1>

        texture { WoodTex }
    }
    
    /*
    #declare Column =
    union
    {
                
        sor
        {
            6,
            
            <ColumnRadius, WheelSize>,
            <ColumnRadius, 2 * WheelSize>,
            <ColumnRadius, 3 * WheelSize>,
            <2.5 * ColumnRadius, C - 3 * WheelSize>,
            <2 * ColumnRadius, C - 2 * WheelSize>,
            <2 * ColumnRadius, C>
        }
        
        box
        {
            <-2 * ColumnRadius,     WheelSize, -2 * ColumnRadius>,
            < 2 * ColumnRadius, 2 * WheelSize,  2 * ColumnRadius + F/2>
        }
        
        box
        {
            <-2 * ColumnRadius, C - 2 * WheelSize, -2 * ColumnRadius>,
            < 2 * ColumnRadius, C -     WheelSize,  2 * ColumnRadius>
        }
        
        texture { WoodTex }
    }
    */
    #include "column.inc"
    #include "deco_box.inc"
    
    #local Column =
    union
    {
        object
        {
            BaseColumn
            
            scale <1/0.45, 1/2.35, 1/0.45>
            scale <2.5 * ColumnRadius, C - 4 * WheelSize, 2.5 * ColumnRadius>
            translate 2 * WheelSize * y
        }
        
        Deco_Box (<-2.2 * ColumnRadius,     WheelSize, -2.2 * ColumnRadius>,
                  < 2.2 * ColumnRadius, 2 * WheelSize,  2.2 * ColumnRadius + F/2>,
                  WheelSize/8)
        
        Deco_Box (<-2.2 * ColumnRadius, C - 2 * WheelSize, -2.2 * ColumnRadius>,
                  < 2.2 * ColumnRadius, C -     WheelSize,  2.2 * ColumnRadius>,
                  WheelSize/8)

        texture { WoodTex }
    }
    
    
    object
    {
        Column 
        
        translate <-D/2, 0, -F/2>
    }

    object
    {
        Column 
        translate <D/2, 0, -F/2>

        texture { WoodTex }
    }

    // Wheel
    cylinder
    {
        <D/2 - 2/3*WheelSize, WheelSize/2, WheelSize/2>,
        <D/2,                 WheelSize/2, WheelSize/2>, WheelSize/2
        
        translate -F/2 * z
        pigment { color rgb 0.1 }
    }

    cylinder
    {
        <-D/2 + 2/3*WheelSize, WheelSize/2, WheelSize/2>,
        <-D/2,                 WheelSize/2, WheelSize/2>, WheelSize/2
        
        translate -F/2 * z
        pigment { color rgb 0.1 }
    }
}


/////////////////////////////////////////////////////////////////////////////////////////////
// Rear panel

#if (USE_REAR_PANEL)
    #local PanelSeed = seed (4233);
    object
    {
        CreateRearPanel (PanelSeed, A, B, C, D + E, E/2, F, G)
    }
#end

/////////////////////////////////////////////////////////////////////////////////////////////
// Keyboard

#if (USE_KEYBOARD)

    object
    {
        Keyboard (seed (176453), 6.5, 6.5, D - 6.5*2, 15.0, 5, 2, 4, 5, 2.5)
        translate <-D/2, C + 5, -F/4>
    }

#end

/////////////////////////////////////////////////////////////////////////////////////////////
// Inner mechanics


#local Seed = seed (16742);

/* Return nonzero if KEY is a black key.  KEY follows the numbering
 * shown below. */
#local IsBlack = function(Key) 
{
    /* +--+-+-+-+--+--+-+-+-+-+-+--+
     * |  | | | |  |  | | | | | |  |
     * |  | | | |  |  | | | | | |  |
     * |  | | | |  |  | | | | | |  |
     * |  | | | |  |  | | | | | |  |
     * |  | | | |  |  | | | | |1|  |
     * |  |1| |3|  |  |6| |8| |0|  |
     * |  +-+ +-+  |  +-+ +-+ +-+  |
     * |   |   |   |   |   |   |   |
     * |   |   |   |   |   |   |   |
     * |   |   |   |   |   |   |   |
     * | 0 | 2 | 4 | 5 | 7 | 9 | 11|
     * +---+---+---+---+---+---+---+
     */
    (mod (Key, 12) = 1.0) | 
        (mod (Key, 12) = 3.0) | 
        (mod (Key, 12) = 6.0) | 
        (mod (Key, 12) = 8.0) | 
        (mod (Key, 12) = 10.0)
}

#if (USE_INNER_MECHANICS)

    union
    {
        #local KeyThickness = 1.40;
        #local KeyA = 19; // 26
        #local KeyB = 1.5;
        #local KeyC = 0.5;
        #local KeyD = 13;
        #local KeyE = 2;
        #local KeyF = 7;
        #local KeyG = 2;
        #local KeyH = 27;
        #local KeyI = 2.5;
        #local KeyJ = 6;
        #local KeyK = 5;
        #local KeyL = 3.5;
        #local KeyM = 10;
        #local KeyN = 6;
        #local KeyP = 2;
        #local KeyQ = 2;

        #local Count = 0;

        #local LeftXPos = -(D - 12) / 2;
        #local LeftShear = 0.05; // 0.1

        #local ScrewRadius = KeyB/3;
        #local ScrewBase =
        cylinder
        {
            <0, 0, -1>, <0, 0, 1>, ScrewRadius
        }

        #macro CreateScrew (ZPos, Angle)
        union
        {
            object
            {
                ScrewBase
                scale <1, 1, ZPos>
            }

            difference
            {
                sphere
                {
                    0, ScrewRadius
                }

                union
                {
                    box
                    {
                        <-1, -1, -1>, <1, 1, 0>
                        scale <0.1, 1, 1> * ScrewRadius
                    }

                    box
                    {
                        <-1, -1, -1>, <1, 1, 0>
                        scale <1, 0.1, 1> * ScrewRadius
                    }
                }

                scale <1, 1, 0.5>
                rotate Angle*z
                translate -ZPos * z
            }

            texture
            {
                pigment
                {
                    color rgb 0.8
                }

                finish
                {
                    specular 0.9
                    roughness 0.05
                    metallic

                    reflection { 0.6 metallic 0.4 }
                }
            }
        }
        #end

        /* Horizontal bars running from one side to another */

        union
        {
            #local HorBarLen = LeftXPos + 71*1.5 + D/2;
            difference
            {
                object
                {
                    Round_Box_Union (<0, 0, 0>, <HorBarLen, KeyB, KeyN + KeyP>, KeyB/8)
                }

                union
                {
                    object 
                    { 
                        ScrewBase
                        translate HorBarLen * 0.25 * x
                    }

                    object 
                    { 
                        ScrewBase 
                        translate HorBarLen * 0.5 * x
                    }

                    object 
                    { 
                        ScrewBase 
                        translate HorBarLen * 0.75 * x
                    }

                    translate -z
                    scale <1, 1, 2 * (KeyN + KeyP)> 
                    translate KeyB/2 * y
                }

                translate <-D/2, KeyQ + 2, 0>
            }

            union
            {
                object 
                { 
                    CreateScrew ((KeyN + KeyP) / 4, 10)
                    translate HorBarLen * 0.25 * x
                }

                object 
                { 
                    CreateScrew ((KeyN + KeyP) / 4, 15)
                    translate HorBarLen * 0.5 * x
                }

                object 
                { 
                    CreateScrew ((KeyN + KeyP) / 4, 20)
                    translate HorBarLen * 0.75 * x
                }

                translate <-D/2, KeyQ + 2 + KeyB/2, (KeyN + KeyP)/4>
            }

            object
            {
                Round_Box_Union (
                    <LeftXPos + 71*1.5, 0, 0>, 
                    <LeftXPos + 75*1.5, KeyQ + 2 + KeyB, KeyN + KeyP>, KeyB/8)
            }

            object
            {
                Round_Box_Union (
                    <LeftXPos + 71*1.5, 0, 0>, <D/2, KeyB, KeyN + KeyP>, KeyB/8)
            }

            translate (KeyB + KeyD + KeyG + KeyH) * y

            texture { WoodTex }
        }

        #local BarScaleFactor = 0.67;
        union
        {
            difference
            {
                cylinder
                {
                    <-D/2, 0, 0>, <D/2, 0, 0>, KeyB

                    scale <1, 1.5, BarScaleFactor>

                    // Highly reflective metal
                    texture
                    {
                        pigment
                        {
                            color rgb <0.6, 0.4, 0.1> * 1.3
                        }

                        finish
                        {
                            specular 1.5
                            roughness 0.005
                            metallic
                            reflection { 0.5 metallic }
                        }
                    }
                }

                union
                {
                    object
                    {
                        ScrewBase
                        translate (-D/2 + ScrewRadius*4)*x
                    }

                    object
                    {
                        ScrewBase
                        translate (D/2 - ScrewRadius*4)*x
                    }

                    scale <1, 1, BarScaleFactor * KeyB * 2>
                }
            }

            object 
            { 
                CreateScrew (BarScaleFactor * KeyB * 0.85, 20) 
                translate (-D/2 + ScrewRadius*4)*x 
            }

            object 
            { 
                CreateScrew (BarScaleFactor * KeyB * 0.85, -10) 
                translate (D/2 - ScrewRadius*4)*x 
            }

            translate (KeyB + KeyD + KeyG + KeyH*0.49) * y
        }

        /* Vertical side bars */

        union
        {
            object
            {
                Round_Box_Union (
                    <-D/2, -B, 0>, 
                    <-D/2 + E/2, KeyB + KeyD + KeyG + KeyH + KeyQ + 1.9, KeyN + KeyP>, KeyB/8)
            }

            object
            {
                Round_Box_Union (
                    <D/2, -B, 0>, 
                    <D/2 - E/2, KeyB + KeyD + KeyG + KeyH, KeyN + KeyP>, KeyB/8)
            }

            texture
            {
                WoodTex
            }
        }

        // Side blocks        
        union
        {
            #local SideBlock =
            union
            {
                object
                {
                    Round_Box_Union (
                        <-D/2 * 1.05, 0, 0>, <(-D/2 - E/2) * 0.95, 5 * KeyB, KeyB>, KeyB/8)
                        
                    texture { LightWoodTex scale 0.1 }
                }
                
                prism
                {
                    cubic_spline
                    0, 0.2, 7
                    
                    <-1, -3>, <-1, 0>, <-0.4, 0.5>, <1, 0>, <-0.4, -0.5>, <-1, 0>, <-1, 3>
                    
                    scale 1.5 * KeyB
                    rotate -90*x
                    translate <-D/2, 2.5 * KeyB, 0>
                    
                    texture { LightWoodTex scale 0.1 }
                }
                
                object
                {
                    CreateScrew (0.7 * KeyB, 0)
                    scale 0.5
                    translate <-D/2 - 2*KeyB/3, 2.5 * KeyB, 0>
                }
            }
            
            object { SideBlock }
            object { SideBlock scale <-1, 1, 1> }
            
            translate 3/4 * (KeyB + KeyD + KeyG + KeyH) * y
            
        }

        // Keys
        #while (Count < 26)
            object
            {
                CreateKey (<LeftXPos, 0, 0>, Seed, true, 5, 0, 
                    IsBlack (Count + 9), -LeftShear, KeyThickness,
                    KeyA, KeyB, KeyC, KeyD, KeyE, KeyF, KeyG, KeyH, KeyI, KeyJ, KeyK,
                    KeyL, KeyM, KeyN, KeyP, KeyQ)
            
                translate Count*1.5*x
            }
            #local Count = Count + 1;
        #end

        #while (Count < 54)
            object
            {
                CreateKey (<LeftXPos + LeftShear*A/2, 0, 0>, Seed, true, -5, 0, 
                    IsBlack (Count + 9), 0.0, KeyThickness,
                    KeyA, KeyB, KeyC, KeyD, KeyE, KeyF, KeyG, KeyH, KeyI, KeyJ, KeyK,
                    KeyL, KeyM, KeyN, KeyP, KeyQ)
            
                translate Count*1.5*x
            }
            #local Count = Count + 1;
        #end
        
        #while (Count < 64)
            object
            {
                CreateKey (<LeftXPos + LeftShear*A, 0, 0>, Seed, true, -10, 0, 
                    IsBlack (Count + 9), LeftShear, KeyThickness,
                    KeyA, KeyB, KeyC, KeyD, KeyE, KeyF, KeyG, KeyH, KeyI, KeyJ, KeyK,
                    KeyL, KeyM, KeyN, KeyP, KeyQ)
            
                translate Count*1.5*x
            }
            #local Count = Count + 1;
        #end
        
        #while (Count < 84)
            object
            {
                CreateKey (<LeftXPos + LeftShear*A, 0, 0>, Seed, false, 0, 0, 
                    IsBlack (Count + 9), LeftShear, KeyThickness,
                    KeyA, KeyB, KeyC, KeyD, KeyE, KeyF, KeyG, KeyH, KeyI, KeyJ, KeyK,
                    KeyL, KeyM, KeyN, KeyP, KeyQ)
            
                translate Count*1.5*x
            }
            #local Count = Count + 1;
        #end
        
        translate <0, B + C - 10, F/2>
    }

#end // USE_INNER_MECHANICS

////////////////////////////////////////////////////////////////////////

// Lights

#switch (LIGHT_MODEL_NUM)

    #case (1)
    
        global_settings
        {
            assumed_gamma 1.0
            ambient_light 0
            max_trace_level 15
        
            #if (USE_RADIOSITY)
            
                radiosity
                {
                    error_bound 3.0
                    adc_bailout 0.1
                    count 20
                    nearest_count 1
                    brightness 0.65
                }
            
            #end
        }
        
        light_source
        {
            <-50, 250, -100>, color rgb 0.7
        }
        
        light_source
        {
            <100, 50, -200>, color rgb <0.5, 0.6, 0.2>*0.5
        }
    #break
    
    #case (2)

        global_settings
        {
            assumed_gamma 1.0
            ambient_light 0
            max_trace_level 15

            #if (USE_RADIOSITY)
            
                radiosity
                {
                    error_bound 3.0
                    adc_bailout 0.1
                    count 20
                    nearest_count 1
                    brightness 0.25
                }
            
            #end
        }
        
        light_source
        {
            0, color rgb <0.4, 0.4, 0.3> * 1.1
            area_light 10*(x-z)/sqrt(2), 10*y, 4, 4
            adaptive 1
            jitter
            
            fade_power 2.0
            fade_distance 100
            
            translate <-D/2 - E/2, C + B + 15, -F>
        }

        light_source
        {
            <100, 250, 30>, color rgb <0.1, 0.1, 0.4> * 0.80
            area_light 50*x, 50*y, 3, 3
            adaptive 1
            jitter
            
            fade_power 2.0
            fade_distance 150
        }
    #break
    
#end
