#ifndef MAZE_H
#define MAZE_H

#include <vector>

//! This class is a data structure representing a true maze.
/*!
  Create a new maze by instantiating the class. To solve the maze, call
  solve().
 */
class Maze
{
   friend class DrawableMaze;

public:
   //! Represents a cell in the maze.
   struct Cell
   {
      int north:1;
      int east:1;
      int south:1;
      int west:1;
      int visited:1;
      int path:1;
   };

   enum GenAlgo { DepthSearch, Disjoint };

   struct CellCoords { int x,y; };

   static const int MaxWidth = 8192;
   static const int MaxHeight = 8192;

   //! Constructor, creates a random maze of specified width and height
   /*!
     \param w is an integer in the range ]0..MaxWidth]
     \param h is an integer in the range ]0..MaxHeight]
     \param seed is an optional unsigned integer used to seed the random
            number generator.
    */
   Maze( int w, int h, unsigned int seed=0 );

   //! Copy constructor, creates a deep copy of the maze
   Maze( const Maze& );

   //! The destructor
   virtual ~Maze();

   //! Returns the width of the maze
   int getW() const { return w; }
   //! Returns the height of the maze
   int getH() const { return h; }

   //! Returns the cell at coordinates x,y
   Cell cellAt( int x, int y ) const;
   //! Returns the cell at coordinates c
   Cell cellAt( const CellCoords& c ) const { return cellAt(c.x,c.y); }

   //! Returns the cell coordinates of a path element
   CellCoords pathCoords(int) const;

   //! Returns the number of path elements
   int nPathCoords() { return path.size(); }

   //! Regenerates the maze randomly
   void regenerate( GenAlgo a=DepthSearch ) { init(); generate(a); }

   //! Solves the maze
   void solve( int sx, int sy, int gx, int gy );

   //! Solves the maze
   void solve( const CellCoords& s, const CellCoords& g );

   //! Returns true if the maze is solved
   bool isSolved() { return path.size()!=0; }
   
   //! Thrown if an illegal maze size is passed
   class IllegalSizeException {};
   //! Thrown if a cell not within the maze is referenced
   class IllegalCellRefException {};
   //! Thrown if a path element index number is out of range
   class IllegalPathRefException {};
   //! Thrown if an illegal maze generation algorithm is specified
   class IllegalAlgoException {};

protected:
   void init();
   void generate(GenAlgo);
   void generateAlgoDepthSearch();
   void generateAlgoDisjoint();
   void solve();

private:
   Cell *maze;
   int w,h;
   unsigned int seed;
   Cell *start,*goal;
   std::vector<Cell *> path;
};


inline Maze::Cell Maze::cellAt( int x, int y ) const
{
   if (!maze || x<0 || x>=w || y<0 || y>=h)
      throw IllegalCellRefException();
   return maze[y*w+x];
}

inline Maze::CellCoords Maze::pathCoords( int i ) const
{
   if (i<0 || (unsigned)i>path.size())
      throw IllegalPathRefException();

   CellCoords c;
   c.x = (path[i]-maze)%w;
   c.y = (path[i]-maze)/w;
   return c;
}

inline void Maze::solve( const CellCoords &s, const CellCoords& g )
{
   solve( s.x, s.y, g.x, g.y );
}


class DrawableMaze : public Maze
{
public:
   DrawableMaze( int w, int h, unsigned int s=0 ) : Maze(w,h,s) {};
   
   void drawMaze();

protected:
   virtual void drawCellEvent(int,int,int,int) {}
   virtual void drawWallEvent(int,int,int,int) {}
   virtual void drawPathEvent(int,int,int,int) {}
   virtual void drawStartEvent(int,int) {}
   virtual void drawGoalEvent(int,int) {}
};


#endif
