#include "colors.inc"
#include "functions.inc"
#include "textures.inc"
#include "woods.inc"
#include "woodmaps.inc"
#include "consts.inc"
#include "strings.inc"

default { finish { ambient 0 diffuse 0.5 } }

#declare Radiosity = on;
#declare Photons = on;
#declare Media = off;

#if (Media)
  media {
  intervals 1
  samples 3, 100
  method 3
  aa_level 4
  aa_threshold 0.1
  variance 1/256
  confidence 0.9
  scattering { 1, <0.9,1,0.7>*0.001 }
  density {
    planar
    scale 10
    translate 10*y
    color_map { [0 rgb 0][1 rgb 1]}
  }
}
#end

//
// Materials
// -------------------------------------------------------------------
//

#declare ShellNormal = function {
  pigment {
    crackle
    color_map {
      [0.0 rgb 0]
//      [0.2 rgb 1]
//      [0.4 rgb 0.5]
//      [0.6 rgb 1]
//      [0.8 rgb 0]
//      [1.0 rgb 0.5]
      [1.0 rgb 1]      
    }
  }
}

   
#declare ShellPlastic = material {
   texture {
      pigment { color rgbf <1,1,1,0.9> }
      finish { reflection { 0, 0.5 fresnel on } }
      //normal {  function { ShellNormal(x,y,z).gray } 0.1 }
      
   }
   interior { ior Quartz_Glass_Ior dispersion Quartz_Glass_Dispersion }
}

#declare FloorPlastic = material {
   texture {
      pigment { color Black }
      finish { specular 1.0 roughness 0.01 }
   }
}

#declare SwingPlastic = material{
   texture {
      pigment { color Orange }
      finish { specular 1.0 roughness 0.01 reflection { 0, 0.2 } }
   }
}

#declare BoardPlastic = material {
   texture {
      pigment { color Green }
      finish { specular 0.5 roughness 0.01 }
   }
}

//
// Objects
// ------------------------------------------------------------------------
//

#macro makeFloor(paneWidth,paneDepth,roomWidth,roomDepth,amb,crazy)
//
//  this macro creates a panelled wooden floor with each panel having
//  a random wood grain from woods.inc
//
//  amb sets the finish { ambient 1 diffuse 0 }, for creating an image_map
//  crazy is a probability for each pane having a random M_Wood colormap,
//  which looks a bit crazy.
//
#local pS = seed(9834562);
#local nPanes = 0;
#local pGap = 0.000;
object {
  intersection {
    union {
      plane { y, -0.01 pigment { color 0.25 } finish { ambient 1 diffuse 0 } }
      #local i = 0;
      #local flop = 0;  // for shifting every other column of panels
      #while (i < roomWidth)
	#local flop = flop + 1;
	#if (flop > 1) #local flop = 0; #end
	#local k = 0;
	#while (k < roomDepth)
	  #local rPos = RRand(0,100000,pS);
	  box {
	    #if (flop)
	      <i+pGap, -0.01, k+pGap>, 
	      <i + paneWidth-pGap, 0.0, k+paneDepth-pGap>
	    #else
	      <i+pGap, -0.01, k-paneDepth/2+pGap>, 
	      <i + paneWidth - pGap, 0.0, k+paneDepth/2-pGap>
	    #end	    

	    #local grain = RRand(0,1,pS);  // random grain
	    texture {
	      pigment {
		#local P = int(RRand(1,19.9,pS));
		#if (P > 9)
		  #local tmp = concat("P_WoodGrain",str(P,2,0),"A");
		#else
		  #local tmp = concat("P_WoodGrain",str(P,1,0),"A");
		#end
		Parse_String(tmp)
		//		P_WoodGrain3A
		#if (grain > 1 - crazy)
		  #local C = int(RRand(1,19.9,pS));
		  #if (C > 9)
		    #local tmp = concat("M_Wood",str(C,2,0),"A");
		  #else
		    #local tmp = concat("M_Wood",str(C,1,0),"A");
		  #end
		  color_map { Parse_String(tmp) }
		#else
		  color_map { M_Wood9A } // 7  // Use this color map if !crazy
		#end
		scale 0.1
		translate rPos
	      }
	      #if (amb)  // for creating an image_map
		finish { ambient 1 diffuse 0 }
	      #else
		finish { specular 0.7 roughness 0.005 reflection { 0.025 } }
	      #end
	    }
	    texture {
	      pigment {
		#if (P > 9)
		  #local tmp = concat("P_WoodGrain",str(P,2,0),"B");
		#else
		  #local tmp = concat("P_WoodGrain",str(P,1,0),"B");
		#end
		Parse_String(tmp)
		#if (grain > 1 - crazy)
		  #if (C > 9)
		    #local tmp = concat("M_Wood",str(C,2,0),"B");
		  #else
		    #local tmp = concat("M_Wood",str(C,1,0),"B");
		  #end
		  color_map { Parse_String(tmp)}
		#else
		  color_map { M_Wood9B } // 7
		#end
		scale 0.1
		translate rPos
	      }
	      #if (amb)
		finish { ambient 1 diffuse 0 }
	      #else
		finish { specular 0.7 roughness 0.005 reflection { 0.025 } }
	      #end
	    }
	  }
	  #local k = k + paneDepth;
	  #local nPanes = nPanes + 1;
	#end
	#local i = i + paneWidth;
      #end
    }
    box { <0, -0.01, 0>, <roomWidth, 0.01, roomDepth> }
  }
  #if (Photons)
    photons { target collect on }
  #end
}
#debug concat("\n  ", Str(nPanes), " floorpanes created\n")
#end  // macro makeFloor()

#declare ShellFloor = object {
  box { <1, 0, 1>, <40, 2, 40> }
  material { FloorPlastic } 
  #if (Photons)
    photons { target collect on reflection on }
  #end
}


#declare ShellCube = object {
   union {
      difference {
         box { <0,0,0>, <41,41,41> } // mm
         box { <2,-1,2>, <39,39,39> }
         object { ShellFloor } // take out the floor
         hollow on
         material { ShellPlastic }
      }
      object { ShellFloor }
   }
  #if (Photons)
    photons { target reflection on refraction on }
  #end
}

#declare Swing = object {
   union {
      intersection {
         plane { y, 0 rotate -30*x rotate -45*y translate 23*y }
         plane { y, 0 rotate  30*x rotate -45*y translate 23*y }
         plane { y, 2 inverse }
         plane { x, 39 }
         plane { x, 2 inverse }
         plane { z, 39 }
         plane { z, 2 inverse }
      } 
      //triangle { <5,20,5>, <36,20,36>, <36, 7, 5> }
 /*     sphere { <5, 20, 5>, 3 }
      sphere { <36, 20, 36>, 3 }
      cylinder { <5, 20, 5>, <36, 20, 36>, 3 } // the tilts
      cylinder { <5, 20, 5>, <36,  7,  5>, 3 }
      cylinder { <36, 7, 5>, <36, 20, 36>, 3 }
      cylinder { <36, 20, 36>, <5, 7, 36>, 3 }
      cylinder { <5,  7, 36>, <5, 20,  5>, 3 }
      
      cylinder { <5, 20, 5>, <5, 2, 5>, 3 }
      cylinder { <36, 7, 5>, <36, 2, 5>, 3 }
      cylinder { <36, 20, 36>, <36, 2, 36>, 3 }
      cylinder { <5, 7, 36>, <5, 2, 36>, 3 }*/
   }
   material { SwingPlastic }
  #if (Photons)
    photons { target collect on reflection on refraction off }
  #end
}


#declare Board = object { 
   difference {
      box { <2.001, 23, 2.001>, <38.999, 24, 38.999> }
      cone { <20.5, 25, 10>, 2, <20.5, 22.9, 10>, 1 }
      cone { <10, 25, 20.5>, 2, <10, 22.9, 20.5>, 1 }
      cone { <20.5, 25, 31>, 2, <20.5, 22.9, 31>, 1 }
      cone { <31, 25, 20.5>, 2, <31, 22.9, 20.5>, 1 } 
   }
   material { BoardPlastic }
  #if (Photons)
    photons { target collect on reflection off refraction off }
  #end
}


#declare Balls = object {
   union {
      sphere { <20.5, 26, 10>, 3 }
      sphere { <10, 26, 20.5>, 3 }
      sphere { <20.5, 26, 31>, 3 }
      sphere { <31, 26, 20.5>, 3 }
   }
   texture { Polished_Chrome }
  #if (Photons)
    photons { target collect off reflection on refraction off }
  #end
}



#declare Ring = object {
   difference { 
      torus { 13, 1 }
      box { <11, -2, -2.5>, <15, 2, 2.5> }
   }
   material { texture { Polished_Chrome } }
   translate <20,3, 20>
  #if (Photons)
    photons { target collect off reflection on refraction off }
  #end
}

#include "rand.inc"
#declare Balls2 = object {
   #local idx = 0;
   #local idxMax = 15;
   #local pS = seed(15);
   union {
      #while (idx < idxMax)
          #local rR = <RRand(2, 38,pS), 10, RRand(3, 38,pS)>;
          #local tN = <0,0,0>;
          #local tN2 = <0,0,0>;
          #local tR = trace(ShellCube, rR, -y, tN);
          #local tR2 = trace(Ring, rR, -y, tN2);
          #if ( (vlength(tN) > 0) & (vlength(tN2) = 0) )
             sphere { tR + y, 1 }
             #local idx = idx + 1;
          #end
      #end
  }
  material { texture { Polished_Chrome } }
  #if (Photons)
    photons { target reflection on collect off refraction off }  
  #end
}


//
// Composition
// -----------------------------------------------------------------------------
// 

#if (0)
#local Floor = makeFloor(15,3, 1000,1000,off,off)

object {
  Floor
  translate <-500,0,-500>
  clipped_by { sphere { 0, 500 } }
}
#end
#if (1)
object {
   union {
      object { ShellCube }
      object { Swing }
      object { Board }
      object { Balls }
   }
   translate <-20, 0, -20>
}
#end
object {
   union {
      object { ShellCube material { texture { pigment { rgbft <0,0,0.5,0.75,0.75> } finish { specular 0.5 roughness 0.001 } } } }
      object { Ring  }
      object { Balls2 }
   }
   hollow on 
   scale 0.2
   translate <-4, 24.001, -4>
}

#if (1)
plane { 
  y, 0
  pigment { color White }
  finish {
    reflection { 0.04, 0.5 fresnel on }
    specular 0.5 roughness 0.0005 
  }
  
  #if (Photons)
    photons {
      target 
      collect on
    }
  #end
}
#end

/*sphere {
  0,1
  hollow on
  pigment {
    gradient y
    color_map {
      [0.0 color White]
      [0.3 color White]
      [1.0 color SkyBlue+0.2]
    }
  }
  finish { ambient 1 diffuse 0 }
  scale 100
}
*/
sky_sphere {
  pigment {
    gradient y
    color_map {
      [0.0 color White]
      [0.3 color White]
      [1.0 color SkyBlue+0.2]
    }
  }
}  
global_settings { 
 
  max_trace_level 20

  #if (Radiosity)
    #version unofficial MegaPov 1.10;
    exposure 0.02
    exposure_gain 60
    radiosity {
      count 50
      error_bound 1
      pretrace_start 1
      pretrace_end 0.01
      nearest_count 20
      low_error_factor 0.5
      brightness 1
      minimum_reuse 0.009
      max_sample -1
      recursion_limit 3
      gray_threshold 0.75
      adc_bailout 1/256 
      load_file "toy.rad"
      always_sample on
      #if (Media)
	media on
      #end
      normal on
    }
  #end

  #if (Photons)
    photons {
      spacing 0.1                 // specify the density of photons
      #if (Media)
	media 100,2  // media photons
      #end
      gather 20,100 // povray defaults (good)
      jitter 0.4                 // jitter phor photon rays
      //max_trace_level 25
      //adc_bailout 1/255          // see global adc_bailout
      radius ,0.5,,0.5  // try for a little quicker ...
      autostop 0                 // photon autostop option
      expand_thresholds 0.2, 40
      max_trace_level 5          // optional separate max_trace_level
      load_file "toy-onepass.pho"       // save photons to file
      //load_file "filename"       // load photons from file
    }
  #end 
}

#declare AreaLight = on;

light_source { 
  <50, 50, 25> 
  color <0.8, 0.8, 1.0>
  fade_power 2
  fade_distance 150
  #if (AreaLight)
     area_light 20*x, 20*y, 8, 8
     circular
     orient
     jitter
     adaptive 2
   #end
  spotlight
  radius 20
  falloff 50
  point_at <0, 20, 0>
  #if (Photons)
    photons { reflection on refraction on area_light }
  #end
  #if (Media)
    media_attenuation on
    media_interaction on
  #end
  
}

#if (0)
light_source {
   <-50, 15, -50>
   color <0.8, 0.8, 1.0>
   fade_power 2
   fade_distance 150
   #if (AreaLight)
      area_light 10*x, 10*y, 2, 2
      circular
      orient
      jitter
   #end
   spotlight
      radius 20
      falloff 50
      point_at <0, 20, 0>
   photons { reflection on refraction on }
}

light_source { 
  <-65, 10, 0> color White 
  fade_power 2
  fade_distance 100
  photons { reflection on refraction on }
}
#end
//sphere { 0, 10000 hollow texture { Shadow_Clouds scale 10000 finish { ambient 1 diffuse 0 }}  }

#declare FocalBlur = off;
camera { 
//  location <50, 65, 95>
  location <39.999,80,80>
  right image_width/image_height*x
  up y
  direction z
  angle 60
   look_at <-28, 0, 0>
   #if (FocalBlur)
      aperture 5.0
      blur_samples 10
      focal_point <0, 20, 0>
      confidence 0.5
      variance 1/128
   #end
}

