// Persistence of Vision Ray Tracer Scene Description File
// File: molecules.inc
// Vers: 3.6
// Desc: Creates some primitives for molecular models
// Date: 6/24/05
// Auth: David Nash
// Todo: (1) With some work, I think I could turn this into
//           a full-blown macro to make hydrocarbons.
//       (2) I never did figure out how to make a decent looking
//           double bond.  Maybe splines could do the trick.

// ==========================
// License
// ==========================
// This work is licensed under the Creative Commons
// Attribution-ShareAlike License. To view a copy of
// this license, visit
// http://creativecommons.org/licenses/by-sa/2.5/ 
// or send a letter to
//      Creative Commons
//      559 Nathan Abbott Way
//      Stanford, California 94305, USA.

// ==========================
// Include files and variables
// ==========================
#version 3.6;
#include "colors.inc"
#include "woods.inc"
#include "math.inc"

// ==========================
// Textures
//      T_Bond      texture for single bond stick
//      T_Carbon    texture for carbon atom model
//      T_Oxygen    texture for oxygen atom model
//      T_Hydrogen  texture for hydrogen atom model
// ==========================

#declare T_Bond = 
texture { T_Wood1 scale 0.2 }

#declare T_Carbon = 
texture {
    pigment { Black }
    finish { phong 0.9 phong_size 60 }
}

#declare T_Oxygen = 
texture {
    pigment { Red }
    finish { phong 0.9 phong_size 60 }
}

#declare T_Hydrogen =
texture {
    pigment { White }
    finish { phong 0.9 phong_size 60 }
}

// ==========================
// Models
//      Bond        wooden stick single bond model
//      Carbon      carbon atom model with no holes
//      E_Carbon    drilled carbon atom model
//      Oxygen      oxygen model with no holes
//      E_Oxygen    drilled oxygen atom model
//      Hydrogen    hydrogen model with no holes
//      E_Hydrogen  drilled hydrogen atom model
//      CH3_1       a methyl group with the carbon at the origin
//                  and an opening in the +y direction
//      CH2_2       a hydrocarbon backbone piece with the carbon at the
//                  origin and an opening in the +y and -z direction.
//      Methane     methane molecule
//      Octane      octane molecule
// ==========================

/*
Carbon tetrahedral bond angle = 109.5
       H            |
       |            | 
109.5  C            y
      /    <-a      |
     H  <-b         +-----z----

Imagine a right triangle with the C-H bond being the hypotneuse
180-109.5=70.5 degrees in angle a
180-90-70.5=19.5 degrees in angle b
so trig gives us the translation in the y and z direction
sin(19.5)=dy/2.5 dy=.8345
cos(19.5)=dz/2.5 dx=2.3566
*/

#declare BondLength = 2.5;
#declare tetra_angle = 19.5;
#declare dy = sind(tetra_angle)*BondLength;
#declare dz = cosd(tetra_angle)*BondLength;

#declare Bond = 
object {
	cylinder {
		<0, 0, 0>, <0, 0, BondLength>, 0.2
		texture { T_Bond }
	}
}


#declare Carbon =
object {
	sphere { <0, 0, 0>, 1
		texture { T_Carbon }
	}
}

#declare E_Carbon = // empty
difference {
	sphere { <0, 0, 0>, 1
		texture { T_Carbon }
	}
	object { Bond rotate x*-90 translate <0, 0, 0> }
	object { Bond rotate -x*tetra_angle translate <0, -dy, -dz> }
	object { Bond rotate -x*tetra_angle translate <0, -dy, -dz> rotate y*120}
	object { Bond rotate -x*tetra_angle translate <0, -dy, -dz> rotate y*-120}
}

#declare Oxygen =
object {
	sphere { <0, 0, 0>, 1
		texture { T_Oxygen }
	}
}

#declare E_Oxygen = // empty
difference {
	sphere { <0, 0, 0>, 1
		texture { T_Oxygen }
	}
	object { Bond rotate x*-90 translate <0, 0, 0> }
	object { Bond rotate -x*tetra_angle translate <0, -dy, -dz> }
}

#declare Hydrogen =
object {
	sphere { <0, 0, 0>, 0.75
		texture { T_Hydrogen }
	}
}

#declare E_Hydrogen =
difference {
    object {
        sphere { <0, 0, 0>, 0.75 }
        texture { T_Hydrogen }
    }
    object { Bond rotate x*-90 translate <0, 0, 0> }
}


#declare CH3_1 =
union {
	object { Carbon}
	object { Hydrogen translate <0, -dy, -dz> }
	object { Bond rotate -x*tetra_angle translate <0, -dy, -dz> }
	object { Hydrogen translate <0, -dy, -dz> rotate y*120}
	object { Bond rotate -x*tetra_angle translate <0, -dy, -dz> rotate y*120}
	object { Hydrogen translate <0, -dy, -dz> rotate y*-120}
	object { Bond rotate -x*tetra_angle translate <0, -dy, -dz> rotate y*-120}
}

#declare CH2_2 =
union {
	object { Carbon}
	object { Hydrogen translate <0, -dy, -dz> rotate y*120}
	object { Bond rotate -x*tetra_angle translate <0, -dy, -dz> rotate y*120}
	object { Hydrogen translate <0, -dy, -dz> rotate y*-120}
	object { Bond rotate -x*tetra_angle translate <0, -dy, -dz> rotate y*-120}
}

#declare Methane = 
union {
	object { Carbon translate <0, 0, 0> }
	object { Hydrogen translate <0, BondLength, 0> }
	object { Bond rotate x*-90 translate <0, 0, 0> }
	
	object { Hydrogen translate <0, -dy, -dz> }
	object { Bond rotate -x*tetra_angle translate <0, -dy, -dz> }

	object { Hydrogen translate <0, -dy, -dz> rotate y*120}
	object { Bond rotate -x*tetra_angle translate <0, -dy, -dz> rotate y*120}
	object { Hydrogen translate <0, -dy, -dz> rotate y*-120}
	object { Bond rotate -x*tetra_angle translate <0, -dy, -dz> rotate y*-120}
}

#declare Octane =
merge {
	object { CH3_1 rotate y*180 }//quick fix
	object { Carbon 
		rotate <180, 180, 0> 
		translate <0, BondLength*1, -dz*0>
	}
	object { Bond rotate -x*90 }
	
	object { CH3_1
		rotate x*180 // flip over
		rotate -x*(90-tetra_angle) // position
		translate y*BondLength // move up one slot
		translate <0, dy, -dz> // move out to proper position
		rotate y*120
	}

	object { Bond
		rotate -x*(90+90-tetra_angle) //position
		translate y*BondLength // move up one slot
		rotate y*120
    }

	object { CH3_1
		rotate x*180 // flip over
		rotate -x*(90-tetra_angle) // position
		translate y*BondLength // move up one slot
		translate <0, dy, -dz> // move out to proper position
		rotate y*-120
	}
	
	object { Bond
		rotate -x*(90+90-tetra_angle) //position
		translate y*BondLength // move up one slot
		rotate y*-120
    }

	object { Methane
		rotate x*180 // flip over
		rotate -x*(90-tetra_angle) // position
		translate y*BondLength // move up one slot
		translate <0, dy, -dz> // move out to proper position
	}
    
	object { Methane 
		rotate <180, 0, 0> 
		translate <0, BondLength*2+dy*1, -dz*1>
	}

	object { Methane 
		rotate <180, 0, 0>
		rotate -x*(90-tetra_angle) // position 
		translate z*-dz
		rotate y*60
		translate <0, BondLength*2+dy*2, -dz>
	}	
	
	object { Methane 
		rotate <180, 0, 0>
		rotate -x*(90-tetra_angle) // position 
		translate z*-dz
		rotate y*-60
		translate <0, BondLength*2+dy*2, -dz>
	}	
}
