
// Glenn's POV-Ray Scene File - "Escape Velocity"
// created 5-Dec-05
// last update 31-Dec-05
//
// Copyright 2005 Glenn McCarter (glenn@mccarters.net). Some rights reserved.
// This work is licensed under the Creative Commons Attribution License. 
// To view a copy of this license, visit http://creativecommons.org/licenses/by/2.0/ 
// You are free:
// - to copy, distribute, display, and perform the work
// - to make derivative works
// - to make commercial use of the work
// Under the following conditions:
// - Attribution. You must give the original author credit.
// - For any reuse or distribution, you must make clear to others the license terms of this work.
// - Any of these conditions can be waived if you get permission from the copyright holder.
// Your fair use and other rights are in no way affected by the above. 
//
//
// This is the main scene file.
// Hydroplane Racing
//
// Scene uses RH coordinate system, Z = up.
// Scene origin is roughly under the main boat,
// with +X towards the right, and +Y heading away from the camera.


//===============================================
//             Scene Switches
//===============================================

// Set this to 2 for best render

// Set these to 1 for best render
#declare ThisIsMaster = 1; // declares this the Master File, leave at 1
#declare Boats        = 1; // 0=off, 1=detailed boat meshes
#declare RoosterQual  = 1; // 0=off, 1=spheres
#declare EjectionSeat = 1; // 0=off, 1=seat + rocket plume
#declare RocketQual   = 1; // 0=quick geometry, 1=full media
#declare DoFog        = 1; // 0=no fog, 1=fog
#declare AreaSun      = 1; // 0=normal light, 1=area light for the sun
#declare Photon       = 0; // 0=off, 1=photons

// Leave these at 0
#declare RenderMe     = 0; // supresses self-rendering of the include files, leave at 0
#declare Radios       = 0; // 0=off, 1=quick, 2=best radiosity


#declare My_Cam = 1;
// 1=	Render_Cam
// 8=	Special_Cam

//===============================================
//             Global Settings
//===============================================

#version 3.5;
global_settings {
	adc_bailout 1/255 // default 1/255
	ambient_light <1.0,1.0,1.0> // default <1,1,1>, use 0 for radiosity
	assumed_gamma 1.0
	hf_gray_16 off
	irid_wavelength <0.25,0.18,0.14>
	max_intersections 64
	max_trace_level 150
	number_of_waves 13 // for ripple texture
	noise_generator 2

#if ( Photon = 1 )
	#debug   concat("Photons on in global settings","\n")
	photons {
		count 5000000
//		autostop 0
		jitter .4
	}
#end

#switch (Radios)
#case (1)
	ambient_light <0.0,0.0,0.0> // this is most realistic with radiosity on
// QUICK TEST
  radiosity{
    adc_bailout 0.01/0.02// CHANGE - use adc_bailout = 0.01 / brightest_ambient_object
		always_sample on     // CHANGE - default on, use off to reduce blotchiness
		brightness 0.75       // CHANGE - default 1.0 for full effect, 0.0 = no radiosity
    count 20             // CHANGE - default 35, 2 is too low, typ 20 to 150, max 1600 rays sent out
    error_bound 1.0      // CHANGE - default 1.8, typ 1 to 3, hi = flat shadows, low = deep shadows
    gray_threshold 0.2   // CHANGE - 0.0 = unrealistically full color bleed, 1.0 = no color bleed
    low_error_factor .6  // leave this, default 0.5
		max_sample -1        // leave this negative unless you have a very bright object causing splotchiness
		media off            // default off, turn on if using media in scene
    minimum_reuse 0.015  // leave this, default 0.015
    nearest_count 5      // CHANGE - default 5, below 4 gets patchy, typ 3 to 10, max 20
		normal off           // default off, turn on for shadow areas to show bumps
    pretrace_start 0.08  // CHANGE - default 0.08, percent of final image size in first mosaic
    pretrace_end   0.02  // CHANGE - default 0.04, percent of final image size in last mosaic
    recursion_limit 2    // CHANGE - default 3, 1 or 2 is often sufficient
  }
#break
#case (2)
// FINAL RENDER
//	ambient_light <0.0,0.0,0.0> // this is most realistic with radiosity on
	ambient_light <0.02,0.02,0.02> // test
  radiosity{
    adc_bailout 0.01/0.02// CHANGE - use adc_bailout = 0.01 / brightest_ambient_object
		always_sample on     // CHANGE - default on, use off to reduce blotchiness
//		brightness 0.7       // CHANGE - default 1.0 for full effect, 0.0 = no radiosity
		brightness 0.55       // CHANGE - default 1.0 for full effect, 0.0 = no radiosity TEST
    count 20             // CHANGE - default 35, 2 is too low, typ 20 to 150, max 1600 rays sent out
    error_bound 2.4      // CHANGE - default 1.8, typ 1 to 3, hi = flat shadows, low = deep shadows TEST
    gray_threshold 0.2   // CHANGE - 0.0 = unrealistically full color bleed, 1.0 = no color bleed
    low_error_factor .5  // leave this, default 0.5
		max_sample -1        // leave this negative unless you have a very bright object causing splotchiness
		media off            // default off, turn on if using media in scene
    minimum_reuse 0.015  // leave this, default 0.015
    nearest_count 8      // CHANGE - default 5, below 4 gets patchy, typ 3 to 10, max 20
		normal off           // default off, turn on for shadow areas to show bumps
    pretrace_start 0.06  // CHANGE - default 0.08, percent of final image size in first mosaic
    pretrace_end   0.01  // CHANGE - default 0.04, percent of final image size in last mosaic
    recursion_limit 3    // CHANGE - default 3, 1 or 2 is often sufficient
		save_file "MainSceneRadiosity"
  }
   // end radiosity section
#break
#else
#end
}


//===============================================
//             Includes
//===============================================

#include "GIncludeRH.inc" // Glenn's standard includes
#if (Boats=1)
	#include "Hydroplane01.inc"
	#include "Hydroplane02b.inc"
	#include "Hydroplane03.inc"
#end
#include "RoosterTail.inc"
#if (EjectionSeat=1)
	#include "EjectionSeat.inc"
#end
#include "Canopy.inc"
#include "Bridge.inc"
#include "Fishy.inc"

//===============================================
//             Variables
//===============================================

//#declare CamPos = <-23*Feet, -100*Feet, 10*Feet>; // Camera Position
#declare CamPos = <-23*Feet, -100*Feet, 5*Feet>; // Camera Position
//sphere { <0,0,0>,1 scale 6*Feet texture{pigment{color Green}} translate CamPos }
//cylinder { CamPos,<-2*Feet,0*Feet,16*Feet>,24 texture{pigment{color Green}} }

//===============================================
//             Cameras
//===============================================

#declare Render_Cam =
	camera {
		perspective
		direction <0.0,     1.0,  0.0>
		sky       <0.12,     0.0,  1.0>  // Use right handed-system
		up        <0.0,     0.0,  1.0>  // Where Z is up
//		right     <16/9, 0.0,  0.0>  // aspect ratio 16:9=1.78 widescreen
		right     < 4/3, 0.0,  0.0>  // aspect ratio  4:3=1.33 normal
//		right     < 1.0, 0.0,  0.0>  // aspect ratio  1:1=1.00 square
//		right     < 7/8, 0.0,  0.0>  // aspect ratio  7:8=0.87 vertical format
//		right     < 4/5, 0.0,  0.0>  // aspect ratio  4:5=0.80 vertical
//		right     < 3/4, 0.0,  0.0>  // aspect ratio  3:4=0.75 vertical
	angle 34
	location CamPos
	look_at <-2*Feet,0*Feet,16*Feet> // main render
//	look_at <45*Feet,280*Feet,61*Feet> // look further right
//	look_at <-45*Feet,280*Feet,61*Feet> // look further left
}
#declare Special_Cam =
	camera {
		perspective
		direction <0.0,     1.0,  0.0>
		sky       <0.0,     0.0,  1.0>  // Use right handed-system
		up        <0.0,     0.0,  1.0>  // Where Z is up
		right     <4/3,     0.0,  0.0>
//	angle 45 // render view
//	location CamPos
//	look_at <45*Feet,280*Feet,61*Feet> // right of main render
//	angle 55
//	location <100*Feet,1200*Feet,200*Feet> // reverse angle
//	look_at <18*Feet,20*Feet,50*Feet>
//	angle 55
//	location <0*Feet,-400*Feet,50*Feet> // backup from origin
//	look_at <1*Feet,1*Feet,4*Feet>
//	angle 55
//	location <0*Feet,-2000*Feet,500*Feet> // backup overview
//	look_at <1*Feet,1*Feet,200*Feet>
//	angle 8
//	location <0*Feet,-3000*Feet,3000*Feet> // blimp zoom
//	look_at <100*Feet,5*Feet,5*Feet>
//	angle 75
//	location <0*Feet,-5000*Feet,5000*Feet> // blimp wide
//	look_at <150*Feet,5*Feet,5*Feet>
//	angle 55
//	location <0*Feet,-30*Feet,400*Feet> // helicopter
//	look_at <1*Feet,1*Feet,2*Feet>
//	angle 40
//	location <-25*Feet,10*Feet,2*Feet> // boat 2 closeup
//	look_at <-24*Feet,40*Feet,10>
	angle 108
	location CamPos // fisheye
//	look_at CamPos+<0,40*Feet,40*Feet>
	look_at CamPos+<0,-40*Feet,40*Feet>	}

camera {
#switch (My_Cam)
#case (1) Render_Cam
#break
#else Special_Cam
#end
}

//===============================================
//             Textures
//===============================================

#declare TestWhiteTex =
	texture {
		pigment { color rgb<1,1,1> }
	}

//===============================================
//             Lighting
//===============================================


#declare Sun =
	light_source {
		<0,0,0>
		color rgb <0.584,0.539,0.543>*2.3 // bluish
		#if (AreaSun=1)
			area_light <0,0,0.5*Miles>,<0,0.5*Miles,0>,4,4 adaptive 1 jitter circular orient
		#end
		#if ( Photon = 1)
		photons {
			#debug concat("Photons on in light_source","\n")
			refraction on
			reflection on
		}
		#end
	}

object { Sun
	translate <0,0,5*Miles> // noon at Equator: directly overhead
	rotate y*-33 // towards left
  rotate z*40 // behind camera
}

#declare FillLight =
	light_source {
		<-140*Feet,-140*Feet,3*Feet>, <1.13,1.08,0.83>*.4
		area_light
		<-40*Feet,40*Feet,0>, <0,0,40*Feet>,4,4
	//  spotlight point_at <0,0,2*Feet> radius 25 falloff 36
		#if ( Photon = 1)
		photons {
			refraction off
			reflection off
			#debug concat("Photons OFF in fill light","\n")
		}
		#end
	}

#declare SeatLight =
	light_source {
		<40*Feet,-30*Feet,100*Feet>,  <1.13,1.08,0.83>*.64
	}

#if ( DoFog )
fog {
	fog_type 2 // ground fog
	fog_alt 150*Feet
	up <0,0,1>
	distance 5000*Feet // extremely thin
	color rgbt<.45,.47,.50,.2> // nice blue-green haze
}

#end

//===============================================
//             Environment
//===============================================

background { color rgb <0.576,0.667,0.832>*.97 } // deep blue grey

sky_sphere {
 pigment {
 	gradient z
 		color_map {
/*
 			[ 0.00 rgb<0,0,0> ] // test
 			[ 0.95 rgb<0,0,0> ]
 			[ 0.95 rgb<1,1,1> ]
 			[ 0.99 rgb<1,1,1> ]
*/
 			[ 0.00 rgb <0.576,0.667,0.832>*.97 ]
 			[ 0.45 rgb <0.076,0.567,1.832>*.57 ]
 			[ 0.75 rgb <0.076,0.167,0.732>*.47 ]

 		}
	}
}

#declare GroundPlane =
	plane { z,-0.1 pigment { checker rgb 1, rgb .75 } scale 10*Feet }
//object { GroundPlane }

#declare ClearPig = pigment { color rgbt<0,0,0,1> }
#declare BackdropPig =
	pigment {
		image_map { // image is always projected from -z, with front facing +z
		png "Backdrop.png"
		once
		}
	}
#declare Backdrop =
	box { 0,1
		texture {
			pigment {
				gradient y
				pigment_map {
					[ 0.25 ClearPig ]
					[ 0.29 BackdropPig ]
				}
			}
			finish { ambient 1.0 diffuse 0.0 }
		}
			rotate x*90
			translate <-0.5,0,0> // center it
//			scale <-1,1,1> // reverse it
			#local ScaleFactor = 250*Feet;
			scale <ScaleFactor,1,ScaleFactor*3/4>
	}

#declare FLP1 = pigment {
	bozo
	color_map {
		[ 0.0 rgb<.022,.029,.022> ]
		[ 1.0 rgb<.020,.160,.040> ]
	}
	scale 20*Feet/(620*Feet)
}

#declare FarlandTex =
	texture {
		pigment {
			gradient z
			pigment_map {
//				[ 0.10 rgb<0.447,0.636,0.351>*.3 ] // rocky
//				[ 0.24 rgb<.17,.25,.14> ] // to dirty rock
//				[ 0.32 rgb<.04,.14,.08> ] // to dark green
				[ 0.50 FLP1 ] // mottled green trees
//				[ 0.95 FLP1 ] // mottled green trees
//				[ 1.00 rgb<0.447,0.636,0.351>*.3 ] // rocky
			}
			warp {turbulence 0.4}
		}
	}
#declare FarlandTex = texture { pigment { color rgb<.1,.2,.1> } }

#declare FarCoastline =
	height_field { // starts extruded into +Y, with image upper left corner at <0,y,1>
		png "FarlandHF1.png"
		smooth
		water_level 1/255
		rotate x*90
		scale <1,-1,1>
		scale <1.1*Miles,1.1*Miles,350*Feet>
		texture { FarlandTex scale 620*Feet translate <0,0,-50*Feet> }
	}


#declare WaveNorm = // this is the boat chop
	normal {
			granite 0.2 scale 24*Feet
	}

#declare RippleNorm = // this is wake tracks
	normal {
			bumps 0.24 scale 5*Feet
		}

#declare AvgWaveNorm =
	normal { 
		average
		normal_map {
			[ 1.0 RippleNorm ]
			[ 1.0 WaveNorm ]
		}
	}
#declare LinearNorm =
	normal {
		gradient x
		slope_map {
			[ 0.0 <0,0> ]
			[ 0.5 <1,1> ]
			[ 1.0 <0,0> ]
		}
		rotate z*-8
		warp { turbulence 0.3 }
		scale 20

	}
#declare SeaWaterPig =
	pigment {
		bozo
		color_map {
//			[ 0.00 rgb<0.167,0.316,0.382> ]
//			[ 1.00 rgb<0.227,0.506,0.762> ]
			[ 0.00 rgb<0.467,0.316,0.382>*.1 ]
			[ 1.00 rgb<0.127,0.366,0.762> ]
		}
		scale 20
	}
#declare SeaWaterTex =
	texture {
//		pigment { color rgbt<0.227,0.416,0.451,0.64> } // too clear
//		pigment { color rgb<0.227,0.416,0.451> } // no trans, too opaque
//		pigment { color rgb<1,1,1> } // test white
//		pigment { color rgbf<0.227,0.416,0.282,0.47> } // brownish
///		pigment { color rgbf<0.167,0.316,0.382,0.07> } // much more blue and clear
//		pigment { color rgb<0.167,0.316,0.382> } // test
//		pigment { color rgb<0.167,0.316,0.382>*4 } // test 2
		pigment { SeaWaterPig } // test 3
//		pigment { color rgbt<0.167,0.316,0.382,0.93> } // test, takes forever to render
//		pigment { color rgbt<0.044,0.084,0.190,0.97> } // blue, too slatelike
//		pigment { color rgbt<.004,0.24,0.490,0.75> } // too bland
//		pigment { color rgbt<.004,0.040,0.190,0.75> } // more cobalt
//		pigment { color rgbt<.004,0.040,0.190,1.0> } // clear
//		pigment { color rgbf<1,1,1,0.97> } // test white filtered
//		pigment { color rgbt<.01,.01,.01,0.97> } // test dark grey
//		normal { bumps .25 scale <2.2*Feet,2.2*Feet,12> } // too oily and large bumps
//		normal { bumps .25 scale <0.7*Feet,0.7*Feet,3> } // ok

//		normal { 
//			average
//			normal_map {
//				[ bumps .20 ]//scale <24*Feet,24*Feet,3*Feet>
//				[ granite .30 ]//scale <24*Feet,24*Feet,3*Feet>
//			}
//			scale <24*Feet,24*Feet,3*Feet>
//		}

		normal { 
			AvgWaveNorm
			translate <120*Feet,-0*Feet,0>
		}


//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { 0.04, 0.64 } conserve_energy}
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.04,0.14,0.04>, <0.99,0.99,0.99> } conserve_energy} // mirrorlike
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.00,0.0,0.00>, <0.42,0.74,0.84> } conserve_energy} // too crystal blue/green
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.00,0.0,0.00>, <0.12,0.34,0.43> } conserve_energy} // much too blue
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.00,0.0,0.00>, <0.22,0.44,0.67> } conserve_energy}
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.00,0.0,0.00>, <0.72,0.74,0.83> } conserve_energy}
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.00,0.0,0.00>, <0.16,0.21,0.28> } conserve_energy} // low reflection test, too blue reflection
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.00,0.0,0.00>, <0.03,0.11,0.15> } conserve_energy} // low reflection test, dk blue/brown
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.08,0.08,0.10>, <0.278,0.361,0.353>*1.8 fresnel on } conserve_energy} // lower reflection test, fresnel causes mirror puddles
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { 0.61 fresnel on } conserve_energy} // lower reflection test
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 conserve_energy} // ZERO reflection test
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection {0.3} } // simple reflection test, no fresnel
//		finish { reflection {0.3 } }
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 } // ZERO reflection test
///		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.08,0.08,0.10>, <0.278,0.361,0.353> } conserve_energy} // mid reflection ok
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.08,0.08,0.30>, <0.278,0.361,0.653> } conserve_energy} // too blue reflections
		finish { ambient 0.1 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.08,0.08,0.10>, <0.278,0.361,0.353> } conserve_energy} // close, too slate blue
//		finish { ambient 0.1 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.08,0.08,0.08>, <0.278,0.278,0.278> } conserve_energy} // test, nogood
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.38,0.38,0.38>, <0.678,0.661,0.653> } conserve_energy} // test, no pillars
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.38,0.38,0.38>, <0.678,0.661,0.653> } } // test 2, no pillars
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection <0.678,0.661,0.653> } // test, no pillars
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection <0.008,0.001,0.003> } // test, no pillars
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 } // test, no pillars
//		finish { ambient 0.1 diffuse 0.02 specular 1.21 roughness .003 } // test ambient,
	}

/*
#declare SeaWaterTex2 = //DELETE
	texture {
		pigment { color rgb<0.84,0.80,0.90> } // white
//		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 } // ZERO reflection test
//		finish { ambient 0.1 diffuse 0.6 specular .21 roughness .003 reflection { 0.61 fresnel on } conserve_energy} // lower reflection test
		finish { reflection {0.3 } }
	}
*/

/*
#declare SeaWaterWaveTex =
	texture {
		pigment { color rgbt<.004,0.040,0.190,1.0> } // clear
		normal { 
			bozo 
			normal_map {
				[ 0.4 Wave2Norm scale 1/(60*Feet) ]
				[ 0.6 RippleNorm scale 1/(60*Feet) ]
			}
			scale 60*Feet
			translate <-400*Feet,0,0>
		}
		finish { ambient 0.001 diffuse 0.02 specular 1.21 roughness .003 reflection { <0.08,0.08,0.10>, <0.278,0.361,0.353> fresnel on } conserve_energy} // mid reflection test
	}
*/

#declare SeaWaterInt =
//	interior { ior 1.33 caustics 10 fade_distance  12 fade_power 1.0 fade_color <0.03,0.25,0.60> }
//	interior { ior 1.33 caustics 10 fade_distance  12 fade_power 1.0 fade_color <0.03,0.55,0.36> }
//	interior { ior 1.33 caustics 10 fade_distance  24 fade_power 1.7 fade_color <0.03,0.55,0.36> } // greenish
//	interior { ior 1.33 caustics 10 fade_distance  24 fade_power 1.7 fade_color <1,1,1> } // clear test
//	interior { ior 1.33 caustics 10 fade_distance   5 fade_power 1.7 fade_color <0.03,0.26,0.55> } // test
//	interior { ior 1.33 caustics 10 fade_distance  10 fade_power 1.7 fade_color <0.03,0.43,0.25> } // test green, too much algea
//	interior { ior 1.33 caustics 10 fade_distance  10 fade_power 2.0 fade_color <0.23,0.43,0.35> } // test blue-green
//	interior { ior 1.33 caustics 10 fade_distance  20 fade_power 2.0 fade_color <0.04,0.18,0.22> } // test depth
//	interior { ior 1.33 caustics 10 fade_distance 100 fade_power 2.0 fade_color <0.04,0.18,0.22> } // test even deeper
//	interior { ior 1.33 caustics 10 fade_distance   1 fade_power 2.0 fade_color <0.04,0.18,0.22> } // still not deep enough
//	interior { ior 1.33 caustics 10 fade_distance  .7 fade_power 2.0 fade_color <0.12,0.22,0.29> } // test (got worse, not better)
//	interior { ior 1.33 caustics 10 fade_distance .007 fade_power 2.0 fade_color <0.12,0.22,0.29> } // test (no help)
//	interior { ior 1.33 caustics 10 fade_distance 100 fade_power 2.0 fade_color <0.12,0.22,0.29> } // test (even worse)
//	interior { ior 1.33 caustics 10 fade_distance   1 fade_power 5.0 fade_color <0.12,0.22,0.29> } // test (slightly better, still sucks)
//	interior { ior 1.33 caustics 10 fade_distance   1 fade_power 5.0 fade_color <0.008,0.008,0.014> } // test (NOW WE'RE TALKING!)
//	interior { ior 1.33 caustics 10 fade_distance 6 fade_power 2.0 fade_color <0.008,0.008,0.014> } // test (too black inky)
//	interior { ior 1.33 caustics 10 fade_distance 6 fade_power 2.0 fade_color <0.002,0.038,0.070> } // test ()
	interior { ior 1.33  } // test ()

#declare SeaWaterMat = // material
	material {
		texture { SeaWaterTex }
//		texture { SeaWaterWaveTex }
		interior { SeaWaterInt }
	}

#declare CausticTex =
	texture {
		pigment {
			bozo
			color_map {
				[ 0.44 rgbt <1,1,1,1> ]
				[ 0.50 rgbt <1,1,1,.8> ]
				[ 0.55rgbt <1,1,1,1> ]
			}
			scale 20
		}
		finish { ambient .1 }
	}

#declare SandyBottom =
	pigment {
		granite
		color_map {
//			[ 0.00 rgb<0.957,0.906,0.851>*.8 ] // too light and dry
			[ 0.00 rgb<0.957,0.906,0.851>*.24 ] // medium grey
//			[ 1.00 rgb<0.957,0.906,0.851>*.6 ] // too light and dry
			[ 0.50 rgb<0.457,0.906,0.451>*.10 ] // greenish
			[ 1.00 rgb<0.957,0.906,0.851>*.10 ] // medium grey
		}
		scale 400*Feet
	}
#declare RockyBottom =
	pigment {
		bozo
		color_map {
			[ 0.25 color rgb <0.345,0.392,0.333>*.1 ]
			[ 0.50 color rgb <0.184,0.292,0.114>*.250 ]
			[ 0.60 color rgb <0.086,0.282,0.302>*.250 ]
			[ 0.80 color rgb <0.086,0.282,0.302>*.03 ]
		}
		scale 2*Feet
	}
#declare RockySand =
	pigment {
		granite
		pigment_map {
//			[ 0.40 SandyBottom scale 1/(200*Feet) ]
			[ 0.20 SandyBottom scale 1/(200*Feet) ]
			[ 0.48 RockyBottom scale 1/(200*Feet) ]
		}
		turbulence 0.16
		scale 200*Feet
	}

#declare SeaFloorTex =
	texture {
		pigment {
			granite
			pigment_map {
				[ 0.50 SandyBottom scale 1/(2000*Feet) ]
				[ 0.58 RockySand scale 1/(2000*Feet) ]
			}
			translate <50,50.04,50> // this sets up a new random layout
			scale 2000*Feet
		}
		normal { bumps .5 scale 2.7*Meter }
		finish { ambient .1 diffuse .7 specular .1 roughness .15 }
	}
/*
#declare SeabedHF =
	height_field { // starts extruded into +Y, with image upper left corner at <0,y,1>
//		png "SeabedHF.png"
//		png "SeabedHF2.png"
		png "SeabedHF3.png"
		smooth
		water_level .01
		rotate x*90 // z=up
		scale <1,-1,1> // align top left to <0,1,z>
		scale <500*Feet,1000*Feet,26*Feet>
		translate <-250*Feet,-100*Feet,-27*Feet>
	}
*/
#declare DistantSeabed =
//	difference {
		box { <-30*KM,-30*KM,-30*KM>,<30*KM,30*KM,-36*Feet> }
/*
		box { // local area
			<-250*Feet,0,-40*Feet>,<250*Feet,900*Feet,1>
		}
	} //end difference
*/

#declare PhotonWaterBlock =
	box {
		<20*Feet,-8*Feet,-5>,<30*Feet,8*Feet,0.1>
/*
		texture {
			pigment { color rgbt<0,0,0,1> }
			normal { bozo 0.2 scale 8 }
			finish { ambient 0 diffuse 0 reflection 1.85 }
		}
*/
			texture { SeaWaterTex } // test
//			texture { pigment { color rgb<1,1,.2> } }
//		no_image
//		no_shadow
		photons {
			target
			refraction on
			reflection on
			collect off
			#debug concat("Photons on in PhotonWaterBlock","\n")
		}
		rotate <0,0,180+80> translate <-20*Feet,60*Feet,0> // match Hydroplane2
	}

#declare SeaWaterSurf =
	height_field { // starts extruded into +Y, with image upper left corner at <0,y,1>
		png "SeaSurfaceHF01.png"
		smooth
		// water_level 1/255
		rotate x*-90
		scale <1,1,-1> // extrude into +z
//		translate <-0.5,-0.5,-0.1*(1/3.5*Feet)>
//		scale <200*Feet,200*Feet,3.5*Feet>
		translate <-0.5,-0.5,-0.1*(1/2.1*Feet)>
		scale <200*Feet,200*Feet,2.1*Feet>
//		texture { TestWhiteTex } // test
		texture { SeaWaterTex }
	}

#declare SeaWaterGroup =
	union {

		box { <-30*KM,200*Feet,-100*Feet>,<30*KM,30*KM,-12> // distant water
			texture { SeaWaterTex }
//			texture { TestWhiteTex } // test
		}
		box { <-30*KM,100*Feet,-100*Feet>,<60*Feet,200*Feet,-12> texture { SeaWaterTex }} // left near
		box { <-30*KM,-30*KM,-100*Feet>,<0,-100*Feet,-12> texture { SeaWaterTex }} // behind camera

		object { SeaWaterSurf }
		object { SeaWaterSurf translate <140*Feet,100*Feet,0> }
//		object { cylinder { <0,0,0>,<0,0,100*Feet>,24 texture{pigment{color rgb<1,1,.2>}} translate <100*Feet,200*Feet,> }}
//		object { cylinder { <0,0,0>,<0,0,100*Feet>,24 texture{pigment{color rgb<.1,.1,1.2>}} translate <0*Feet,100*Feet,> }}
//		object { cylinder { <0,0,0>,<0,0,40*Feet>,24 texture{pigment{color rgb<1,1,1>}finish{ambient 0.4}} translate <50*Feet,80*Feet,> }}
//		object { cylinder { <0,0,0>,<0,0,40*Feet>,24 texture{pigment{color rgb<1,1,1>}finish{ambient 0.4}} translate <70*Feet,80*Feet,> }}
//		object { PhotonWaterBlock }
//		#debug concat("PhotonWaterBlock is on","\n")
		object { DistantSeabed 
			texture { SeaFloorTex }
		}
	} // end union


//===============================================
//             Object Groups
//===============================================



//===============================================
//             Object Placement
//===============================================

light_group {
#if (AreaSun =1)
	object { FillLight }
#end

#if (Boats = 1)
//object { Hydroplane01 rotate x*125 rotate y*-20 rotate z*(180+90)  translate <6*Feet,-6*Feet,12*Feet>}
object { Hydroplane01 rotate x*125 rotate y*-20 rotate z*(180+90)  translate <6*Feet,-6*Feet,15*Feet>}
#end // boats

global_lights on
} // light group

// Following items only get illuminated by the Sun
object { Backdrop translate <0,360*Feet,-54*Feet> rotate z*-12 translate CamPos }
object { Backdrop scale 1.3 translate <0,440*Feet,-70*Feet> rotate z*-12 translate CamPos }
object { SeaWaterGroup }
object { FarCoastline rotate z*-50 translate <-0.6*Miles,0.4*Miles,-20> }
object { FarCoastline rotate z*80 translate <0.2*Miles,0.7*Miles,-20> }
object { FarCoastline scale <1,1,0.6> rotate z*140 translate <0.34*Miles,0.15*Miles,-20> } // left side

#if (Boats = 1)
//object { Hydroplane02 rotate <2,-2,(180+80)> translate <-20*Feet,60*Feet,10>}
object { Hydroplane02 rotate <2,-2,(180+75)> translate <-19*Feet,50*Feet,10>}
//object { Hydroplane03 rotate <-5,0,(180+55)> translate <60*Feet,120*Feet,10>}
object { Hydroplane03 rotate <0,0,(180+45)> translate <60*Feet,120*Feet,2>}
#end // boats

#if (EjectionSeat=1)
light_group {
	object { EjectionSeatPilot(RocketQual) rotate <18,-18,-120> translate <-14*Feet,-10*Feet,30*Feet> }
	object { SeatLight }
	global_lights on
} // light group
	object { EjectionSeatSmoke(RocketQual) rotate <18,-18,-120> translate <-14*Feet,-10*Feet,30*Feet> }
#end // ejection seat

object { CanopyS rotate <120,0,120> translate <-10*Feet,8*Feet,19*Feet> }
object { DebrisField translate <-10*Feet,8*Feet,12*Feet> }
object { FloatingDebris rotate z*-12 rotate <5,-80+4*0,2*0> translate <3.9*Feet,-6.8*Feet,36> } // motion blur
object { FloatingDebris rotate z*-12 rotate <5,-80+4*1,2*1> translate <3.9*Feet,-6.8*Feet,36> } // motion blur
object { FloatingDebris rotate z*-12 rotate <5,-80+4*2,2*2> translate <3.9*Feet,-6.8*Feet,36> } // motion blur
object { FloatingDebris rotate z*-12 rotate <5,-80+4*3,2*3> translate <3.9*Feet,-6.8*Feet,36> } // motion blur
object { FloatingDebris rotate z*-12 rotate <5,-80+4*4,2*4> translate <3.9*Feet,-6.8*Feet,36> } // motion blur

object { DualHardware rotate <14,-67,209> translate <-0.5*Feet,-4*Feet,26> }
object { Fishy rotate <0,-25,40> translate <-13*Feet,-54*Feet,0> }
object { SplashGroup rotate <0,0,-90> translate <-11*Feet,-52*Feet,0> } 

#if (RoosterQual = 1) // use textured spheres
// FadeRoosterTail(Zrot,TurnRad,TotNum,TTime,Velocity,RandomSeed)
object { FadeRoosterTail(-10,95,200,2.5,58,30495) rotate z*90 translate <0*Feet,0*Feet,0>} // Boat 1
//		object { cylinder { <0,0,0>,<0,0,100*Feet>,24 texture{pigment{color rgb<1,1,.2>}} translate <80*Feet,200*Feet,> }}
//#macro FadeSphere2(Elong,Xrot,Zrot,Yrot,Dens)
object { FadeSphere2(1,0,75,0,.65) scale <100,100,150> translate <90*Feet,200*Feet,8*Feet> } // Boat 1 filler
object { FadeRoosterTail(15,120,200,2.5,38,20394) rotate z*80 translate <-20*Feet,60*Feet,0>} // Boat 2
object { FadeRoosterTail(45,75,200,2.5,48,51056) rotate z*55 translate <60*Feet,120*Feet,0>} // Boat 3

// SideBlast(Xrot,Zrot,Yrot,TotNum,Elong,TTime,RandomSeed)
object { SideBlast(0,-0, 0,12,1.8,2.2,20936) rotate z*90 translate <-4*Feet,-12*Feet,0>} // Boat 1 left
//object { SideBlast(00,-70,0,10,1.8,1.7,50892) rotate <0,-50,170> translate <0*Feet,12*Feet,0>} // Boat 1 plume
object { SideBlast(0,0,-20,12,1.1,2.5,77340) rotate <0,20,95> translate <2.8*Feet,-11*Feet,40>} // Boat 1 right

object { SideBlast(0, 0, 0,7,1.0,1.0,90212) rotate z*90 translate <-6*Feet,-24*Feet,18>} // Boat 1 front impact
object { SideBlast(0,-20,-30,16,1.0,1.2,31354) rotate <0,30,110>  translate <-2*Feet,-18*Feet,48>} // Boat 1 front impact
object { DualHardware rotate <-82,102,-229> translate <-2*Feet,-18*Feet,48>} // Boat 1 front impact

//SideBlastLight(Xrot,Zrot,Yrot,TotNum,Elong,TTime,RandomSeed,Dens) // 0=opaque, 1=transparent
//object { SideBlastLight(0,-30,-30,33,1.0,1.2,13201,0.9987)
object { SideBlastLight(0,-30,-30,33,1.0,1.2,75012,0.9987)
	rotate z*180 translate <19.4*Feet,8.6*Feet,0>
	rotate x*125 rotate y*-20 rotate z*(180+90) translate <11*Feet,-6*Feet,15*Feet>} // Boat 1 midair L

//SideBlastLight(Xrot,Zrot,Yrot,TotNum,Elong,TTime,RandomSeed,Dens) // 0=opaque, 1=transparent
object { SideBlastLight(0,-0,30,26,1.0,1.2,90956,0.9987)
	rotate z*175 translate <11*Feet,-1.3*Feet,-14>
	rotate x*125 rotate y*-20 rotate z*(180+90) translate <11*Feet,-6*Feet,15*Feet>} // Boat 1 midair R

object { SideBlast(0,-20, 0,7,1.0,1.5,12439) rotate <0,-10,110> translate <-29*Feet,35*Feet,0>} // Boat 2 far left
object { SideBlast(0,-20, 0,7,1.0,2.2,50943) rotate <0,-10,90> translate <-29*Feet,38*Feet,0>} // Boat 2 less far left
object { SideBlast(0, 50, 0,7,1.6,0.6,44398) rotate <0,-10,30> translate <-26*Feet,47*Feet,0>} // Boat 2 underneath
object { SideBlast(0, 50, 0,5,1.6,0.6,11209) rotate <0,-10,30> translate <-24*Feet,46*Feet,0>} // Boat 2 underneath
object { SideBlast(0, 10, 0,13,1.0,3.2,22821) rotate <0,-20,80> translate <-27*Feet,50*Feet,0>} // Boat 2 left plume
object { SideBlast(0, 45, 0,12,1.8,2.5,44235) rotate z*45 translate <-16*Feet,60*Feet,0>} // Boat 2 right

object { SideBlast(0, 40, 0,12,2.2,2.5,20395) rotate z*50 translate <57*Feet,103*Feet,0>} // Boat 3 right
#end // Roostertails





