
// POV-Ray Object File - Rooster Tail
// created 15-Nov-05
// last update 27-Dec-05
// 2005 Glenn McCarter  ( glenn@mccarters.net )
//
// Required #include files: "GIncludeRH.inc"
//
// Objects:
// RoosterTail(Qual,TurnRad,TotNum,TTime,Velocity,RandomSeed) // media
// FadeRoosterTail(Zrot,TurnRad,TotNum,TTime,Velocity,RandomSeed) // fading spheres
// FakeRoosterTail(TurnRad,TotNum,TTime,Velocity,RandomSeed) // simple spheres
// SideBlast(Zrot,Yrot,Xrot,TotNum,Elong,TTime,RandomSeed)
// HydroplaneS // a few blocks to simulate hydroplane boat
//
// Intended scale is English system (inches)
// Uses RH coordinate reference system (Z=up)
// RoosterTail origin is at the start of the tail,
// with the height of the plume in +Z,
// and the length of the plume in +X
//

#ifndef (ThisIsMaster) // If this include file was not called by another, then
	#declare RenderMe = 0; // supress self-rendering of other include files
	#declare ThisIsMaster = 1;
	#include "GIncludeRH.inc"
	// renderable sub includes go here
	#declare RenderMe = 1;
#end

//===============================================
//             Includes
//===============================================

// non-renderable sub includes go here

//===============================================
//             Variables
//===============================================


//===============================================
//             Textures
//===============================================

#declare FakePlumeTex =
	texture {
		pigment { color rgb<.967,.977,.996>*1.0 }
		normal { granite 0.35 scale 5 }
		finish { specular 0.3 roughness 0.24 diffuse 0.4 ambient 0.6 }
	}
#declare FakePlumeTex2 =
	texture {
		pigment { color rgb<.067,.377,.996>*1.0 }
		normal { granite 0.35 scale 5 }
		finish { specular 0.3 roughness 0.24 diffuse 0.4 ambient 0.6 }
	}

#declare FakeWaterTex = // used in FakeRoosterTail()
	texture {
		pigment { color rgb<.967,.977,.996>*1.0 }
		normal { granite 0.35 scale 5 }
		finish { specular 0.3 roughness 0.24 diffuse 0.4 ambient 0.6 }
	}


//===============================================
//             Media-Based Rooster Tail
//===============================================

#macro SpraySphere(Center,SRad)
	sphere {
		Center,SRad
	}		
#end

#macro VaporInt(RandomSeed,Time,OverallDensity,Darkness)
	#local OD = OverallDensity;
	#local RS = seed(RandomSeed);
	interior {
		media {
//emission rgb<0.0075,0.0075,0.0075>
//absorption rgb<0.75,0.75,0.75>
//			scattering { 1, <.01,.01,.01> extinction 3.0 } // good density, but totally grey-black
//			scattering { 1, <.01,.01,.01> extinction 1.0 } // just a bit too full
//			scattering { 1, <.01,.01,.01> extinction 1.0 } // just a bit too full
			scattering { 1, 0.01/(Time*5) extinction 1.0 } // just a bit too full
//			scattering { 1, <.002,.002,.002> extinction 1.0 } // too see-thru except where several overlaps occur
//			scattering { 1, <.0002,.0002,.0002> extinction 1.0 } // much too diffuse

	    density { spherical
				color_map{
					[0.0 color rgb <0.0,0.0,0.0> ] // outside is transparent
//					[0.2 color rgb <0.15,0.15,0.15> ] // test
					[0.14 color rgb <0,0,0> ] // this prevents it from reaching the container
					[0.42 color rgb <1,1,1> ] // test
					[1.0 color rgb <1,1,1> ] // inside is opaque
				}
/*
				warp {
					turbulence .18
					octaves 6 // default 6.0, use 1-10
					lambda 1.5 // default 2.0
					omega 0.835 // default 0.5, higher = rougher
				}
*/
    	}
		}
	}
#end

#declare VaporSurfTex =
	texture {
//		pigment { color rgbf<1,1,1,.1> } // test
		pigment { color rgbf<1,1,1,1> }
		finish { ambient .1 diffuse 0.7 }
	}

#macro VaporMaterial(RandomSeed,OverallDensity,Darkness)
	material {
		texture { VaporSurfTex }
//		texture { FakePlumeTex2 } // test
		VaporInt(RandomSeed,Time,OverallDensity,Darkness)
	}
#end

// Rooster Tail Macro
//   variables
// Qual=1 for media-based, 0 for simple spheres
// TotNum = total number of spheres in the tail
// TurnRad = radius of plan-view curvature of boat path (meters)
// TTime = max time to calculate (seconds) (2.5 is good)
//     decrease to avoid having too much tail underwater
//     increase if tail ends before reaching water level
// Velocity = speed of the boat (m/s)
// RandomSeed makes this tail different than the next
// RoosterTail is in XZ plane, with height in +Z and length in +X
// Boat makes left turns only!

#macro RoosterTail(Qual,TurnRad,TotNum,TTime,Velocity,RandomSeed)
	#declare RS = seed(RandomSeed);
	#declare OverallDensity = .13;
	#declare Darkness = 0.8;	
	union {
		#declare SphereNum = 1;
		#while (SphereNum<TotNum)
			#local Time = rand(RS)*TTime; // the instant for this sphere
			#local UpVel = (0.3+rand(RS)*.7)*Velocity*0.32; // upwards velocity
			// Max upwards plume velocity is some fraction boat speed; 0.32 factor corrects for this
			sphere { <0,0,0>,1
				#if (Qual=1)
					hollow
					VaporMaterial(RandomSeed,OverallDensity,Darkness)
//					texture { VaporSurfTex } // test
					scale 15+Time*20+Time*Time*15 // mist spheres grow over time
				#else
		 			texture { FakePlumeTex }
					scale 15+Time*20+Time*Time*15 // mist spheres grow over time
				#end

				scale <1,1,6/(Time+1)> // tall spurt at first, later flattened out
//				translate <Time*UpVel*39.4,Time*Time*Time*(-100),(UpVel*Time-9.8*Time*Time)*39.4>
				// 39.4 is conversion factor to scene units (inches)

//				translate <0, TurnRad*39.4+(UpVel*Time*20),(UpVel*Time-9.8*Time*Time)*39.4>
//				translate <0, TurnRad*39.4,(UpVel*Time-9.8*Time*Time)*39.4>
//				translate <0, TurnRad*39.4+rand(RS)*Time*120,(UpVel*Time-9.8*Time*Time)*39.4>
				translate <
						0,
						TurnRad*39.4-rand(RS)*Time*150+(UpVel*Time-9.8*Time*Time)*9,
						(UpVel*Time-9.8*Time*Time)*39.4
						>
				rotate <0,40,-1000*Time/TurnRad>
				translate <0,-TurnRad*39.4,0>
			}
			#declare SphereNum = SphereNum+1;
		#end
	} // end union
#end


//===============================================
//             Sphere-Based Rooster Tail
//===============================================


#macro FadeSphere(Zrot)
	sphere {
		<0,0,0>,1

		texture {
			pigment {
				spherical
				color_map {
//					[ 0.0 rgbf<1,1,1,0> ]
					[ 0.2 rgbf<1,1,1,1> ] // edge is clear
					[ 1.0 rgbf<1,1,1,0.14> ] // center is dense
				}
//				turbulence 0.3
				scale <10,1,1>
			}
//			finish { ambient 0.4 diffuse 0.6 }
			finish { ambient 0.4 diffuse 0.6 specular 0.03 roughness .14 }
		}

		rotate z*Zrot
//	texture { FakeWaterTex } // test
	}
#end


// Fade Rooster Tail Macro
// Variables
// Zrot = angle of view to the camera
//        0 = best viewed from -X direction
// TotNum = total number of spheres in the tail
// TurnRad = radius of plan-view curvature of boat path (meters)
// TTime = max time to calculate (seconds) (2.5 is good)
//     decrease to avoid having too much tail underwater
//     increase if tail ends before reaching water level
// Velocity = speed of the boat (m/s)
// RandomSeed makes this tail different than the next
// RoosterTail is in XZ plane, with height in +Z and length in +X
// Boat makes left turns only!

#macro FadeRoosterTail(Zrot,TurnRad,TotNum,TTime,Velocity,RandomSeed)
	#local RS = seed(RandomSeed);
	union {
		#declare SphereNum = 1;
		#while (SphereNum<TotNum)
			#local Time = rand(RS)*TTime; // the instant for this sphere
			#local UpVel = (0.3+rand(RS)*.7)*Velocity*0.3; // upwards velocity
			// Max upwards velocity is some fraction boat speed; 0.3 factor corrects for this
			#local ThisSpherePos =
				<
					0,
					TurnRad*39.4-rand(RS)*Time*150+(UpVel*Time-9.8*Time*Time)*9,
					(UpVel*Time-9.8*Time*Time)*39.4
				>;
			#local ThisSphereRad =(15+Time*20+Time*Time*15);
			#if (ThisSpherePos.z+ThisSphereRad>0) // render spheres that are visible
				object { FadeSphere(Zrot)
					scale ThisSphereRad
					scale <1,1,1.3+(3/(Time+1))>
					translate ThisSpherePos
					rotate -z*2000*Time/TurnRad
					translate <0,-TurnRad*39.4,0>
				}
			#else // don't render spheres if they are totally underwater
				#declare SphereNum = SphereNum-1; // (and don't count them either)
			#end
			#declare SphereNum = SphereNum+1;
		#end
//		texture { FakeWaterTex }
	} // end union
#end

#macro FadeSphere2(Elong,Xrot,Zrot,Yrot,Dens)
	sphere {
		<0,0,0>,1
		texture {
			pigment {
				spherical
				color_map {
					[ 0.2 rgbt<1,1,1,1> ] // edge is clear
					[ 1.0 rgbt<1,1,1,Dens> ] // center is variable density
				}
//				turbulence 0.3
				scale <10,1,1>
			}
//			finish { ambient 0.4 diffuse 0.6 }
			finish { ambient 0.4 diffuse 0.6 specular 0.04 roughness .14 }
		}
		scale <1,Elong,1>
		rotate <Xrot,0,Zrot>
		rotate <0,Yrot,0>
//	texture { FakeWaterTex } // test
	}
#end

// SideBlast macro
// Elong = X-elongation ratio of each sphere
// SideBlast length in +X
#macro SideBlast(Xrot,Zrot,Yrot,TotNum,Elong,TTime,RandomSeed)
	#local RS = seed(RandomSeed);
	union {
		#declare SphereNum = 1;
		#while (SphereNum<TotNum)
			#local Time = rand(RS)*TTime; // the instant for this sphere
			#declare Dens = 0.75*Time/TTime; // max density of this sphere, 0=opaque, 1=transparent
			object { FadeSphere2(Elong,Xrot,Zrot,Yrot,Dens)
				#local ThisSphereRad =(6+Time*18);//+Time*Time*5);
				scale ThisSphereRad
				#local ThisSpherePos =
					<
						Time*100,
						rand(RS)*Time/TTime*10,
						ThisSphereRad/2+rand(RS)*Time/TTime*10 // unaffected by gravity
					>;
					translate ThisSpherePos

			}
			#declare SphereNum = SphereNum+1;
		#end
//		texture { FakeWaterTex }
	} // end union
#end

#macro SideBlastLight(Xrot,Zrot,Yrot,TotNum,Elong,TTime,RandomSeed,Dens) // 0=opaque, 1=transparent
	#local RS = seed(RandomSeed);
	union {
		#declare SphereNum = 1;
		#while (SphereNum<TotNum)
			#local Time = rand(RS)*TTime; // the instant for this sphere
//			#declare Dens = 0.75*Time/TTime; // max density of this sphere, 0=opaque, 1=transparent
			object { FadeSphere2(Elong,Xrot,Zrot,Yrot,Dens)
				#local ThisSphereRad =(6+Time*18);//+Time*Time*5);
				scale ThisSphereRad
				#local ThisSpherePos =
					<
						Time*100,
						rand(RS)*Time/TTime*10,
						ThisSphereRad/2+rand(RS)*Time/TTime*10 // unaffected by gravity
					>;
					translate ThisSpherePos

			}
			#declare SphereNum = SphereNum+1;
		#end
//		texture { FakeWaterTex }
	} // end union
#end

// Fake Rooster Tail Macro
// Variables
// TotNum = total number of spheres in the tail
// TurnRad = radius of plan-view curvature of boat path (meters)
// TTime = max time to calculate (seconds) (2.5 is good)
//     decrease to avoid having too much tail underwater
//     increase if tail ends before reaching water level
// Velocity = speed of the boat (m/s)
// RandomSeed makes this tail different than the next

#macro FakeRoosterTail(TurnRad,TotNum,TTime,Velocity,RandomSeed)
// RoosterTail is in XZ plane, with height in +Z and length in +X
// Boat makes left turns only!
//	#local RS = seed(20951);
	#local RS = seed(RandomSeed);
	union {
		#declare SphereNum = 1;
		#while (SphereNum<TotNum)
			#local Time = rand(RS)*TTime; // the instant for this sphere
			#local UpVel = (0.3+rand(RS)*.7)*Velocity*0.3; // upwards velocity
			// Max upwards velocity is some fraction boat speed; 0.4 factor corrects for this
			sphere { <0,0,0>,15+Time*20+Time*Time*15
				scale <1,1,3/(Time+1)>
//				translate <Time*UpVel*39.4,Time*Time*Time*(-100),(UpVel*Time-9.8*Time*Time)*39.4>
				// 39.4 is conversion factor to scene units (inches)

//				translate <0, TurnRad*39.4+(UpVel*Time*20),(UpVel*Time-9.8*Time*Time)*39.4>
//				translate <0, TurnRad*39.4,(UpVel*Time-9.8*Time*Time)*39.4>
//				translate <0, TurnRad*39.4+rand(RS)*Time*120,(UpVel*Time-9.8*Time*Time)*39.4>
				translate <
						0,
						TurnRad*39.4-rand(RS)*Time*150+(UpVel*Time-9.8*Time*Time)*9,
						(UpVel*Time-9.8*Time*Time)*39.4>

				rotate -z*1000*Time/TurnRad
				translate <0,-TurnRad*39.4,0>

			}
			#declare SphereNum = SphereNum+1;
		#end
		texture { FakeWaterTex }
	} // end union
#end

//===============================================
//             Boat
//===============================================

#declare HydroplaneS = // for testing
	union {
		box { <0,-20,-8>,<18*Feet,20,42> } // engine
		box { <30,-70,6>,<18*Feet,70,20> } // hull
		box { <12*Feet, 16,-8>,<26*Feet, 90,24> } // pontoon
		box { <12*Feet,-16,-8>,<26*Feet,-90,24> } // pontoon
		texture { pigment { rgb<1,.9,0.4> } }
	} // end union

//===============================================
//             Viewpoint 01
//===============================================

#if (RenderMe = 1)
global_settings {	max_trace_level 150 }

#declare TestSphere = sphere{ 0,12 texture { FakePlumeTex } }
#declare CamPos = <-2*Feet, -40*Feet, 5*Feet>;

	camera {
		perspective
		direction <0.0,     1.0,  0.0>
		sky       <0.0,     0.0,  1.0>  // Use right handed-system!
		up        <0.0,     0.0,  1.0>  // Where Z is up
		right     <1.33333, 0.0,  0.0>

		angle 34
		location  CamPos // overall
		look_at <2*Feet,0*Feet,3*Feet>
//		location  <23*Feet, -224*Feet, 5235*Feet> // top down
//		look_at <30*Feet,20*Feet,18*Feet>
//		location  <3, -10*Feet, 3*Feet> // detail
//		look_at <0,0,20>
//		location  <3, -24*Feet, 5*Feet> // main wing
//		look_at <0,0,14*Feet>
//		location  <23*Feet,-5*Feet, 19*Feet> // upper RW
//		look_at <21*Feet,8*Feet,14*Feet>
//		location  <23*Feet,-25*Feet, 5*Feet> // lower RW
//		look_at <21*Feet,8*Feet,7*Feet>
//		location  <20*Feet, -20*Feet, 1*Feet> // down up view
//		look_at <10*Feet,0,4*Feet>

	}

background { color rgb <0.276,0.567,0.982>*.97 } // sky blue

light_source
{ <140*Feet,-300*Feet,250*Feet>, 
		color rgb <0.584,0.539,0.543>*2.3 // bluish
//  spotlight point_at <0,0*Feet,2*Feet> radius 25 falloff 36
}
/*
light_source
{ <-140*Feet,-200*Feet,130*Feet>, <1.13,1.08,0.83>*.08
//  spotlight point_at <0,0,2*Feet> radius 25 falloff 36
}
*/
//plane { z,-0 pigment { rgb<.8,.8,1.0>*.75 } normal { bumps 0.3 scale .7*Feet} finish { reflection 0.14 } }
plane { z,-0 pigment { rgb<.8,.8,1.0>*.75 } normal { bumps 0.3 scale .7*Feet} finish { specular 0.4 } }

#include "GRuler.inc" // for testing only
//object { TestSphere scale 1 translate <0,0,10> }

/*
object { FadeSphere scale 20 rotate z*0 translate <0,0,20> }
object { FadeSphere scale 10 rotate z*30 translate <30,0,20> }
object { FadeSphere scale 10 rotate z*-30 translate <-30,0,20> }
*/

// FadeRoosterTail(Zrot,TurnRad,TotNum,TTime,Velocity,RandomSeed)
#declare F2Loc =<-50*Feet,200*Feet,0>;
//#declare F2Vec = vnormalize(CamPos-F2Loc);
//object { FadeRoosterTail(90,100,200,2.5,50,30495) scale 1.0 rotate z*0 translate F2Loc }

#declare F2aLoc =<-28*Feet,-4*Feet,0>; // auxiliary
//#declare F2Vec = vnormalize(CamPos-F2Loc);
//object { FadeRoosterTail(-30,100,50,1.5,30,10125) scale 1.0 rotate z*130 translate F2aLoc }

#declare F3Loc =<0*Feet,0*Feet,0>;
//#declare F3Vec = CamPos-F3Loc;
//object { FadeRoosterTail(90,100,200,2.5,50,30495) scale 1.0 rotate z*0 translate F3Loc }

//object { RoosterTail(1,100,200,2.5,50,30495) scale 1.0 rotate z*90 translate <-20*Feet,0*Feet,0>}
//object { RoosterTail(1,100,200,2.5,50,30495) scale 1.0 rotate z*0 translate <10*Feet,0*Feet,0>}

object { HydroplaneS rotate z*180 translate <20*Feet,14*Feet,0> }
// SideBlast(Xrot,Zrot,Yrot,TotNum,Elong,TTime,RandomSeed)
object { SideBlast(0,90,0,22,2,2.2,78541) scale 0.5 translate <0,0,1*Feet>}
//object { SideBlast(20,90,0,,4,1.0,2.2,78541) rotate <0,-70,70> scale 0.5 translate <20,0,5*Feet>} // upblast
object { SideBlast(90,20,70,4,2.0,2.2,78541) rotate <0,-70,70> scale 0.5 translate <-50,0,5*Feet>} // upblast
object { SideBlast(0,30,0,14,2,2.2,78541) rotate z*60 scale 0.5 translate <0,0,3*Feet>}

//object { SideBlast(90,10,1,2.2,15634) }
//object { SideBlast(45,10,1,2.2,49838) rotate z*45 translate <8*Feet,0,0>}
/*
object { HydroplaneS rotate x*-5 rotate z*(180+90)  translate <0*Feet,0*Feet,0>}
object { RoosterTail(1,195,200,2.5,58,30495) rotate z*90 translate <0*Feet,0*Feet,0>}

object { HydroplaneS rotate z*(180+80) translate <-20*Feet,60*Feet,0>}
object { RoosterTail(1,120,200,2.5,38,20394) rotate z*80 translate <-20*Feet,60*Feet,0>}

object { HydroplaneS rotate z*(180+45) translate <80*Feet,160*Feet,0>}
object { RoosterTail(1,75,200,2.5,48,51056) rotate z*45 translate <80*Feet,160*Feet,0>}
*/
#end

