//command line: +w720 +h1080

/*******************************
*           Volcano            *
*      by Tekno Frannansa      *
*    www.evilsuperbrain.com    *
********************************/


//final toggles, all on = final scene
#declare toggle_atmos = on;
#declare toggle_background = on;
#declare toggle_pillowlava = on;
#declare toggle_seafloor = on;
#declare toggle_sea = on;
#declare toggle_undersea_steam = on;
#declare toggle_volcano = on;
#declare toggle_lava_rivers = on;
#declare toggle_plume = on;
#declare toggle_smoke = on;
#declare toggle_steam = on;
#declare toggle_lensfx = on;

//placeholder toggles, all off = final scene
#declare placeholder_volcano = off;
#declare placeholder_seafloor = off;
#declare cheaper_smoke = off; //ugliness factor go!
#declare cheapest_smoke = off;
#declare cheaper_sea = off;

//some other toggles
#declare preview_gamma = on; //<-dammit I screwed up, this changes how it looks, leave it on!
#declare save_lava_flows = off; //you need to run with this turned on at least first time


#version unofficial MegaPov 1.21;


#include "functions.inc"
#include "rad_def.inc"
#include "transforms.inc"
#include "pprocess.inc"
PP_Init_Alpha_Colors_Outputs()

#include "lens-fx.inc" //my own little box of tricks


//-----------------------------------global-stuff--------------------------------------
//lighting, cameras, etc.


global_settings {
	max_trace_level 20 //lower value fixed a red line at the water level, not sure why.
	//max_trace_level 80 //much higher than anything needs, doesn't really slow it down much apart from the water surface.

	//use some megapov post processing functions I wrote to fake some subtle lens effects
	#declare gamma_function1 =
		//kind of like film exposure but funkier
		//quite stylised but does a nice thing to brighter than 1.0 colours that would normally saturate out badly
		#local A=.1;
		#local B=.8;
		function (x) {
			1 - A/(pow(x,B)+A)
		}

	#if (preview_gamma)
		assumed_gamma 1
		#declare gamma_function = function(x) { gamma_function1(pow(x,2.2)) } //undo the assumed_gamma (which was used to make sure it looked okay *during* rendering)
	#else
		#declare gamma_function = function(x) { gamma_function1(x) }
	#end

	#declare lens_vignetting = 1;
	#declare lens_dispersion = -0.02;	
	#if (toggle_lensfx)
		lens_fx()
	#end
}


//note: no radiosity but there's enough lights that we can get away with 0 ambient (which looks more real anyway, objects don't glow)
#default { pigment { rgb 1 } finish { diffuse 1 ambient 0 } }


//build scene in-camera, with camera at the origin
//we're using a quite unrealistic composition (seeing both above and below water, as if camera was behind glass)
//so don't really need something that holds up from all angles.
//-note, in retrospect this scene is actually pretty good, there's no 2D things pretending to be 3D, and most of it works from any angle.

camera {
	right	x
	up		y*image_height/image_width
	
	direction z*.7*3/2 look_at <0,1,5> //default
//direction z*50 look_at z-x/5
	//direction z*.7 location y*500 look_at <0,0,1000> //higher (volcano form study)
	//direction z*3 location <-3000,600,1500> look_at <0,1000,2000> //side view (from left)
	//direction 8*z*3/2 location 40000*<1,1,-1> look_at 0	//isometric view
	//direction z*2*3/2 look_at <0,4,7> //fire zoom
	//direction z*1 location -4000*z+1000*y //far off, good for cloud placement
	//direction z*1.3 look_at <0,-1,6> //undersea framed for 4:3
	//direction z*1.3 location <0,200,0> look_at <0,0,50> //undersea overhead
}

//box the camera's in, differenced against anything the camera wants to see a cross-section of e.g. the ocean surface.
#declare camera_box =
	//box { -1, 1 }
	sphere { 0, 1 } //no distortion due to refraction, cunning

/*//partial render
sphere {
	0, .01
	clipped_by {
		intersection {
			plane { y, 0 rotate -x*2 }
			plane { -y, 0 rotate x*2 }
			inverse
		}
	}
	hollow on
	no_shadow
	no_reflection
	pigment { rgb 0 }
}*/


//sun
#declare light_dir = <-4,2,0>;
light_source { light_dir*10000, rgb <2,2,1.9> parallel point_at 0 } //almost pure white, winter lighting

//fake rad
//light_source { <2,1,-3>*10000, rgb .2 parallel point_at 0 shadowless } - actually looks better without it.
 
//fire light
light_source {
	<0,900,2000>,
	rgb <4,.5,0>
	fade_power 2 fade_distance 30
}



//-------------------------------------background-------------------------------------
#if (toggle_atmos)
	background { rgb 0 }

	//all sky colour comes from fog	
	#declare Fog_Offset = 0;
	#declare Fog_Alt = 2500;
	#declare Fog_Distance = 100000;
	#declare Fog_Colour = <1,2,4>;
	#if (0)
		fog {
			fog_type 2
			rgb Fog_Colour
			distance Fog_Distance
			fog_alt Fog_Alt
			fog_offset Fog_Offset
		}
	#else
		//damn fog doesn't play nicely with the media for the smoke :(
		//can't find an exact match but this looks pretty good.
		#declare media_fog =
			media {
				emission Fog_Colour
				absorption 1//vlength(Fog_Colour)
				density {
					function {
						1/pow(1 + max(0,y - Fog_Offset) / Fog_Alt, 2)
					}
					density_map {
						[0 rgb 0]
						[1 rgb 1/Fog_Distance]
					}
				}
			}
	#end
	

	//high-altitude clouds, cheat by just sticking the texture in 2 different colours on 2 planes, to give appearance of depth with no expensive media effects.	
	#macro wispy_cloud(X)
		//wispy clouds, high altitude wintery stuff
		pigment {
			pigment_pattern { spotted poly_wave 3 }
			turbulence 1 lambda 3
			translate 7
			scale 3000
			cubic_wave
			colour_map {
				[0 rgb X transmit 1]
				[1 rgb X]
			}
		}
		finish {
			ambient 1 diffuse 0
		}
	#end
	
	plane { -y, -6000 wispy_cloud(1.5) hollow on }
	plane { -y, -5700 wispy_cloud(.5) hollow on }
	
#else
	//debug navy blue BG if atmos is off
	background { rgb <.02,.1,.2> }
#end
	
#if (toggle_background)
	//mountain range
	#declare f_mountain =
		function(x,y) {
			(.6 + .4*f_wrinkles(x*3,y*3,0))
			*
			(1-sqrt(x*x+y*y+.01))
		}

	#declare mountain =
		height_field {
			function 1024, 1024 {
				min(1,max(0,
					f_mountain(x*2-1,y*2-1)
				))
			}
			translate <-.5,0,-.5>
			scale <2,1,2>
			scale <1.5,1.2,1.5>*2000
		}
		
	
	union {
		object { mountain rotate y*200 scale 1.1 translate <-1900,0,5500> }
		object { mountain rotate y*90  scale .6  translate <-2200,0,3000> }
		object { mountain rotate y*270 scale 1.2 translate <2400,0,6000>  }

		texture {
			pigment_pattern {
				slope {
					y-x*.3, .5, 1
				}
				pigment_map {
					[0 rgb 0]
					[1 function { min(1,max(0,y)) } turbulence .8 scale 1200]
				}
			} 
			texture_map {
				[.4 //rock
					pigment { rgb <.12,.1,.09> }
				]
				[.4
					//snow
					pigment { rgb 1 }
					finish {
						brilliance 1 diffuse 1
						phong 1 phong_size 200
						reflection { .0, .7 falloff 5 }
					}
					normal {
						spotted 2 scale .05
					}
				]
			}
		}		
	}
#end


//--------------------------------------undersea--------------------------------------
#if (placeholder_seafloor)
	cone { z*2000-y*10, 2000, z*2000+y*50, 0 pigment { rgb .2 } } //beach & undersea beach
#end

#if (toggle_seafloor)
	union {
		cone { z*2000-y*10, 2000, z*2000+y*50, 0 } //beach & undersea beach
		plane { y, -10 } //this is out of shot
		
		pigment {
			function {
				min(1,max(0,y))
			}
			scale 1/300 warp { turbulence .1 lambda 4 } scale 300
			colour_map {
				[0.01 rgb .02] //underwater
				[0.01 rgb 1] //snow
			}
		}
		finish {
			specular .3 roughness 1/10
		}
		normal {
			granite scale .1
		}
	}

	//nearby sea floor
	//loose rocks on rock being "pushed" by the lava flow
	isosurface {
		function {
			y+1*f_wrinkles(x,y,z)
		}
		max_gradient 1.2
		contained_by { box { <-20,-1,15>, <20,0,70> } }
		scale <1,.7,1>
		translate -7.8*y
		
		pigment {
			spotted
			colour_map {
				[0 rgb .1]
				[1 rgb .3]
			}
		}
		finish { brilliance 2 diffuse 1.5 }
	}
#end


#if (toggle_pillowlava)
	//reminder to me, LOOK AT THE PHOTO REFERENCE!!! IT'S PERFECT!
	//video linked at the bottom of this page: http://www.pmel.noaa.gov/vents/nemo/explorer/concepts/pillows.html

	#local turb_scale = 7;

	#local cold_lava_pattern =
		pigment { //greyscale pigment to control normal and pigment
			pigment_pattern {
				crackle scale 8
				turbulence .3 lambda 4 omega .5 //this gives the noisy high-frequency variation along the edges of the hot bits
				scale 1/turb_scale warp { turbulence 2 octaves 3 } scale turb_scale //this gives lower frequency distortions making the whole thing look more liquid
				//beautiful! but can I make the cracks relate to the shape?
			}
			//modulate to fade at specific distance (this will make the cracks only appear on the ones at the front)
			pigment_map {
				[0
					function {
						min(1,max(0,
							(z-20)/30
						))
					}
					cubic_wave
				]
				[1 rgb 1]
			}
		}
	
	//pillow lava
	//a blob with complex normal map and pigment
	blob {
		#local r = 1.2;
		#local s = 200;
		
		//spew forth in pillow shapes
		//tree like structure, so randomly give up and find some earlier branch point
		//=> recursive algorithm, length randomized from fixed value (i.e. probability of life decreases exponentially)
		//n.b. this is far from scientific, I stopped improving the algorithm when it made a nice shape.
		#local prob_branch = .3;
		#local prob_life = .985; //multiplied each time
		#local lifeseedoff = 4;
		#local shapeseedoff = 42;
		#macro pillow_lava( pos, dir, life )
			//make rand deterministic - sort of
			#local shapeseed = seed(pos.z*10+shapeseedoff);
			sphere {
				0, r, s
				//0, 1
				scale (<2,1,3>+<rand(shapeseed),rand(shapeseed),rand(shapeseed)>*2)/2
				rotate z*90*rand(shapeseed)*(1-2*rand(shapeseed))
				Reorient_Trans(z,dir)
				translate pos
			}
			#local num_branches = 0;
			#local lifeseed = seed(pos.z*10+lifeseedoff);
			#while ( rand(lifeseed)<life & rand(shapeseed)<pow(prob_branch,num_branches) )
				#local pos2 = pos + dir;
				#local dir2 = vrotate(dir, y*(rand(shapeseed)*2-1)*10*(num_branches*5+1));
				#local pos2 = pos2 + dir2;
				pillow_lava( pos2, dir2, life*prob_life )
				#local num_branches = num_branches+1;
			#end
		#end
		pillow_lava(<0,0,0>, -z, 1)
		rotate -0*y
		translate <20,-8,95>
		
		/*clipped_by {
			plane { z, 300 }
		}*/

		pigment {
			pigment_pattern { cold_lava_pattern }
			poly_wave .3
			pigment_map {
				[0 rgb <10,1,.2>*4]   //hot lava
				[.5 rgb .2]  //cooling rim
				[1 granite colour_map{ [0 rgb .1][1 rgb .07] }] //cooled lava
			}
		}
		normal {
			pigment_pattern { 
				pigment_pattern { cold_lava_pattern }
				poly_wave .3
				pigment_map {
					[.45 rgb 0] //molten is smooth
					[.55 granite warp { turbulence 2 octaves 3 } scale turb_scale colour_map { [0 rgb 0][1 rgb 1] }] //solid is lumpy, note this duplicates the low-frequency turbulence from earlier, to reinforce the impression that it's all bent and twisted
				}
			} 1
		}
		finish { brilliance 2 diffuse 1.5 }
	}
#end


//-----------------------------------------sea----------------------------------------
#if (toggle_sea)
	difference {
	
		//cunning trick, water surface is a plane with an inverted height-field cut out near the camera so we get a smooth transition
		plane {
			y, -.1
			texture {
				pigment { rgb 0 transmit 1 }
				finish {
					reflection { 0, .95 falloff 5 } conserve_energy
				}
				normal { granite -.1 scale 5 }
			}
		}

		height_field {
			//this is gonna be rotated, so we're looking from the <0,0> corner in the <1,1> direction
			function 1024, 1024 {
				f_granite(x*2,y*2,0) //upside down wave shape, scaled -1*y then subtracted from the plane
				*pow(max(0,(1-(x*x+y*y))),2) //ramp down towards far edge, so smooth connection to plane
			}
			scale <1,1,-1> translate z //FFS!
			rotate -45*y
			scale <1,0,1>*10 - .05*y
			translate -.099*y
			
			texture {
				pigment { rgb 0 transmit 1 }
				finish {
					reflection { 0, .95 falloff 5 } conserve_energy
				}
			}
		}

		//cut out the camera box
		object {
			camera_box
			#if (cheaper_sea)
				pigment { rgbf <.3,.6,.6,1> }
				finish { diffuse 0 }
			#end

			texture {
				pigment { rgb 0 transmit 1 }	
				finish {
					reflection { 0, .95 falloff 5 } conserve_energy
				}
				//fake heat ripples
				//note: could make a texture by painting over the lava shape! (remember, cloudy pattern, not ripply)
				//but I'm too lazy so I'll just do a pov pattern that seems to fit really well by total coincedence!
				normal {
					pigment_pattern {
						average
						pigment_map {
							[1
								spherical poly_wave 2	//nice distortion on the righthand bit
								scale 1
								turbulence .7 lambda 3 omega .7
								scale .2 translate vnormalize(<0,-.3,1>)
								colour_map { [0 rgb 0][1 rgb 2] }
							]
							[1
								spherical poly_wave 2	//nice distortion on the lefthand bit
								scale 1
								translate .1 
								turbulence .7 lambda 3 omega .7
								scale <-1,1,1>
								scale .2 translate vnormalize(<-.1,-.3,1>)
								colour_map { [0 rgb 0][1 rgb 2] }
							]
						}
					} .01
				}
			}
		}
   
		
		hollow on

		#if (cheaper_sea)
			no_shadow
		#end

		//ludicrously complex media for the sea, consisting of 2 effects:
		//  -fog which absorbs and emits differently to give a nice depth-based tint to things. density modulated by height but the effect is pretty subtle.
		//  -rays of light, totally faked using an emissive media which is stretched away from the light and falls off shortly beneath the surface.
		//the colours and densities for these effects have been extensively tweaked to get something that looks good.
		//though I'm still not sure what colour the water should be in this scene, somewhere between tropical turquoise and murky green/brown...
		#if (1)
			interior {
				ior 1.33
				media {
					emission rgb <.2,.6,.7>
					absorption rgb <12,7,5> //everything tint - very visible in surface reflection
					density {// rgb .003 }
						gradient y
						scale 11
						translate -10*y
						poly_wave 1/10
						density_map {
							[0 rgb .1]
							[1 rgb 0]
						}
					}
				}
				media {
					//fake rays of light shining through crap in the water
					emission rgb <.5,1,.8>
					absorption 0
					aa_threshold 1/128
					//method 1 intervals 10 samples 3,10 //noise, to make artefacts look more like crap in the water. - n.b. slow
					density {
						function {
							1/(.3-y*1)
						}
						density_map {
							[0 rgb 0]
							[1
								#declare light_x = light_dir.x;
								#declare light_y = light_dir.y;
								#declare light_z = light_dir.z;
								function {
									max(0,
										2*f_spotted(x-y*light_x/light_y,0,z-y*light_z/light_y)
										-1
									)
								}
								cubic_wave
								density_map {
									[0 rgb 0]
									[1 rgb .05]
								}
							]
						}
					}
				}
			}
		#end
	}
#end

//---------------------------------------volcano--------------------------------------
#local volcano =
	object {
		#if (placeholder_volcano)
			union {
				cone { <0,-10,2000>,800,<0,900,2000>,150 }
				cone { <0,-10,2000>,1500,<0,600,2000>,50 }
			}
		#else
			isosurface { 
				#local base =
					function {
						y
						-1100*(
							+ 1.1 / (1 + 4* //control steepness of curve
								pow(.002+abs( //round lip a little
									pow(x/1800,2) + pow(z/1800,2)
									-.007), //crater size, relative to other stuff
								.8))
							-.3
						)
					}
				//distort this perfect shape using turbulence.
				//I was using additive lumpy functions for the distortion, but turbulence gives a better effect.
				#local distorted_base_p =
					function {
						pigment {
							function { base(x,y,z)/10000+.5 }
							#local Scale = <600,100000,600>;
							#local Offset = 9;
							scale 1/Scale
							translate -Offset
							warp {
								turbulence <100,150,100>/Scale
								octaves 10
								lambda 2
								omega .5
							}
							translate Offset
							scale Scale
							rotate y*150
						}
					}
				#local distorted_base =
					function {
						(distorted_base_p(x,y,z).x-.5)*10000
					}
				
				function {
					distorted_base(x,y,z)
				}
				max_gradient 2
				accuracy 1/10 //makes it faster and is fine for this distance
				contained_by { box { <-1800,0,-1995>, <1800,1100,1800> } }
			
				translate <0,-10,2000>
			}
		#end

		//texture combining slope-based with altitude & turbulence to control snow placement
		texture {
			pigment_pattern {
				slope {
					y-.3*x, .5, 1
				}
				pigment_map {
					[0 rgb 0]
					[1
						function {
							min( 1, max( 0, 1-y )) //inverted, so no snow near eruption
						}
						scale 600
						translate 300*y + 200*z
						scale 1/300 warp { turbulence 2 lambda 4 octaves 2 } scale 300
					]
				}
			}
			texture_map {
				[.5
					//volcanic rock
					pigment { rgb .05 }
					finish { brilliance 2 specular 1 roughness 1/100 reflection { .0, .5 falloff 5 } }
				]
				[.5
					//snow
					pigment { rgb 1 }
					finish {
						brilliance 2 diffuse 1
						phong 1 phong_size 200
						reflection { .0, .7 falloff 5 }
					}
					normal {
						spotted 2 scale .05
					}
				]
			}
		}
	}

#if (toggle_volcano | placeholder_volcano)
	light_group {
		object { volcano }
		
		//light for specular highlights on right side
		light_source { <1000,1000,2200>, rgb <.5,.7,1> }
		global_lights on
	}
#end

#if (toggle_lava_rivers)

	//amount we move is much smaller than distance we trace, because lava can melt through small bumps.
	//this was a work-round to stop it getting stuck on the very lumpy rocky surface, but actually gives a more natural looking smoother path.
	#declare sample_spacing = 2;
	#declare sample_look_dist = 10;

	//lava flow macro - make a river of lava
	//notes:
	//-can do vertical drops, though will always break line
	//-forked river: drop 2 particles so close they start out overlapping.
	#macro lava_flow( source_point, rad, target_y, rand_ang_seed )
		#local pseudorandang = seed(rand_ang_seed);
		//roll down the volcano, dropping spheres as we go, these will be put in a blob outside this function
		#local norm = <0,0,0>;
		#local newpos = trace( volcano, source_point, -y, norm );
		#local top = newpos.y;
		#local oldpos = newpos;
		#local stationary = false;
		#local static_restarts = 10;
		#local dist = 0;
		
		#while ( !stationary & newpos.y > target_y )

			#local static_count = 100;		
			#local Once = true;
			#while ( Once | (vlength( newpos-oldpos ) < sample_spacing*.99 & static_count > 0) )

				#if ( norm.y > 0.9 )
					//almost vertical, use random ang instead
					#local norm = -z; //<-looks better vrotate(z, y*360*rand(pseudorandang));
				#end

				//always sample in previously implied slope direction
				#local dir = vnormalize(norm*<1,0,1>);
				#local tracepos = newpos + 1000*y + dir*sample_look_dist;
				#local lowestpos = trace( volcano, tracepos, -y, norm );
				
				//compare that sample to other ones nearby
				#local testnorm = <0,0,0>;
				#local numtests = 20;
				#local testcount = numtests;
				#while (testcount > 0)
					#local tracepos = newpos + 1000*y + vrotate(dir,y*7*360*(testcount+rand(pseudorandang))/(numtests+1))*sample_look_dist;
					#local testpos = trace( volcano, tracepos, -y, testnorm );
					#if (testpos.y < lowestpos.y)
						#local norm = testnorm;
						#local lowestpos = testpos;
					#end
					#local testcount = testcount - 1;
				#end

				#if ( lowestpos.y >= oldpos.y )//+rad*2 )
					#local static_count = static_count - 1;
				#end

				//move towards lowest pos by sample spacing, this allows us to push through rocks in our way.
				#local deltapos = lowestpos - newpos;
				#local newpos = newpos + deltapos * min( 1, sample_spacing / vlength(deltapos) ); //move by exactly sample spacing unless sample pos was closer than that.
				
				#local Once = false;
			#end
			#local dist = dist + vlength(newpos-oldpos);
			
			#write( lava_inc,
						"		sphere { ",
							"0, ", str(rad*2,0,5), ", 1 ",
							"scale <1,.2,1> ",		//scale it so we get a flat, wide lava river
							"Reorient_Trans(y,<", vstr(3,norm,",",0,5),">) ",
							"translate <", vstr(3,newpos,",",0,5),"> ",
						"}\n"
						)

			//#debug concat(vstr(3,newpos,",",0,1),"...",str(static_count,0,0),"...",str(dist,0,0),"\n")
			#debug concat(str(100*(top-newpos.y)/(top-target_y),3,0),"% of lava flow\n")
			
			#if (static_count <= 0 | newpos.y > oldpos.y)
				//damn it got stuck, try restarting it a little further out.
				//this is a total cheat, but from our low angle it's not too obvious.
				#local static_restarts = static_restarts - 1;
				//#local newpos = newpos + 20*vnormalize((newpos-<0,0,2000>)*<1,0,1>);
				#local newpos = newpos - 20*z; //camera is very forgiving if we move towards it.
				#local newpos = trace( volcano, newpos+2000*y, -y, norm );
				#debug concat("restart ",str(static_restarts,0,0))
			#end
			
			#local oldpos = newpos;
			
			#if (static_count <= 0 | static_restarts < 0)
				#local stationary = true;
			#end
			
		#end
		
		#if ( stationary )
			#warning concat("\nlava-flow stationary after ",str(dist,0,2),"m")
		#end
	#end

	//lava_rivers
	#if (save_lava_flows)
		#fopen lava_inc "lava_flows.inc" write
		
		#write( lava_inc, 
					"#declare lava_flows =\n",
					"	blob {\n",
					"\n")
					
		lava_flow( <0,2000,1580>, 8, 20, 0 ) //nice forground zig-zag
		lava_flow( <-4,2000,1820>, 8, 500, 0 ) //connect that to the top
		
		lava_flow( <-100,2000,1850>, 5, 400, 0 ) //left side trickle
		lava_flow( <-110,2000,1850>, 5, 400, 0 ) //left side trickle
		
		#write(lava_inc, "		threshold .5\n	}\n")
		
		#fclose lava_inc
	#end
	
	#include "lava_flows.inc" //If you get an error here, you need to run once with save_lava_flows turned on (it's at the top of this source)
	
	object {
		lava_flows

		//texture based on slope, so lava solidifies more where it lies static for a while, and more at lower altitudes
		texture {
			pigment_pattern {
				pigment_pattern {
					slope { y, .5, 1 }
					pigment_map {
						[0 //vertical
							//tortured cracks - similar to the pillow lava
							crackle scale 8
							turbulence .3 lambda 4 omega .5
							#local turb_scale = .7;
							scale 1/turb_scale warp { turbulence 2 octaves 3 } scale turb_scale
						]
						[1 rgb 1] //horizontal
					}
				}
				colour_map {
					[0 rgb 0]
					[.5 rgb 1]
				}
			}
			cubic_wave
			texture_map {
				[0
					//molten lava
					pigment { rgb <5,.5,.05> }
					finish { ambient 1 }//diffuse 1 brilliance 2 }
				]
				[1.005 //don't ever get completely solid
					//solidified crust
					pigment { granite colour_map{ [0 rgb .1][1 rgb .07] } }
					finish { diffuse 1.5 brilliance 2 }
					normal {
						pigment_pattern {
							granite
							warp { turbulence 2 octaves 3 }
							colour_map { [0 rgb 0][1 rgb 1] }
						} 1
					}
				]
			}
		}
	}

#end

#if (toggle_plume)
	#if (1)
		#declare jiggle =
			function(x,y) {
				x + sin(x*y*2*pi)/(y*2*pi)
			} //provided y is an integer > 0 this maps x in [0,1] to [0,1] with y lumps

		#macro flying_stuff()
			//flying lava
			#local init_pos_center = <0,930,2000>;
			#local init_pos_range = <50,0,50>; //if want more order move particles down, make this smaller and give more vel, so position they appear to come from has more to do with vel than pos.
			#local init_vel_center = <0,80,0>;
			#local init_vel_range = <20,20,20>;
			#local acc = <0,-9.81,0>;
			#local trail_length = .5; //seconds
			#local trail_length_on_ground = 5;
			#local trail_num_samples = 20;
			#local rad_trail_lead = 5;//8;
			#local rad_trail_tail = .01;
			#local time_range = 30;
			#local num_particles = 1000;
	
			#local blob_rad = 1; //leave this, twiddle the rad_'s above
			#local blob_strength = 1;
		
			#local iparticle = 0;
			#local rs = seed(11);
			#while (iparticle < num_particles)
			
				//randomize this particle's vals
				#local particle_time = -pow(rand(rs),1.5)*time_range; //time this particle was spawned (current time = 0 + trail_time)
//				#local init_pos = init_pos_center + (<rand(rs),rand(rs),rand(rs)>*2-1)*init_pos_range;
//				#local init_vel = init_vel_center + (<rand(rs),rand(rs),rand(rs)>*2-1)*init_vel_range;

				//adjust randomness to give clumps
				#local init_pos = init_pos_center + (<jiggle(rand(rs),2),jiggle(rand(rs),1),jiggle(rand(rs),1)>*2-1)*init_pos_range;
				#local init_vel = init_vel_center + (<jiggle(rand(rs),3),jiggle(rand(rs),8),jiggle(rand(rs),5)>*2-1)*init_vel_range;

		 		//if tail has landed, do a longer sample to give mini-flow
		 		#local _time = -particle_time;
		 		#local pos = init_pos + init_vel*_time + 0.5*acc*_time*_time; //s=ut + 1/2at^2
				#local newpos = trace( volcano, pos+y*10000, -y );
				#if ( newpos.y > pos.y )
					#local this_trail_length = trail_length_on_ground;
				#else
					#local this_trail_length = trail_length;
				#end
				
				#local trail_count = 0;
			 	#while (trail_count < trail_num_samples)
			 		#if ( trail_count = 0 )
						#local trail_param = 1;
					#else
						#local trail_param = 1 - trail_count / (trail_num_samples-1);//rand(rs);
					#end
					#local trail_time = trail_param*this_trail_length;
			 		#local _time = trail_time - particle_time;
			 		#local pos = init_pos + init_vel*_time + 0.5*acc*_time*_time; //s=ut + 1/2at^2
					#local newpos = trace( volcano, pos+y*10000, -y );
					#if ( newpos.y > pos.y ) #local pos = newpos; #end
					sphere { pos, blob_rad*rad_trail_lead, (rad_trail_tail + (rad_trail_lead-rad_trail_tail)*pow(trail_param,2))*blob_strength/rad_trail_lead }
					#local trail_count = trail_count + 1;
				#end
				
				#local iparticle = iparticle + 1;
			#end
		#end
		blob {
			flying_stuff()
		
			no_shadow
			hollow on

			pigment {
				//brighter near the vent
				spherical
				scale 600
				translate <0,1000,2000>
				poly_wave 2
				colour_map {
					[0 rgb <1,0,0> transmit 0]
					[1 rgb 10*<1,.3,.1> transmit 0]
				}
			}		
			finish { ambient 1 diffuse 0 }
		}
	#end
#end

//----------------------------------------smoke---------------------------------------
#if (cheapest_smoke)
	union {
		disc { <1700,1800,4000>, y, 3000 }
		disc { <0,1500,2000>, -z, 600 }
		pigment { rgb .1 }
		normal { granite 2 scale 1000 }
	}
#end

#if (toggle_smoke)
	#if (1)
		light_group {
			difference {
				#declare smoke_trans =
					transform {
						rotate <0,50,65> //rotate so a sticking out bit of the pattern meets the top of the volcano
		
						translate y //so everything that follows is relative to the base of the cloud.				
						scale <1,.5,1>
		
						rotate <20,0,-20> //brute force tilt it, so we get a nice lopsided effect.
						scale  3000
						translate <0,1100,1800>
					}
					
				//smoke shape is just elaborate turbulence effect applied to spherical pattern,
				//with adjustments to give sharp transition from 0 to full density
				#declare smoke_form =
					function {
						pigment {
							pigment_pattern { spherical colour_map { [.15 rgb 0][.35 rgb 1] } }
							#local t_mag = .7;
							#local t_scale = 1/3;
							scale 1-t_mag/2
							scale 1/t_scale warp { turbulence t_mag/t_scale lambda 3 omega .45 octaves 12 } scale t_scale
							cubic_wave
						}
					}
            
				sphere {
					0, 1
					transform { smoke_trans }
				}
				#if (toggle_plume)
					blob { flying_stuff() threshold 0.9 } //this is needed to fix some glitches
				#end
				
				hollow on
				
				pigment { rgbt 1 }
				interior {
					#ifdef (media_fog) media { media_fog } #end
					media {
						scattering { 3, rgb .5 extinction 0 }
						absorption 1
						density {
							function { smoke_form(x,y,z).x }
							
							transform { smoke_trans }
							
							density_map {
								[0 rgb 0]
								[1 rgb .1]
							}
						}
						#if (!cheaper_smoke)
							intervals 5 //fix some banding in the smoke
							aa_threshold 1/128 //black spots, presumably visible only near black because gamma correction
							aa_level 4 //4 default, higher makes it much slower but doesn't seem to fix any more problems.
							jitter 0.3 //disguise the remaining errors with noise (I can't afford to increase the render time any more)
						#end
					}
				}
				
			}
			
		
			//add a little shape to the blackness
			light_source {
				<3000,50,2500>, rgb .03
			}
			
			global_lights on
		}
	#end
#end

#ifdef (media_fog)
	//owing to a pov bug with un-bounded media or fog and bounded-media, where the bounded media gets too much fog compared to an opaque object at the same distance...
	//instead acheive fog with bounding, and difference with smoke container (which will also contain a duplicate of the fog)
	difference {
		plane { -z, -.001 }
		plane { -z, -1000000 } //far plane, media calcs don't like running to infinity
		plane { y, -.09 } //ocean surface
		#ifdef(smoke_trans)
			sphere { 0, 1.001 transform { smoke_trans } }
		#end
		hollow on
		pigment { rgbt 1 }
		interior {
			media { media_fog }
		}
	}
#end




//----------------------------------------steam---------------------------------------
#if (toggle_steam)
	#declare wind_shear = <.5,0,0>;

	//macro to create sheared column of steam	
	#macro steam( base_pos, quantity, dense_col, seed_pos )
		#local cheat = 30; //need some rescaling, too lazy to push it through neatly.
		#local height = quantity * 5/cheat;
		#local rad = quantity * .5/cheat;
		
		//cylindrical media with randomized turbulence
		cylinder {
			0, y*height, rad
			hollow on
			pigment { rgbt 1 }
			interior {
				media {
					emission 1
					absorption 1
					density {
						pigment_pattern {
							gradient y
							scale height
							poly_wave .2
						}
						density_map {
							[0
								#local tmag = 1; 
								#local ts = 2;
								cylindrical
								scale rad
								translate -seed_pos
								scale 1/ts
								warp {
									turbulence tmag/ts omega .4 lambda 2 //octaves 3
								}
								scale ts
								translate seed_pos
								scale 1/(1+tmag)
								cubic_wave
								colour_map {
									[0 rgb 0]
									[1 dense_col]
								}
							]
							[1
								rgb 0
							]
						}
					}
				}
			}
			
			scale cheat
			Shear_Trans( x, y+wind_shear, z )
			translate base_pos
		}
	#end

	//lowest snow
	//steam( <0,200,1060>, 20, rgb .02, 0 )
	steam( <-10,180,1000>, 30, rgb .02, 1 )
	steam( <10,160,950>, 30, rgb .04, 2 )
	steam( <-10,140,900>, 20, rgb .1, 3 )
	steam( <10,120,850>, 30, rgb .1, 4 )
	steam( <-10,100,800>, 30, rgb .04, 5 )

	//top snow patch
	steam( <0,500,1450>, 40, rgb .04, 7 )
	steam( <-20,470,1420>, 30, rgb .2, 24 )
	steam( <0,470,1420>, 30, rgb .2, 8 )
	steam( <-10,550,1480>, 30, rgb .2, 6 )
	
	//snowy crest in the middle
	steam( <-30,260,1150>, 30, rgb .3, 9 )
	steam( <-20,250,1200>, 30, rgb .1, 14 ) //implied snow behind crest
	steam( <-40,260,1150>, 30, rgb .1, 17 )
	steam( <-35,250,1100>, 30, rgb .1, 21 )

	//implied from lava on left behind crest
	steam( <-250,300,1400>, 50, rgb .2, 71 )
	steam( <-220,300,1400>, 50, rgb .1, 43 )

	//burst bubbles from below
	union {
		#declare comp = .1;
		steam( <-5,-.1,40>/comp, 3/comp, rgb .04, -1 )
		steam( <-4,-.1,40>/comp, 3/comp, rgb .04, -3 )
		steam( <-3,-.1,40>/comp, 3/comp, rgb .07, -20 )
		steam( <-2.5,-.1,40>/comp, 3/comp, rgb .07, -22 )
		steam( <.5,-.1,40>/comp, 3/comp, rgb .04, -5 )
		steam( <1,-.1,40>/comp, 3/comp, rgb .04, -7 )
		scale comp
		scale <1,.7,1>
	}


#end



//--------------------------------------undersea steam-------------------------------------
#if (toggle_undersea_steam)

	//trail of bubbles, as if from a source point that has been moving forwards
	#macro trail_of_bubbles (pos)
		//thrusting forwards (towards us)
		//each bubble goes up but the source has been moving mostly forwards with some turbulence
		//bubbles hold steam so they can get smaller as they go up (it condenses back into surrounding water)
		#local turb_scale = 1;
		#local turb_mag = 1; //turbulence is cumulative, so pos is always hit
		#local speed = <0,1,4>; //trail direction
		#local life_mod = .5; //how quickly they change from full violent bubbles to calm small ones
		#local rad = .1;
		#local delta_T = 1; //arbitrary, controls number of bubbles emitted
		
		#local life = 1;
		#local pos = pos + delta_T*.2*rand(rs)*(speed + turb_mag*vturbulence(2,.5,6,pos/turb_scale));
		#while ( pos.y < -.1 ) //water surface
		
			//cluster of bubbles
			sphere { pos, rad*(life+.5) }
			sphere { pos-y*.15+<rand(rs),rand(rs),rand(rs)>*.14-.07, rad*.5*(life+.5) }
			sphere { pos-y*.35+<rand(rs),rand(rs),rand(rs)>*.14-.07, rad*.25*(life+.5) }

			#local life = life*life_mod;
			#local pos = pos + delta_T*(.2+rand(rs))*(speed + turb_mag*vturbulence(2,.5,6,pos/turb_scale));
		#end
	#end

	#macro line_of_bubbles ( start_pos, end_pos )
		//line of trails of bubbles
		#local foo = 0;
		#while (foo <= 1)
			trail_of_bubbles( start_pos*(1-foo) + end_pos*foo )
			#local foo = foo + (.5+rand(rs))*.1/vlength(start_pos-end_pos); //.1 unit spacing
		#end
		
		#if (0) //debug, show source line
			cylinder {
				start_pos, end_pos, .1
				pigment { rgb y }
				finish { ambient 1 diffuse 0 }
			}
		#end
	#end
	

	union {
		#declare rs = seed(17);	
		
		line_of_bubbles( <1,-6.8,23>, <3.5,-6.5,23> )
		line_of_bubbles( <2,-6.7,23>, <2.5,-7.5,22> )
		
		#declare rs = seed(11);	//too much clumping at the vent

		line_of_bubbles( <-5.2,-7.5,20>, <-2,-7,22> )

		#declare rs = seed(3);

		line_of_bubbles( <-2,-7,24>, <-1,-7,24> )

		hollow on		
		pigment { rgb 0 transmit 1 }
		finish { reflection { 0, 1 fresnel } conserve_energy }
		interior {
			#if (toggle_sea)
				ior 1
			#else
				ior 1/1.33
			#end
			media {
				emission 1
				density {
					gradient y
					scale 8
					translate -8
					poly_wave 1/5
					colour_map {
						[0 rgb 5]
						[1 rgb 0]
					}
				}
			}
		}
	}
#end


