// Persistence of Vision Ray Tracer Scene Description File
// File: snowman.pov
// Vers: Megapov 1.2
// Desc: Basic Scene Example
// Date: 26-01-06
// Auth: Sean Day
//

#version unofficial MegaPov 1.2;  

#include "colors.inc"  
#include "functions.inc"    
#include "transforms.inc"   

#declare useRad = 1; // 0 = none, 1 = load, 2 = save  (When saving radiosity data turn off the showClouds and maybe fog?  
#declare radQuality = 50; //1-100  1=low 100=Very high  

#declare showLandscape = 1;
#declare useIsoLandscape = 0; // Never used it as it was too slooowww!  
#declare showSnowman = 1;    
#declare hfLandRes=4000; // values above 1500 seem to be OK depending on resolution use lower values for test renders     
#declare showTree = 1;  // shows background trees  
#declare showMainTree = 1; // shows the main tree but needs the above set to 1
#declare useFocalBlur = 50; // 0 - 100   0=none 1=very low quality 100=very high quality 
#declare showClouds = 1; // turn off when generating radiosity data    
#declare showFence = 1;         
#declare isoFence = 1; // Non-iso fence is not the same as the iso one so I would not use it.    
#declare useFog = 1;


       
#if(showTree)
#include "black_tupelo.inc" 
#include "my_lombardy_poplar.inc"
#include "my_quaking_aspen.inc"
#end

#declare PRETRACE_END=2/image_width;

global_settings {    

	assumed_gamma 1
	max_trace_level 20

#if(useRad)
	radiosity {
  // save settings: modified Outdoor settings from rad_def.inc
  #if (useRad=2) 
  		pretrace_start 1
		pretrace_end PRETRACE_END

		error_bound { 0.15 adaptive 1.5, 20 }

		count 5*radQuality
		recursion_limit 1       
		normal on
		save_file "ice.rad"
  #end
  #if(useRad=1)
    // load settings  
    pretrace_start 1 pretrace_end 1 
    load_file "ice.rad" 
    error_bound { 0.15 adaptive 1.5, 20 }    // it's important to specify the same value used to save data
    recursion_limit 1  // also the same if specified
    always_sample off  //   
    normal on
  #end  
	}       
#end
}   



// ----------------------------------------

camera {
  location  <0.0, 10, -100.0>
  direction 1.5*z
  right     x*image_width/image_height
  look_at   <0.0, 4.0,  0.0> 
  #if(useFocalBlur)  
  aperture 0.3           // [0...N] larger is narrower depth of field (blurrier)
  blur_samples 10+useFocalBlur        // number of rays per pixel for sampling
  focal_point <0,1,100>    // point that is in focus <X,Y,Z>
  confidence 0.95+(useFocalBlur/2500)           // [0...<1] when to move on while sampling (smaller is less accurate)
  variance 1/200*useFocalBlur            // [0...1] how precise to calculate (smaller is more accurate)    
  #end
} 

#if(useFog)
fog {
  fog_type   2
  distance   10000
  color      rgb 0.9
  fog_offset 0
  fog_alt    250
  turbulence 1
  turb_depth 0.5
  omega 0.7
  lambda 3   
  scale <100, 10000, 10>     
  rotate <10, 0, 0>  
}
#end 

sky_sphere 
{
        pigment 
        {
                gradient y lambda 4 omega 0.7 turbulence 0.01 
                color_map 
                { 
                        [0.0 rgb <0.525,0.635,1.0>] 
                        [0.135 rgb <0.525,0.635,1.0>] 
                        [0.375 rgb <0.90,0.20,0.01>]      //r=0.6
                        [0.475 rgb <0.7,0.2,0.05>]       //g=0.5
                        [0.625 rgb <0.15,0.25,0.7>]       //rgb <0.5,0.6,1.0
                } 
                scale 2 translate <0, -1, 0> rotate <-14, 0, 0> 
        }
  
}


#declare generalRand = seed(582);   

// Macro to return a random number

#macro gr(num)
        rand(generalRand)*num
#end


// ----------------------------------------

#declare snowTex = texture
{
        pigment { rgb <1, 1, 1> }  // was 0.001
        finish { ambient 0 diffuse 0.9 brilliance 1.75 specular 1.5 roughness 0.0001 reflection 0.025  }
        normal 
        { 
                average 
                normal_map 
                {
                        [ 1 granite 0.5 scale 0.002 turbulence 1 omega 0.4 lambda 1.2 ]
                        [ 1 wrinkles 0.75 scale 0.15 ] 
                }     
        }
}          

#declare snowTexClear = material 
{
        texture
        {
                pigment 
                { 
                        wrinkles 
                        turbulence 0.75 
                        omega 0.75 
                        lambda 5 
                        colour_map 
                        { 
                                [ 0.0 rgbt <1, 1, 1, 0.975> ] 
                                [ 1.0 rgbt <1, 1, 1, 0.05> ] 
                        } 
                        scale <0.5, 0.3, 0.2>                 
                }
                finish { ambient 0 diffuse 0.9 brilliance 1.75 specular 1.5 roughness 0.0001 reflection 0.025  }
                normal 
                { 
                        average 
                        normal_map 
                        {
                                [ 1 granite 0.5 scale 0.002 turbulence 1 omega 0.4 lambda 1.2 ]
                                [ 1 wrinkles 0.75 scale 0.15 ] 
                        }     
                }
        
        }          
        // describes inside of a shape (similar to texture, but for inside, not surface)
        interior 
        {
                ior 1.4              // index of refraction
                caustics 1.0          // faked caustics (0..1)
                fade_power 1001       // values larger than 1000 give realistic exponential attenuation
                fade_distance 5       // distance where light reaches half intensity
                fade_color <0, .1, 0.1, 0.5> // color for fading     
                media {absorption 0.1 } 
 
        }     
}

#declare snowTexClearGradient = material 
{
        texture
        {     
                pigment
                {    
                        gradient y triangle_wave       
                        turbulence 0.065
                        omega 0.6
                        lambda 2.3
                        pigment_map
                        {
                                [ 0.0 rgb 1 ] 
                                [ 1.0 wrinkles turbulence 0.75 omega 0.75 lambda 5 
                                        colour_map 
                                        { 
                                                [ 0.0 rgbt <1, 1, 1, 0.975> ] 
                                                [ 1.0 rgbt <1, 1, 1, 0.05> ] 
                                        } 
                                        scale <0.5, 0.3, 0.2> 
                                ]
                        }                
                }
                finish { ambient 0 diffuse 0.9 brilliance 1.75 specular 1.5 roughness 0.0001 reflection 0.025  }
                normal 
                { 
                        average 
                        normal_map 
                        {
                                [ 1 granite 0.5 scale 0.002 turbulence 1 omega 0.4 lambda 1.2 ]
                                [ 1 wrinkles 0.75 scale 0.15 ] 
                        }     
                }
        
        }          
        // describes inside of a shape (similar to texture, but for inside, not surface)
        interior 
        {
                ior 1.4              // index of refraction
                caustics 1.0          // faked caustics (0..1)
                fade_power 1001       // values larger than 1000 give realistic exponential attenuation
                fade_distance 5       // distance where light reaches half intensity
                fade_color <0, .1, 0.1, 0.5> // color for fading     
                media {absorption 0.1 } 
 
        }     
}



#declare fn_scallop_ripples=function { pigment {
        ripples          
        scallop_wave
        turbulence 0.5
        omega 0.5
        lambda 5
        colour_map
        {
                [ 0.0 rgb 0.5 ]
                [ 0.9 rgb 0.5 ]
                [ 0.92 rgb 1 ]
                [ 0.94 rgb 1 ]
                [ 1.0 rgb 0.5 ]
                
        }  
        scale <0.25, 0.6, 0.15>
}

}
    
#declare snowMan1 = union
{

// Bottom Body
isosurface 
{
        function 
        { 
                f_sphere(x, y, z, 1)
                -f_wrinkles(x*0.3,y*0.5,z*0.03)*0.35
                -f_wrinkles(x*3,y*4,z*4)*0.05 
                -f_dents(x*3,y*4,z*4)*0.05 
        }
        accuracy 0.001               
        max_gradient 2.75
        contained_by{sphere{0,1.3}}  
}        

// Body
isosurface 
{
        function 
        { 
                f_sphere(x, y, z, 0.75)
                -f_wrinkles(x*0.3,y*0.5,z*0.03)*0.35
                -f_wrinkles(x*3,y*4,z*4)*0.05 
                -f_dents(x*3,y*4,z*4)*0.05 
        }
        accuracy 0.001               
        max_gradient 2.75
        contained_by{sphere{0,2.2}}
        translate <0, 1.7, 0>    
}
      
// Head
isosurface 
{
        function 
        { 
                f_sphere(x, y, z, 0.45)
                -f_wrinkles(x*0.3,y*0.5,z*0.03)*0.35
                -f_wrinkles(x*3,y*4,z*4)*0.05 
                -f_dents(x*3,y*4,z*4)*0.05 
        }
        accuracy 0.001               
        max_gradient 2.75
        contained_by{sphere{0,2.2}}
        translate <0, 2.9, 0>    
}        

}
// Snowman 2
    

#if(showSnowman)
object { snowMan1 
        scale 12  
        rotate <-0.5, -15, -0.5>
        translate <-20, -31, 65>
        texture { snowTex }
}    
#end

#declare LOAD_SNOW_DATA = 1;
#declare SAVE_SNOW_DATA = 2;

#macro Place_Snow(objectSurface, flakeSize, flakeSpace, random, fallDirection, flakeDepth, driftFactor, dataLoadSave, dataFilename)
    //Allow the user to pass scalars as vectors
    #local objStartPos = min_extent(objectSurface);
    #local objEndPos = max_extent(objectSurface);    
    #local fallHeight = objEndPos.y+1;               
    #local yDiff = fallHeight-objStartPos.y;     
    #local lastDrift=0;
    
    #if(fallDirection.x > 0)
        #if(fallDirection.z > 0)
                #local startPos=<objStartPos.x-(fallDirection.x*yDiff), objStartPos.y, objStartPos.z-(fallDirection.z*yDiff)>;
                #local endPos=<objEndPos.x, objEndPos.y, objEndPos.z>;  
        #else
                #local startPos=<objStartPos.x-(fallDirection.x*yDiff), objStartPos.y, objStartPos.z>;
                #local endPos=<objEndPos.x, objEndPos.y, objEndPos.z-(fallDirection.z*yDiff)>;  
        #end
    #else
        #if(fallDirection.z > 0)
                #local startPos=<objStartPos.x, objStartPos.y, objStartPos.z-(fallDirection.z*yDiff)>;
                #local endPos=<objEndPos.x-(fallDirection.x*yDiff), objEndPos.y, objEndPos.z>;  
        #else
                #local startPos=<objStartPos.x, objStartPos.y, objStartPos.z>;
                #local endPos=<objEndPos.x-(fallDirection.x*yDiff), objEndPos.y, objEndPos.z-(fallDirection.z*yDiff)>; 
        #end
    #end                      

    #declare startX = startPos.x;  
    #declare endX = endPos.x;
    #if(startX > endX)
        #declare startX = endPos.x;
        #declare endX = startPos.x;
    #end

    #declare startZ = startPos.z;  
    #declare endZ = endPos.z;
    #if(startZ > endZ)
        #declare startZ = endPos.z;
        #declare endZ = startPos.z;
    #end   
    
    #declare lFlakeSize = flakeSize;
    #declare lastHit = false;

        #declare flakeX = 0.0;
        #declare flakeY = 0.0;
        #declare flakeZ = 0.0;  
        #declare N = <0, 0, 0>;
        #declare S = <0, 0, 0>;

#declare cCount = 0;   
        #declare firstLine=true;     
        #declare percentDone = -1;
blob {
  threshold 0.6
    
        #if(dataLoadSave != LOAD_SNOW_DATA)     
                #if(dataLoadSave = SAVE_SNOW_DATA) // open file
                        #fopen snowDataFile dataFilename write
                #end                                              
                
                
                #declare zPos=startZ;
                #while(zPos < endZ)
                        #declare xPos=startX;     
                        #if(floor(((zPos-startZ)/(endZ-startZ))*100) > percentDone)  
                                #declare percentDone=floor((((zPos-startZ)/(endZ-startZ)))*100);
                                #warning concat(concat(concat(str(percentDone, 3, 1), "% Done. "), str(cCount,0,0)), " Flakes.")
                        #end      
                        #while(xPos < endX)
                                #declare xPos2 = xPos;
                                #declare zPos2 = zPos;
            
                                #local N = <0,0,0>;                                                          
                                
                                #local S = trace(objectSurface, <xPos2, fallHeight, zPos2>, fallDirection, N); 
                                #declare moved = false;
                                #if(vlength(N) > 0) // hit     
                                        #if(N.y < driftFactor)
                                                #local drift=true;      
                                        #else
                                                #local drift=false;
                                        #end   
                                        
                                        #while(drift)  
                                                #declare yPos2 = S.y;  
                                                #declare xPos2 = xPos2+((flakeSpace-((flakeSpace*random)/2)+gr(flakeSpace*random))*N.x); 
                                                #declare zPos2 = zPos2+((flakeSpace-((flakeSpace*random)/2)+gr(flakeSpace*random))*N.z); 
                                                #local S = trace(objectSurface, <xPos2, yPos2, zPos2>, fallDirection, N); 
                                                #declare moved=true;
                                                
                                                #if(vlength(N) > 0) // still hit
                                                        #if(N.y < driftFactor+gr(driftFactor/2))
                                                                #declare drift=true; 
                                                        #else
                                                                #declare drift=false;
                                                        #end      
                                                #else
                                                        #declare drift=false;
                                                #end
                                        #end 
                                #end      
                                
                                #local nextN = <0, 0, 0>;
                                #local nextS = trace(objectSurface, <xPos+flakeSpace, fallHeight, zPos+flakeSpace>, fallDirection, nextN);
                                
                                //Check if we hit the surface
                                #if (vlength(N)>0)  
                                        #declare lFlakeSize = flakeSize-((flakeSize*random)/2)+gr(flakeSize*random);
                                        //place the snow
                                        #if(vlength(nextN)>0) // next flake should hit
                                                #if(lastHit = false) // we did not hit last time so do smaller snow
                                                        #declare flakeX = 0.7+gr(0.3);
                                                        #declare flakeY = flakeDepth+gr(flakeDepth);
                                                        #declare flakeZ = 0.7+gr(0.3);
                                                #else // last flake hit so do normal
                                                        #declare flakeX = 0.9+gr(0.2);
                                                        #declare flakeY = flakeDepth+gr(flakeDepth);
                                                        #declare flakeZ = 0.9+gr(0.2);
                                                #end
                                        #else
                                                        #declare flakeX = 0.9+gr(0.2);
                                                        #declare flakeY = flakeDepth+gr(flakeDepth);
                                                        #declare flakeZ = 0.9+gr(0.2);
                                        #end          
                                        #if(moved)     
                                                        #declare mFlakeSize=lFlakeSize+gr(lFlakeSize);
                                                        #declare mFlakeDepth=flakeDepth+(gr(flakeDepth/2));
                                        #else
                                                        #declare mFlakeSize=lFlakeSize;
                                                        #declare mFlakeDepth=flakeDepth;
                                        #end         
                                                        sphere { <0, 0, 0>, mFlakeSize, mFlakeDepth scale <flakeX, flakeY, flakeZ>   
                                                        
                                                #if(dataLoadSave = SAVE_SNOW_DATA)  
                                                        #declare scaleV = <flakeX, flakeY, flakeZ>;
                                                        #if(firstLine)
                                                                //#write(snowDataFile, lFlakeSize,",",scaleV,",",fallDirection,",",N,",",S)
                                                                #declare firstLine=false;
                                                        #else
                                                                //#write(snowDataFile, ",",lFlakeSize,",",scaleV,",",fallDirection,",",N,",",S)  
                                                                #declare snowStr="sphere { 0, ";
                                                                #declare snowStr=concat(snowStr, str(mFlakeSize, 5, 5));
                                                                #declare snowStr=concat(snowStr, ", ");
                                                                #declare snowStr=concat(snowStr, str(mFlakeDepth, 5, 5));
                                                                #declare snowStr=concat(snowStr, " scale <");
                                                                #declare snowStr=concat(snowStr, str(scaleV.x, 5, 5));
                                                                #declare snowStr=concat(snowStr, ",");
                                                                #declare snowStr=concat(snowStr, str(scaleV.y, 5, 5));
                                                                #declare snowStr=concat(snowStr, ",");
                                                                #declare snowStr=concat(snowStr, str(scaleV.z, 5, 5));
                                                                #declare snowStr=concat(snowStr, "> ");

                                                                #declare snowStr=concat(snowStr, "Reorient_Trans(<");
                                                                #declare snowStr=concat(snowStr, str(fallDirection.x, 5, 5));
                                                                #declare snowStr=concat(snowStr, ",");
                                                                #declare snowStr=concat(snowStr, str(fallDirection.y, 5, 5));
                                                                #declare snowStr=concat(snowStr, ",");
                                                                #declare snowStr=concat(snowStr, str(fallDirection.z, 5, 5));
                                                                #declare snowStr=concat(snowStr, "> ");

                                                                #declare snowStr=concat(snowStr, ", <");
                                                                #declare snowStr=concat(snowStr, str(N.x, 5, 5));
                                                                #declare snowStr=concat(snowStr, ",");
                                                                #declare snowStr=concat(snowStr, str(N.y, 5, 5));
                                                                #declare snowStr=concat(snowStr, ",");
                                                                #declare snowStr=concat(snowStr, str(N.z, 5, 5));
                                                                #declare snowStr=concat(snowStr, ">) ");
                                                               
                                                                #declare snowStr=concat(snowStr, " translate <");
                                                                #declare snowStr=concat(snowStr, str(S.x, 5, 5));
                                                                #declare snowStr=concat(snowStr, ",");
                                                                #declare snowStr=concat(snowStr, str(S.y, 5, 5));
                                                                #declare snowStr=concat(snowStr, ",");
                                                                #declare snowStr=concat(snowStr, str(S.z, 5, 5));
                                                                #declare snowStr=concat(snowStr, "> ");
                                                                #if(moved)
                                                                        #declare snowStr=concat(snowStr, " translate -<");
                                                                        #declare snowStr=concat(snowStr, vstr(3, fallDirection, ", ", 0, 4));
                                                                        #declare snowStr=concat(snowStr, ">*((");
                                                                        #declare snowStr=concat(snowStr, str(flakeSize/2, 5, 5));
                                                                        #declare snowStr=concat(snowStr, ")+");
                                                                        #declare snowStr=concat(snowStr, str(gr(flakeSize/2), 5, 5));
                                                                        #declare snowStr=concat(snowStr, ") }");
                                                                #else
                                                                        #declare snowStr=concat(snowStr, " }");                                                                
                                                                #end
                                                                #write(snowDataFile, snowStr)
                                                                //sphere { 0, lFlakeSize,",",scaleV,",",fallDirection,",",N,",",S)
                                                        #end
                                                #end

                                                //Translate to final position     

                                                
                                                Reorient_Trans(fallDirection,N)
                                                
                                                #if(moved)
                                                        translate S
                                                        translate -fallDirection*((flakeSize/2)+gr(flakeSize/2))
                                                #else
                                                        translate S
                                                #end
                                        }    
                                        #declare cCount = cCount+1;
                                        #declare lastHit = true;   // remember we had a hit this time
        
                                #end
                                #declare xPos = xPos+(flakeSpace-((flakeSpace*random)/2)+gr(flakeSpace*random)); 
                        #end                           
                        #declare zPos = zPos+(flakeSpace-((flakeSpace*random)/2)+gr(flakeSpace*random));
                #end         
        #else                   // Load the snow data  
                #include dataFilename  // this works sooo much faster than above attempt.
                
                                #declare cCount=cCount+1; // only problem is you lose the sphere count (not that that matters)

        #end
        #if(cCount = 0)
                #error "Place_Snow Error: no Snow has hit the object try using different direction or start/end pos for snow!"
        #end   
        #warning concat("Total: ", str(cCount,1,0), " snow flakes used")      
        #fclose snowDataFile
        
}    

#end // End Macro


#declare testObj = union { 
sphere { <0, 0, 0> 1 }
//cylinder { <0, 0, -1> <0, 0, 1> 1 }
box { <-3, -2, -4> <3, -1, 4> rotate <10, 0, 0> }
box { <2, -1, -2> <4, 1, 4> }

pigment { Red } 
}




// landscape


#declare Land1=function{pattern{quilted
   turbulence 0.275 scale <1.1, 1.0, 1.2> translate <-0.7, 0, 0.4> }}
#declare Land2=function{pattern{marble 
   turbulence 0.42 scale <0.95, 1, 1> }}
#declare Land3=function{pattern{crackle 
   turbulence 0.3 scale <0.7, 1, 0.25> rotate <0, 0, 0> translate <0.6, 0, 0.2> }}
#declare Land4=function{pattern{wrinkles
   turbulence 0.275 scale 0.05  }}

#declare Land=function{(sin(z*x*pi*4)+1)/4+Land1(x*5,0,z*5)*0.35+Land4(x*0.4, 0, z*0.8)*0.2+Land4(x*400, 0, z*800)*0.001}//Land1(x*5,0,z*5)*0.6+Land4(x*0.4, 0, z*0.8)*0.1}

#declare MidLand=function{Land3(x,0,z*0.5)+Land4(x*0.4, 0, z*0.8)*0.2+Land4(x*400, 0, z*800)*0.002}//+Land4(x*0.4, 0, z*0.8)*0.2+Land4(x*400, 0, z*800)*0.001}//Land1(x*5,0,z*5)*0.6+Land4(x*0.4, 0, z*0.8)*0.1}

#declare FarLand=function{Land2(x,0,z*0.5)*0.5+Land4(x*0.4, 0, z*0.8)*0.1+Land4(x*400, 0, z*800)*0.005}//+Land4(x*0.4, 0, z*0.8)*0.2+Land4(x*400, 0, z*800)*0.001}//Land1(x*5,0,z*5)*0.6+Land4(x*0.4, 0, z*0.8)*0.1}



#if(showLandscape)
// Main land
#if(useIsoLandscape)
#declare nearLand = isosurface {
    function { y - Land(x,0,1-z)}
    contained_by{box{<0,0,0>,<1,1,1>}}
    all_intersections
    accuracy 0.1
    max_gradient 4.522
    translate <-0.5,0,-0.5>
#else
#declare nearLand = height_field{function hfLandRes,hfLandRes {Land(x,0,y)} smooth 
  translate <-0.5,0,-0.5>
#end                   
   scale <500, 45, 500>  // was 50 
   translate <0, -50, 200>
   texture { snowTex } 
} 
object { nearLand }   
// middle landscape
#if(useIsoLandscape)
#declare middleLand = isosurface {
    function { y - MidLand(x,0,1-z)}
    contained_by{box{<0,0,0>,<1,1,1>}}
    all_intersections
    accuracy 0.1
    max_gradient 4.522
    translate <-0.5,0,-0.5>
#else
#declare middleLand = height_field{function hfLandRes/3,hfLandRes/3 {MidLand(x,0,y)} smooth 
  translate <-0.5,0,-0.5>
#end                   
   scale <5000, 250, 5000>   // was 500 
   rotate <-3, 0, 0>
   translate <0, -500, 2500>
   texture { snowTex } 
}

object { middleLand }
// middle landscape
#if(useIsoLandscape)
#declare farLand = isosurface {
    function { y - FarLand(x,0,1-z)}
    contained_by{box{<0,0,0>,<1,1,1>}}
    all_intersections
    accuracy 0.1
    max_gradient 4.522
    translate <-0.5,0,-0.5>
#else
#declare farLand = height_field{function hfLandRes/5,hfLandRes/5 {FarLand(x,0,y)} smooth 
  translate <-0.5,0,-0.5>
#end                   
   scale <15000, 1500, 15000>    
   rotate <-6, 0, 0>
   translate <0, -500, 12500>
   texture { snowTex } 
}      
object { farLand }  

#end 

#declare treeBaseTex = texture
{
                 pigment {color rgb <0.175, 0.1, 0>} 
                          finish { ambient 0.0 diffuse 0.5 } 
                          normal { average normal_map { 
                                [ 1 wrinkles scale <0.15, 2, 0.1> turbulence 3 omega 0.65 ]
                                [ 1 granite 0.1 scale <0.5, 1, 0.4> turbulence 3 omega 0.65 ]
                                 }   
                                 }
} 
 

#if(showTree)
#declare tree_1_base = object { black_tupelo_13_stems    

texture
{
        slope { <0.1, -1, 0.1> }
        texture_map
        {
                [ 0 snowTex ] 
                [ 0.3 snowTex ] 
                [ 0.375 treeBaseTex ]   
                [ 1 treeBaseTex ]   
        }     
}
         rotate 90*y 
} 
#declare tree = union
{
 object { tree_1_base }
}

#declare tree_2_base = union { 
         object { my_quaking_aspen_13_stems
texture
{
        slope { <0.1, -1, 0.1> }
        texture_map
        {
                [ 0 snowTex ] 
                [ 0.3 snowTex ] 
                [ 0.375 treeBaseTex ]   
                [ 1 treeBaseTex ]   
        }     
}
}
         rotate 90*y } 

#declare tree_3_base = union { 
         object { my_lombardy_poplar_13_stems
texture
{
        slope { <0.1, -1, 0.1> }
        texture_map
        {
                [ 0 snowTex ] 
                [ 0.3 snowTex ] 
                [ 0.375 treeBaseTex ]   
                [ 1 treeBaseTex ]   
        }     
}
}
         rotate 90*y } 

#declare tree2 = union
{
 object { tree_2_base }
}

#declare tree3 = union
{
 object { tree_3_base }
}
         
// Main Tree
#declare mainTree = object { tree_1_base scale 7 translate <55, -45, 125> }

#if(showMainTree)
object { mainTree }
object { Place_Snow( mainTree, 0.225, 0.06, 1.6, <0.4, -1, 0.25>, 0.625, 0.7, LOAD_SNOW_DATA, "mainTreeSnow.txt") 
material { snowTexClear }
}
#end

                                              

#declare treeBase = union
{
        difference 
        {
                cone { <0, 0, 0> 8 <0, 2, 0> 2.5  }
                cylinder { <0, -1, 0> <0, 3, 0> 2.4  }
        }                              
        #declare cr=0;
        #while(cr < 300)
        
                cone { <0, 0, 0> 0.4+gr(0.2) <0, 0.2+gr(1.25), 0> 0.03+gr(0.05) rotate <0, 0, -10> translate <-2.4, 2, 0> rotate <0, 80-cr, 0> }
                #declare cr=cr+25+gr(20);
        #end                          
        translate <54.5, -42.5, 124.5> 
}
object { Place_Snow( treeBase, 0.6, 0.15, 1.95, <0.25, -1, 0.2>, 0.7, 0.005, LOAD_SNOW_DATA, "treeBaseSnow.txt") 
material { snowTexClearGradient scale 3.5 translate <0, -43, 0>} 

}

// macro to see if tree is going to be visible to save memory on snow object         
#declare treeNeedsSnow=false;

#macro nearEdge(startPos, endPos, treePos, treeSpace)   
        #declare treeNeedsSnow=false;
        #if(treePos.x < (startPos.x+(treeSpace*6)))
                #declare treeNeedsSnow=true;
        #end
        #if(treePos.x > (endPos.x-(treeSpace*7)))
                #declare treeNeedsSnow=true;
        #end                
        #if(treePos.z < (startPos.z+(treeSpace*8)))
                #declare treeNeedsSnow=true;
        #end
        #if(treePos.z > (endPos.z-(treeSpace*6)))
                #declare treeNeedsSnow=true;
        #end
#end

// place Trees

#macro placeTrees(landObj, startPos, endPos, treeSpace, treeScale)
#declare treeZ=startPos.z;                                           
#declare treeCount=0;
#while(treeZ < endPos.z)
        #declare treeZ=treeZ+gr(treeSpace); 
        #declare treeX=startPos.x;  
        #while(treeX < endPos.x)
                #declare treeX=treeX+gr(treeSpace);
                #local treeHit = <0, 0, 0>;
                #local treePos = trace(landObj, <treeX, 100000, treeZ>, <0, -1, 0>, treeHit);  
                #if(vlength(treeHit) > 0) // hit     
                        #declare treeCount=treeCount+1;
                        #declare treeType=gr(10);
                        #if(treeType < 4)        
                                nearEdge(startPos, endPos, treePos, treeSpace)
                                #if(treeNeedsSnow)
                                        object { tree rotate <gr(1), gr(360), gr(1)> translate <0, -(0.1+gr(0.25)), 0> scale treeScale+gr(treeScale) translate treePos }     
                                #else
                                        object { tree_1_base rotate <gr(1), gr(360), gr(1)> translate <0, -(0.1+gr(0.25)), 0> scale treeScale+gr(treeScale) translate treePos }     
                                #end
                        #else
                                #if(treeType < 7)
                                        #if(treeNeedsSnow)
                                                object { tree2 rotate <gr(1), gr(360), gr(1)> translate <0, -(0.1+gr(0.25)), 0> scale treeScale+gr(treeScale) translate treePos }     
                                        #else
                                                object { tree_2_base rotate <gr(1), gr(360), gr(1)> translate <0, -(0.1+gr(0.25)), 0> scale treeScale+gr(treeScale) translate treePos }     
                                        #end                                        
                                
                                #else
                                        #if(treeNeedsSnow)
                                                object { tree3 rotate <gr(1), gr(360), gr(1)> translate <0, -(0.1+gr(0.25)), 0> scale treeScale+gr(treeScale) translate treePos }                        
                                        #else
                                                object { tree_3_base rotate <gr(1), gr(360), gr(1)> translate <0, -(0.1+gr(0.25)), 0> scale treeScale+gr(treeScale) translate treePos }                        
                                        #end        
                                #end
                        #end
                #end   
                #declare treeX=treeX+treeSpace+gr(treeSpace);
        #end   
        #declare treeZ=treeZ+treeSpace+gr(treeSpace);   
        
#end
        
        #warning concat(str(treeCount,5,5), " Trees added")
#end

#macro placeSmallTrees(landObj, startPos, endPos, treeSpace, treeScale)
#declare treeZ=startPos.z;                                           
#declare treeCount=0;
#while(treeZ < endPos.z)
        #declare treeZ=treeZ+gr(treeSpace); 
        #declare treeX=startPos.x;  
        #while(treeX < endPos.x)
                #declare treeX=treeX+gr(treeSpace);
                #local treeHit = <0, 0, 0>;
                #local treePos = trace(landObj, <treeX, 100000, treeZ>, <0, -1, 0>, treeHit);  
                #if(vlength(treeHit) > 0) // hit     
                        #declare treeCount=treeCount+1;
                        #declare treeType=gr(10);
                        #if(treeType < 5)
                                object { tree2 rotate <gr(1), gr(360), gr(1)> translate <0, -(0.1+gr(0.25)), 0> scale treeScale+gr(treeScale) translate treePos }     
                        
                        #else
                                object { tree3 rotate <gr(1), gr(360), gr(1)> translate <0, -(0.1+gr(0.25)), 0> scale treeScale+gr(treeScale) translate treePos }                        
                        #end
                #end   
                #declare treeX=treeX+treeSpace+gr(treeSpace);
        #end   
        #declare treeZ=treeZ+treeSpace+gr(treeSpace);   
        
#end
        
        #warning concat(str(treeCount,5,5), " Trees added")
#end

union
{

        placeTrees( union { object { farLand} object { middleLand } object { nearLand } }, <-12000, 0, 2000>, <-2500, 0, 12000>, 150, 35) 
        placeTrees( union { object { farLand} object { middleLand } object { nearLand } }, <-6000, 0, 13000>, <500, 0, 18000>, 110, 35)         
        placeTrees( union { object { farLand} object { middleLand } object { nearLand } }, <3500, 0, 8000>, <15000, 0, 18000>, 110, 35)         
        placeSmallTrees( farLand , <-1000, 0, 18000>, <5000, 0, 20000>, 35, 7)         


}

#end



// Sun
#declare sunPos = <0, 500000, 0>;
#declare sunRot = <82, 10, 0>;   


#if(showClouds)    

#declare cloudAmbient = 0.1;
#declare cloudDiffuse = 0.9;
#declare cloudBrilliance = 0.15;

light_group {


light_source
{
                
                        0
                        color rgb <5.0, 2, 0.5>*60
                        area_light
                        <20000, 0, 0> <0, 0, 20000> // lights spread out across this distance (x * z)
                        10, 10                // total number of lights in grid (4x*4z = 16 lights)
                        adaptive 0          // 0,1,2,3...
                        jitter              // adds random softening of light
                        circular            // make the shape of the light circular
                        orient              // orient light 
                        fade_distance sunPos.y/10
                        fade_power 2

looks_like {
sphere // transparent sphere containing media
 { 0,1 pigment { rgbt 1 } hollow
   interior
   { media
     { emission 1
       density
       { spherical poly_wave 10 density_map
         { 
                [0 rgb 0]
                [1 rgb <1, 0.5, 0.05>]
         }
       }
     }
  }  
 } 
 }   
  scale 20000
  translate sunPos
  rotate sunRot
} 


plane {
  y, // <X Y Z> unit surface normal, vector points "away from surface"
  50000 // distance from the origin in the direction of the surface normal
  hollow on // has an inside pigment?
        texture 
        {
                pigment 
                {

                        wrinkles  
                        turbulence <0.375, 0.27, 0.1>    
                        octaves 10
                        lambda 3.5
                        omega 0.65     
                        scallop_wave
                        color_map
                        {

                                [ 0.0 color rgbt <1, 1, 1, 1> ]
                                [ 0.935 color rgbt <1, 1, 1, 1> ] 
                                [ 0.96 color rgbt <2, 2, 2, 0.985> ] 
                                [ 0.98 color rgbt <2, 2, 2, 0.965> ] 
                                [ 1.0001 color rgbt <2, 2, 2, 0.915> ]                                     
                        }       
                        rotate <0, 20, 0>
                        scale <2000000, 500, 5000000>
                }
                finish 
                {
                        ambient cloudAmbient 
                        diffuse cloudDiffuse
                        brilliance cloudBrilliance
                }   
                
        }  
        no_shadow       
        rotate <0.5, 0, 0>
   
                
}


plane {
  y, // <X Y Z> unit surface normal, vector points "away from surface"
  52000 // distance from the origin in the direction of the surface normal
  hollow on // has an inside pigment?
        texture 
        {
                pigment 
                {

                        bozo  
                        turbulence <0.325, 0.32, 0.1>    
                        octaves 10
                        lambda 3.5
                        omega 0.75     
                        scallop_wave
                        color_map
                        {

                                [ 0.0 color rgbt <1, 1, 1, 1> ]
                                [ 0.925 color rgbt <1, 1, 1, 1> ] 
                                [ 0.95 color rgbt <2, 2, 2, 0.935> ] 
                                [ 0.975 color rgbt <2, 2, 2, 0.985> ] 
                                [ 1.0001 color rgbt <2, 2, 2, 0.915> ]                                     
                        }       
                        rotate <0, 15, 0>
                        scale <10000000, 500, 5000000>
                }
                finish 
                {
                        ambient cloudAmbient 
                        diffuse cloudDiffuse
                        brilliance cloudBrilliance
                }   
                
        }  
        no_shadow       
        rotate <0.5, 0, 0>
   
                
} 

plane {
  y, // <X Y Z> unit surface normal, vector points "away from surface"
  45000 // distance from the origin in the direction of the surface normal
  hollow on // has an inside pigment?
        texture 
        {
                pigment 
                {

                        bozo  
                        turbulence <0.325, 0.32, 0.1>    
                        octaves 10
                        lambda 3.5
                        omega 0.75     
                        scallop_wave
                        color_map
                        {

                                [ 0.0 color rgbt <1, 1, 1, 1> ]
                                [ 0.925 color rgbt <1, 1, 1, 1> ] 
                                [ 0.95 color rgbt <2, 2, 2, 0.935> ] 
                                [ 0.975 color rgbt <2, 2, 2, 0.985> ] 
                                [ 1.0001 color rgbt <2, 2, 2, 0.915> ]                                     
                        }       
                        rotate <0, 15, 0>
                        scale <100000000, 500, 50000000>
                }
                finish 
                {
                        ambient cloudAmbient 
                        diffuse cloudDiffuse
                        brilliance cloudBrilliance
                }   
                
        }  
        no_shadow       
        rotate <0.5, 0, 0>
   
                
}




        
} 

		glow {
			type 2
			location sunPos/50 //vrotate(<0, 10000, 0>, <80, 0, 0>)
			radius 5000
			fade_power 0.0125
			color	rgb <2, 1,0.05>*5    
                        rotate sunRot
		}     
		glow {
			type 2
			location sunPos/50 //vrotate(<0, 10000, 0>, <80, 0, 0>)
			radius 6000
			fade_power 0.011
			color	rgb <3,1.5,0.1>*5    
                        rotate sunRot
		}     
		glow {
			type 2
			location sunPos/50 //vrotate(<0, 10000, 0>, <80, 0, 0>)
			radius 7000
			fade_power 0.01
			color	rgb <2, 1,0.1>*5    
                        rotate sunRot
		}     


                
#end


light_source
{
                
                        0
                        color rgb <5.0, 3, 1>/2 
                        area_light
                        <2000, 0, 0> <0, 0, 2000> // lights spread out across this distance (x * z)
                        10, 10                // total number of lights in grid (4x*4z = 16 lights)
                        adaptive 0          // 0,1,2,3...
                        jitter              // adds random softening of light
                        circular            // make the shape of the light circular
                        orient              // orient light 
                        fade_distance sunPos.y
                        fade_power 2
  translate sunPos/5
  rotate sunRot
}    




// Fence macro

#declare fenceRand = seed(1452);   

// Macro to return a random number

#macro fr(num)
        rand(fenceRand)*num
#end


#macro buildFence(landObject, startPos, endPos, postHeight, postWidth, postDepth, sections, plankHeight, plankDepth, numPlanks, plankBottom, plankTop)

        #local fenceLength=vlength(endPos-startPos);
        #local sectionLength=fenceLength/sections;      
        #local dropHeight=max_extent(landObject); 
        #local xStep = (endPos.x-startPos.x)/sections;
        #local zStep = (endPos.z-startPos.z)/sections;
        #local rotation=degrees(asin((endPos.z-startPos.z)/-fenceLength));
        
        #local sectionCount=0;   
        #local lastPos=<0, 0, 0>;    
        #local startPlanks=false;
        #local currentPostPos=startPos; 
        #local plankTopPos=postHeight-plankTop;
        #local plankSpace=(postHeight-plankTop-plankHeight-plankBottom-postDepth)/(numPlanks-1);   
        union
        {
        #while(sectionCount < sections)   
                #local postHit = <0, 0, 0>;
                #local postPos = trace(landObject, <currentPostPos.x, dropHeight.y+10, currentPostPos.z>, <0, -1, 0>, postHit);  
        
                #if(vlength(postHit) > 0) // hit   
                        #if(isoFence)
                        isosurface 
                        {       
                                #local isoOff=fr(0.15)+1;
                                function{f_rounded_box(x,y,z,postWidth/(10*isoOff), postWidth/(2*isoOff), postHeight/(2*isoOff), postWidth/(2*isoOff))+f_wrinkles(x/2,y/2,z/2)*0.2}
                                threshold 0
                                contained_by {box {<-postWidth,-postHeight,-postWidth> <postWidth,postHeight,postWidth>} }
                                accuracy 0.005
                                max_gradient 1.586   
                                                            
                                translate <0, postHeight/2, 0>
                        #else
                        box 
                        { 
                                <-postWidth/2, 0, -postWidth/2> <postWidth/2, postHeight, postWidth/2>                          
                        #end
                                
                                pigment {
                                 rgb <0.06, 0.02, 0> }
                                finish { ambient 0 diffuse 0.5 } 
                                rotate <0, rotation, -1+fr(2)>
                                translate <postPos.x, postPos.y-postDepth, postPos.z> 
                        }    
                        // planks
                        #if(startPlanks)    
                                #local plankY=plankBottom;
                                #local plankCount=0;
                                        #while(plankCount < numPlanks)
                                        #local plankLength=vlength(postPos-lastPos); 
                                        #local plankRotation=degrees(asin((postPos.y-lastPos.y)/plankLength));
                                                #if(isoFence)
                                                isosurface 
                                                {       
                                                        #local isoOff=fr(0.2)+1;                                 // was length +(postWidth/(3*isoOff))
                                                        function{f_rounded_box(x,y,z,postWidth/(20*isoOff), plankLength/2, plankHeight/(2*isoOff), plankDepth/(2*isoOff))+f_bozo(x/20,y*2,z*10)*0.05+f_wrinkles(x*0.1, y*2, z*10)*0.05}
                                                        threshold 0         
                                                        all_intersections
                                                        contained_by {box {<-(plankLength+(postWidth*2)),-plankHeight,-plankDepth> <(plankLength+postWidth*4),plankHeight,plankDepth>} }
                                                        accuracy 0.005
                                                        max_gradient 2.586                               
                                                        translate <(plankLength/2), plankHeight/2, -(postWidth+(plankDepth/2))>
                                                #else
                                                box 
                                                { 
                                                        <-(postWidth/2), 0, -(postWidth+plankDepth)> <plankLength+(postWidth/2), plankHeight, -postWidth> 
                                                #end
                                                        rotate <0, 0, plankRotation-1+fr(2)>
                                                        rotate <0, rotation, 0>  
                                                        translate <lastPos.x-(postWidth/3), lastPos.y+plankY+fr(plankHeight/6), lastPos.z>
                                                pigment { rgb <0.06, 0.02, 0> }
                                                finish { ambient 0 diffuse 0.5 } 
                                                    
                                        }
                                        #local plankY=plankY+plankSpace;   
                                        #local plankCount=plankCount+1;
                                #end
                        #end
                        #local lastPos=postPos;
                #end        
                
                #local startPlanks=true;
                #local currentPostPos = <currentPostPos.x+xStep, currentPostPos.y, currentPostPos.z+zStep>;
                #local sectionCount=sectionCount+1;
        #end
        }
#end



#if(showFence)
#declare fence = object { buildFence(nearLand  , <-200, 0, 80>, <200, 0, 240>, 30,         2.75,       2,         9,       4.5,           0.35,         2,         6,           6) }


object { fence }

object { Place_Snow( fence, 0.275, 0.075, 1.7, <0.1, -1, 0.1>, 0.6, 0.1, LOAD_SNOW_DATA, "fenceSnow.txt") 
 texture { snowTex }
}  

#end
 

