
// Glenn's POV-Ray Scene File - Brick Maker
// created 11-Dec-00
// last update 2-Apr-06
// Copyright 2000-2006 Glenn McCarter (glenn@mccarters.net). Some rights reserved.
// This work is licensed under the Creative Commons Attribution License
// To view a copy of this license, visit http://creativecommons.org/licenses/by/2.0/
//
// Makes a pretty good brick wall
// To change the character of the bricks,
//  - lighten or darken the mortar to offset it
//  - reduce range of brick HF colors to make a more even wall
//  - change superellipsoid parameter to make more square edges

//#include "Colors.inc"
#include "GIncludeRH.inc"
//#include "GRuler.inc"


global_settings {
	adc_bailout 0.003922
	ambient_light <1.0,1.0,1.0>
	assumed_gamma 1.0
	hf_gray_16 off
	irid_wavelength <0.25,0.18,0.14>
	max_intersections 64
	max_trace_level 10
	number_of_waves 10
}

//===============================================
//             Variables
//===============================================

#declare WallLength = 0.8*Feet;
//#declare WallHeight = WallLength/1.33333; // this allows standard camera to capture full wall without distortion
#declare WallHeight = 8*Feet; // use this for free-aspect-ratio walls (will need to stretch image maps accordingly)

#declare BrickLength = 7.5;
#declare BrickHeight = 2.25;
#declare MortarThickness = 0.5;

// Image must be rendered twice.
// Once for the color map, once (in greyscale) for the height field
#declare    Color_On = 0; // turn on color version for image map
#declare       HF_On = 1; // turn on greyscale version for heightfield generation
#declare Material_On = 0; // turn on solid-color version for material map (DO NOT USE YET)

#declare RS = seed(48832); // randomize brick colors
#declare RS2 = seed(20984); // randomize brick heights

//===============================================
//             Cameras
//===============================================

#declare Closeup_Cam =
	camera {
		perspective
		direction <0.0,     1.0,  0.0>
		sky       <0.0,     0.0,  1.0>  // Use right handed-system
		up        <0.0,     0.0,  1.0>  // Where Z is up
		right     <1.33333, 0.0,  0.0>
//	angle 6
	angle 45
//	location  <0.01*Feet, -30*Feet, 0.01*Feet>
	location  <1.01*Feet, -3*Feet, 3.01*Feet>
//	location  <8.01*Feet, -.61*Feet, 4.01*Feet>
	look_at <1*Feet,0,8>
//	location  <WallLength/2, -30*Feet, WallHeight/2>
//	look_at <WallLength/2, 0, WallHeight/2>
}

#declare WL=WallLength;
#declare WH=WallHeight;
#declare BL=BrickLength;
#declare BH=BrickHeight;
#declare MT=MortarThickness;

#declare CamRight = div(WL,BL+MT)*(BL+MT)+BL/2;
#declare CamUp = div(WH,(BH+MT)*2)*(BH+MT)*2+BH*2+MT+MT;

#declare Ortho_Cam =
	camera {
		orthographic

		sky       <0.0,     0.0,  1.0>  // Use right handed-system
//		up        <0.0,     0.0,  WallHeight>  // use for 4:3 walls
		up        <0.0,     0.0,  CamUp>  // use for full walls
		right     <CamRight, 0.0,  0.0>
//		up        <0.0,     0.0,  WallHeight/20>  // test small patch
//		right     <WallLength/20, 0.0,  0.0>
//		location <CamRight/2, -30*Feet,WallHeight/2> // use for 4:3 walls
//		look_at <CamRight/2, 0, WallHeight/2>
		location <CamRight/2, -30*Feet,CamUp/2> // use for full walls
		look_at <CamRight/2, 0, CamUp/2>
	}

camera {
//	Closeup_Cam // for testing only
	Ortho_Cam // for principal renders
  }


//===============================================
//             Textures
//===============================================

#declare PlainTex =
	texture {
		pigment { color rgb <1,.5,.4> }
	}
#declare DarkTex =
	texture {
//		pigment { color rgb <.004,.004,.004> }
		pigment { color rgb <.004,1.004,.004> } // test
		finish { ambient 1 diffuse 0}
	}
#declare SimpleBrickTex =
	texture {
		pigment {
			brick
				color rgb <0.812,0.776,0.718> // morter
				color rgb <0.553,0.204,0.180> // brick
				brick_size <8,3,4.5>
				mortar 0.4
				rotate x*90
		}
		finish { ambient 0.1 diffuse 0.6 specular 0.3 roughness 0.2 }
	}


#macro RandomBrickColorTex()
	#declare Brightness = 0.215+rand(RS)*0.25; // range 0.2 to 1.0
	#declare Redness = 0.50+rand(RS)*0.20; // range 0.4 to 0.6
	#local Greenness = 0.18+rand(RS)*0.05+Redness*0.12;
	#local Blueness = 0.01+rand(RS)*0.02+Redness*0.1;
//	pigment { color rgb<Redness*Brightness,0.21*Brightness,0.08*Brightness> }
	pigment { color rgb<Redness,Greenness,Blueness>*Brightness }
	finish { ambient 1 diffuse 0}
#end
#macro RandomBrickMaterialTex()
	#declare Brightness = rand(RS)*0.8+0.2; // range 0.2 to 1.0
	pigment { color rgb<1,1,1>*Brightness }
	finish { ambient 1 diffuse 0}
#end

#declare MortarColorTex =
	texture {
		pigment {
			bozo
			color_map {
				[ 0.00 color rgb <0.878,0.660,0.344> ]
				[ 1.00 color rgb <0.733,0.567,0.330>*0.5 ]
			}
			scale 1
		}
	}

#declare MortarHFTex =
	texture {
		pigment {
			bozo
			color_map {
				[ 0.00 color rgb <1,1,1>*0.82 ]
				[ 1.00 color rgb <1,1,1>*0.33 ]
			}
			scale <5,1,3>
		}
		finish { ambient 1.4 diffuse 0 }
	}
#declare MortarBumpHFTex =
	texture {
		pigment {
			granite
			color_map {
//				[ 0.13 color rgbt<0.082,0.082,0.082,0.2> ]
				[ 0.13 color rgbt<0.082,0.082,0.082,0.72> ]
				[ 0.960 color rgbt<0.066,0.066,0.066,1.0> ]
			}
//			scale .36
			scale .6
		}
		finish { ambient 1.4 diffuse 0 }
	}

#declare GradientYHFTex =
	texture {
		pigment {
			gradient y
			color_map {
				[ 0.0 color rgb<1,1,1> ]
				[ 1.0 color rgb<0,0,0> ]
			}
		}
		finish { ambient 1.0 diffuse 0 }
	}


//===============================================
//             Lighting
//===============================================

#declare MyLight =
	light_source {
		<0,0,0>
		color rgb<1,1,1>
//		fade_distance 12*Feet
//		fade_power 1
	}

#if ( Color_On=1 )
	object { MyLight translate <WallLength/2, -1000*Feet, WallHeight/2> } // render
#end

//===============================================
//             Environment
//===============================================

background { color rgb <0,0,0> }
//background { color rgb <0.93, 0.99, 0.4> } // bright yellow

//===============================================
//             Objects
//===============================================

#declare RedSph =
	sphere {
		<0,0,0>,.5
		texture { PlainTex }
		translate <0,0,0.5>
	}

#declare RectBrick =
	box {
		<0,0,0>,<7.5,3.5,2.25>
		texture { PlainTex }
	}

#declare SimpleBricks =
	box {
		<0,0,-1>,<WallLength,1,WallHeight>
		texture { SimpleBrickTex }
	}

#macro BrickHFTex(ZTO)
	#local RTS=seed(ZTO*999);
	#local WD=.30+rand(RTS)*.25; // Wave Depth (0=deep waves, 1=none)
	#local WS=.50+rand(RTS)*4.0; // Wave Size (0=small waves, 1=large)
	#local BD=.45+rand(RTS)*.30; // Bump Depth (0=deep pits, 1=none)

	texture { GradientYHFTex scale <1,0.7,1> translate <0,-0.4-rand(RTS)*0.15,0> }

	texture{
		pigment{
			bozo
			color_map {
				[ 0.2 rgbt<.01,.01,.01,WD> ]
				[ 0.5 rgbt<.3,.3,.3,.93> ]
			}
			turbulence 0.33
			scale WS
			translate <ZTO*999,0,0>
		}
		finish{ambient 1 diffuse 0}
	}

	texture{
		pigment{
			dents
			color_map {
				[ 0.05 rgbf<BD,BD,BD,.83> ]
				[ 0.15 rgbt<1,1,1,1> ]
			}
			turbulence 0.33
			scale .082
			translate <ZTO*999,0,0>
		}
		finish{ambient 1 diffuse 0}
	}

#end

#macro BrickShape(RSeed,BrickLength,BrickHeight) // origin at center
	superellipsoid {
		<0.2,0.1>
		scale <BrickLength/2,3.5/2,BrickHeight/2> // brick dimensions
		#if ( HF_On = 1 )
			BrickHFTex(RSeed)
		#end
		#if ( Material_On = 1 )
			RandomBrickMaterialTex()
		#end
		#if ( Color_On = 1 )
			RandomBrickColorTex()
		#end
	}
#end

#macro HalfBrickShape(RSeed,BrickLength,BrickHeight) // origin at center
	superellipsoid {
		<0.2,0.1>
		scale <BrickLength/4,3.5/2,BrickHeight/2> // brick dimensions
		#if ( HF_On = 1 )
			BrickHFTex(RSeed)
		#end
		#if ( Material_On = 1 )
			RandomBrickMaterialTex()
		#end
		#if ( Color_On = 1 )
			RandomBrickColorTex()
		#end
	}
#end
#macro WallMortar(WML,WMH) // WallMorterLength,WallMorterHeight
	box {
		<0,0,0>,<WML,1,WMH>
		translate <0,0.4,0>
		#if ( HF_On = 1 )
			texture { MortarHFTex }
			texture { MortarBumpHFTex }
		#end
		#if ( Material_On = 1 )
			texture { pigment { rgb 0.1 } finish { ambient 1 diffuse 0 } }
		#end
		#if ( Color_On = 1 )
			texture { MortarColorTex }
		#end
	}
#end

#macro BrickWall(WL,WH,BL,BH,MT,RS2) // WallLength, WallHeight,BrickLength,BrickHeight,MortarThickness,RandomSeed
// Note, the wall length and height are merely "suggestions",
// the acaual wall will always be larger --
// The width will have an integral number of full&half bricks
// and the height will have an integral number of double-rows
	#declare BRS = seed(RS2); // change this to randomize the brick heights
//	#local BL = 7.5; // BrickLength Test
//	#local BH = 2.25; // BrickHeight Test
	union {
		#local TDRH = 0; // ThisDoubleRowHeight
		#local RowCount =0;
		#while (TDRH<WH)
			object { HalfBrickShape(rand(BRS),BL,BH) // first half-bricks
				translate <BL/4,1.5,BH/2+TDRH+(BH+MT)+MT>
			}
			object { HalfBrickShape(rand(BRS),BL,BH) // end half-bricks
				translate <div(WL,(BL+MT))*(BL+MT)+BL/4,1.5,BH/2+TDRH+MT>
			}
			#local TBP = 0; // ThisBrickPosition
//			#while (TBP < WL )
			#while (TBP < div(WL,(BL+MT))*(BL+MT)-BL/2 )

				object { BrickShape(rand(BRS),BL,BH) // even rows
					translate <TBP+BL/2,1.5,BH/2+TDRH+MT> 
				}
				object { BrickShape(rand(BRS),BL,BH) // odd rows
					translate <TBP+BL+MT,1.5,BH/2+TDRH+(BH+MT)+MT> 
				}

				#local TBP=TBP+BL+MT;
			#end

			#local TDRH=TDRH+(BH+MT)*2;
			#local RowCount = RowCount + 1;
		#end
		object { WallMortar( div(WL,BL+MT)*(BL+MT)+BL/2-MT*2,div(WH,(BH+MT)*2)*(BH+MT)*2+BH*2+MT ) translate <MT,0,0> }
//		#if ( HF_On = 1 ) // provides corner for mortar
//			box { <MT-0.1,2,0><div(WL,BL+MT)*(BL+MT)+BL/2-MT+0.15,3,div(WH,(BH+MT)*2)*(BH+MT)*2+BH*2+0.15> texture { DarkTex } }
//		#end
	} // end union
#end


//===============================================
//             Object Placement
//===============================================

//object { TestSphere scale 2 translate <0,-2,1> }
//object { SimpleBricks translate <0,0,0> }

//object { BrickWall1 }
//object { WallMortar(1,2) }
#declare RS=seed(19795);

//object { BrickShape(rand(RS)*15) translate <7.5/2,3.5/2-0.1,2.25/2> }
//object { RectBrick translate <0,0,0> }

//object { BrickWall(2*Feet,1.4*Feet,0.5) }

//object { BrickShape(rand(RS)*15) translate <8,0,0> }
//object { BrickShape(rand(RS)*15) translate <-8,0,2.5> }
//object { BrickShape(rand(RS)*15) translate <0,0,2.5> }
//object { BrickShape(rand(RS)*15) translate <8,0,2.5> }

/*
object { BrickShape(21993) translate <-8,0,5> }
object { BrickShape(49323) translate <0,0,5> }
object { BrickShape(91943) translate <8,0,5> }
*/

//box { <0,0.5,0>,<80,40,80> texture { PlainTex } }
object { BrickWall(WallLength,WallHeight,BrickLength,BrickHeight,MortarThickness,RS2) }