

/*_______________________________________________________________________________________________________*\

FILE NAME: Make_Flower_Stalk.inc
CREATED BY: Isaac Kulka, Copyright 12-28-2006
NOTES: Created for the Internet Raytracing Competition's Nov-Dec 2006 Contest: 'Complexity' (www.irtc.org)

DESCRIPTION: The macros included in this file can be used to create tall stalks covered with flowers.

\*_______________________________________________________________________________________________________*/




  //                                                            \\
 //------------------POPULATE FLOWER STALK ARRAY-----------------\\
//                                                                \\

// Use this macro to create an array of unique flower stalks.  NOTE: before running this macro you must run the Populate_FS_Petal_Array(), Populate_FS_Flower_Array(), and Populate_FS_Bud_Array() macros first, as they generate global objects which this macro depends upon.


#macro Populate_Flower_stalk_Array(FLOWER_STALK_CNT, VARIETY, HEIGHT, RADIUS, SMOOTHNESS, BEND, STEM_COLOR, FILTER, LUSTER, AMBIENT, SEED, FLOWER_DIST, FLOWERINESS)

/**********   VARIABLES   **************\
FLOWER_STALK_CNT - Integer - total number of unique flower_stalk to generate in the array
VARIETY -     0-1 -     determines how much certain values can deviate below the maximum specified values
HEIGHT -      Float -   maximum height of each plant
RADIUS -      Float -   maximum thickness of each plant's stem  
SMOOTHNESS -  0-1 -     determines the number of triangles in the mesh (a higher value means more triangles)
BEND -        Radian -  maximum bend of the stem around the z-axis
STEM_COLOR -  <r,g,b> -     color of the stem
FILTER -      0-1 -     filter of the stem
LUSTER -      0-1 -     specular highlight of the stem
AMBIENT -     0-1 -     ambient value of the stem
SEED -        Integer - random seed
FLOWER_DIST - Float -   the distance along the stem between flowers (this value may also be thought of as "flower density")
FLOWERINESS - 0-1 -     percentage of the stem covered in flowers, starting from the top on down
\***************************************/   


#local R1 = seed(SEED);


//these are global variables, intended to be called from the main scene in which these plants are used (i.e. Grassland_Foreround.pov)
#declare FLOWER_STALK_ARRAY = array[FLOWER_STALK_CNT];
#declare FLOWER_STALK_MAX = FLOWER_STALK_CNT;

#declare STEM_COLOR = STEM_COLOR;
#declare FILTER = FILTER;
#declare LUSTER = LUSTER;
#declare AMBIENT = AMBIENT;
#declare FLOWER_DIST = FLOWER_DIST;
#declare FLOWERINESS = FLOWERINESS;

//generate the unique plants
#local i = 0;
#while(i < FLOWER_STALK_CNT)
  
  #local Var = rand(R1)*VARIETY;
  
  #local Height = HEIGHT*(rand(R1)*Var + (1-Var)); 
  #local Bend = BEND*(rand(R1)*Var + (1-Var));
  
  #local Seed = floor(rand(R1)*10000);
   
  #declare FLOWER_STALK_ARRAY[i] = 
    Make_Flower_Stalk(Height, RADIUS, Bend, Seed)
  
  
  #local i = i + 1;

#end


#end //Populate_Flower_stalk_Array()  






  //                                                            \\
 //--------------POPULATE FLOWER STALK FLOWER ARRAY--------------\\
//                                                                \\

//This macro creates five-petaled flowers.  This macro must be run before the Populate_Flower_Stalk_Array() macro.

#macro Populate_FS_Flower_Array(FLOWER_CNT, SEED)


/**********   VARIABLES   **************\
FLOWER_CNT -  Integer - number of unique flowers to generate in the array
SEED -        Integer - random seed to be used with this macro
\***************************************/



#local R1 = seed(SEED);

//these are global variables, intended to be called from the main scene in which these flowers are used
#declare FLOWER_STALK_FLOWER_ARRAY = array[FLOWER_CNT];
#declare FLOWER_STALK_FLOWER_MAX = FLOWER_CNT;

//generate the flowers
#local j = 0;
#while(j < FLOWER_CNT)

  #declare FLOWER_STALK_FLOWER_ARRAY[j] =
  union
  {    
    #local bend = rand(R1)*20;
    #local Petals = 5;
    
    //add petals around the flower's head
    #local i = 0;
    
    #while(i < Petals)
      
      #local i_ratio = i/Petals;
      
      object
      {
        FLOWER_STALK_PETAL_ARRAY[floor(rand(R1)*FLOWER_STALK_PETAL_MAX)]  
        //translate<0,0,0>
        rotate x*(-85 + rand(R1)*30)
        rotate y*(i_ratio*360 + rand(R1)*10)
      }
      
      #local i = i + 1;
    #end
  }

  #local j = j + 1;
#end


#end //Populate_FS_Flower_Array()








      
      
      
      
                              

  //                                                            \\
 //--------------------MAKE FLOWER STALK PETAL-------------------\\
//                                                                \\

// This macro is called by the Populate_FS_Petal_Array() macro to create petals for the FLOWER_STALK_PETAL_ARRAY.

#macro Make_FS_Petal(HEIGHT, WIDTH, BEND, RUFFLE, COLOR, SEED)

/**********   VARIABLES   **************\
HEIGHT -      Float -   length of the petal
WIDTH -       Float -   width of petal at it widest point
BEND -        Radian -  amount the petal curls around the x-axis
RUFFLE -      Float -   amount the petal curves out in the z-direction along its width from the crease
COLOR -       <r,g,b> - color of the petal
SEED -        Integer - random seed
\***************************************/


#local R1 = seed(SEED);

  mesh
  {
    #local RUFFLE = max(RUFFLE, 0.2); //the ruffle cannot go below 0.2 (no flat petals)
    
    #local h_step = HEIGHT*(1-SMOOTHNESS)*0.1;
    #local h = 0;
    
    #local points = max(3, ceil(WIDTH/h_step));
    
    #local lower = array[points];
    #local upper = array[points];
    
    #local rpx = rand(R1)*1000;
    #local rpy = rand(R1)*1000;
    #local rpz = rand(R1)*1000;
    
    #local petal_wrinkle = 0.2; //determines the wrinkliness of the petal
    
    #local i = 0;
    #while(i < points)
      
      #local lower[i] = <0,0,0>;
      
      #local i = i + 1;
    #end
    
    
    #while(h < HEIGHT)
      
      #local h_ratio = h/HEIGHT;
      
      #local rad = sqrt(pow(WIDTH,2)*(1-(pow(h_ratio-0.5,2)/pow(0.5,2))));
      #if(h+h_step >= HEIGHT) #local rad = 0; #end
            
      #local i = 0;
      #while(i < points)
        #local i_ratio = i/points;
        
        
        #local z_bend = BEND * h_ratio;
        
        #local pp = <rad*i_ratio - rad/2,      cos(z_bend)*(h + HEIGHT*0.15) + cos(z_bend)*((1-h_ratio) * rad*RUFFLE*(1-h_ratio)*abs(sin(i_ratio*pi*2))),      sin(z_bend)*(h + HEIGHT*0.15) + sin(z_bend)*((1-h_ratio) * rad*RUFFLE*(1-h_ratio)*abs(sin(i_ratio*pi*2)))>;
        
        #local upper[i] = <pp.x,      pp.y + LS_PETAL_PATTERN(pp.x,pp.y,pp.z)*petal_wrinkle*HEIGHT,      pp.z + LS_PETAL_PATTERN(pp.x+rpx,pp.y+rpy,pp.z+rpz)*petal_wrinkle*HEIGHT>;
        
        #local i = i + 1;
      #end
      
      
      #local i = 0;
      #while(i < points-1)
             
        triangle{upper[i], lower[i], lower[i+1]}
        triangle{upper[i+1], upper[i], lower[i+1]}
        
        #local i = i + 1;
      #end
      
      
      #local i = 0;
      #while(i < points)
        
        #local lower[i] = upper[i];
        
        #local i = i + 1;
      #end
      
      #local h = h + h_step;
    #end
    
   
  
   
   pigment{COLOR filter FILTER} 
   finish{ambient AMBIENT diffuse 0.7 specular LUSTER roughness 0.6}
   
   double_illuminate
    
   rotate x*90
  }

#end //Make_FS_Pedal()









  //                                                            \\
 //---------------POPULATE FLOWER STALK PETAL ARRAY--------------\\
//                                                                \\

// Use this macro to create an array of unique flower petals for your flower stalks.  Once this macro has been run, you should use the Populate_FS_Flower_Array() macro to create flowers from your petals
  
  
#macro Populate_FS_Petal_Array(PETAL_CNT, HEIGHT, THICKNESS, VARIETY, BEND, RUFFLE, SMOOTHNESS, COLOR_1, COLOR_2, C_VAR, FILTER, LUSTER, AMBIENT, SEED)

/**********   VARIABLES   **************\
PETAL_CNT -   Integer - number of unique petals to generate
HEIGHT -      Float -   maximum length of each petal
THICKNESS -   Percent - determines the width of petal as a percentage of the height
VARIETY -     0-1 -     determines how much certain values can deviate below the maximum specified values 
BEND -        Radian -  maximum amount a petal curls around the x-axis
RUFFLE -      Float -   maximum amount a petal curves out in the z-direction along its width from the crease
SMOOTHNESS -  0-1 -     determines the number of triangles in the mesh   
COLOR_1 -     rgb -     primary color of the petals
COLOR_2 -     rgb -     secondary color of the petals
C_VAR -       0-1 -     amount by which the primary color may deviate to the secondary color with any given petal 
FILTER -      0-1 -     filtered light value
LUSTER -      0-1 -     specular highlight
AMBIENT -     0-1 -     ambient value
SEED -        Integer - random seed
\***************************************/


#local R1 = seed(SEED);

//these two values will be used to control just how much these petals differ from one another
#local INV_VAR = 1-VARIETY; 
#local INV_VAR2 = 1-VARIETY*3;

//global values
#declare FLOWER_STALK_PETAL_ARRAY = array[PETAL_CNT];    
#declare FLOWER_STALK_PETAL_MAX = PETAL_CNT;

#declare SMOOTHNESS = SMOOTHNESS;
#declare FILTER = FILTER;
#declare LUSTER = LUSTER;
#declare AMBIENT = AMBIENT;


//randomly generate values and call Make_FS_Petal() to populate the FLOWER_STALK_PETAL_ARRAY with 
#local i = 0;
#while(i < PETAL_CNT)
  
  #local Height = HEIGHT*(rand(R1)*VARIETY + INV_VAR);
  #local Width = THICKNESS*(rand(R1)*VARIETY + INV_VAR)*Height;
  
  #local Bend = BEND*(rand(R1)*VARIETY*3 + INV_VAR2);
  #local Ruffle = RUFFLE*(rand(R1)*VARIETY*3 + INV_VAR2);
  
  #local C_Var = rand(R1)*C_VAR;
  #local Color = rgb<COLOR_1.red*(1-C_Var) + COLOR_2.red*(C_Var), COLOR_1.green*(1-C_Var) + COLOR_2.green*(C_Var), COLOR_1.blue*(1-C_Var) + COLOR_2.blue*(C_Var)>;
     
  #local Seed = rand(R1)*1000;
  
  #declare FLOWER_STALK_PETAL_ARRAY[i] = Make_FS_Petal(Height, Width, Bend, Ruffle, Color, Seed);
  
  
  #local i = i + 1;
#end


#end //Populate_FS_Petal_Array()













  //                                                            \\
 //----------------MAKE FLOWER STALK BUD-------------------------\\
//                                                                \\

// This macro is called by the Populate_FS_Bud_Array() macro in order to create buds for the FLOWER_STALK_BUD_ARRAY.


#macro Make_FS_Bud(HEIGHT, RADIUS, COLOR) 
  
/**********   VARIABLES   **************\
HEIGHT -      Float -   height of the bud
RADIUS -      Float -   radius of the bud at its widest point
COLOR -       <r,g,b> - color of the bud
\***************************************/ 


#local h_step = (1-max(SMOOTHNESS,0.7))/2;
#local rad = 0; //radius of the current cross section in the mesh
#local h = h_step;
#local theta = 0; //rotation around the radius of the current cross section of the mesh
#local points = ceil(HEIGHT/h_step); //number of points in each circular cross section of the mesh

#local upper = array[points];
#local lower = array[points];

//initialize all lower array points at <0,0,0>
#local i = 0;
#while(i < points)
  #local lower[i] = <0,0,0>;
  #local i = i + 1;
#end

mesh
{
  #while(h <= HEIGHT)
    
    #if(h + h_step <= HEIGHT)
      #local rad = sin(h*(pi/HEIGHT))*RADIUS;
    #else
      #local rad = 0;
    #end
    
    //define upper points  
    #local i = 0;
    #while(i < points)
      
      #local theta = (i/points)*(2*pi);
      
      #local upper[i] = <rad*cos(theta), h, rad*sin(theta)>;
    
      #local i = i + 1;
    #end
    
    //make triangles
    #local i = 0;
    #while(i < points)
      #local j = i+1;
      #if(j = points) #local j = 0; #end 
      
      triangle{upper[i],lower[i],lower[j]}
      triangle{upper[i],upper[j],lower[j]}
      
      #local i = i + 1;
    #end
    
    //copy upper array into the lower array
    #local i = 0;
    #while( i < points)
      
      #local lower[i] = upper[i];
      
      #local i = i + 1;
    #end
    
    #local h = h + h_step;
  #end   
                           
  //make one last tall narrow triangle. This gives the buds a fuzzy appearance at their tips
  #local dd = RADIUS * 0.25; //distance to offset this triangle by
  triangle{<upper[0].x-dd,upper[0].y-dd,upper[0].z>, <upper[0].x,upper[0].y+HEIGHT*0.25,upper[0].z>, <upper[0].x+dd, upper[0].y-dd,upper[0].z>}
                           
  pigment{COLOR filter FILTER}
  finish{ambient AMBIENT diffuse 1.00 specular LUSTER roughness 0.5}
}

#end //Make_FS_Bud()












  //                                                            \\
 //----------------------POPULATE BUD ARRAY----------------------\\
//                                                                \\

// Use this macro to generate round buds with pointed tips for your flower stalk.  This macro must be run before the Populate_Flower_Stalk_Array() macro.


#macro Populate_FS_Bud_Array(BUD_CNT, HEIGHT, RADIUS, SMOOTHNESS, R_SEED, COLOR, COLOR_VAR, FILTER, LUSTER, AMBIENT)

/**********   VARIABLES   **************\
BUD_CNT -     Integer - number of unique buds in the FLOWER_STALK_BUD_ARRAY
HEIGHT -      Float -   maximum height of any one bud
RADIUS -      Float -   maximum radius of any one bud
SMOOTHNESS -  0-1 -     determines the number of triangles in the bud meshes
R_SEED -      Integer - random seed
COLOR -       <r,g,b> - primary color of the buds
COLOR_VAR     0-1 -     percentage by which the color of any one bud may deviate from COLOR
FILTER -      0-1 -     filtered light value  
LUSTER -      0-1 -     specular highlight
AMBIENT -     0-1 -     ambient value
\***************************************/  


#local R1 = seed(R_SEED);

#declare FLOWER_STALK_BUD_ARRAY = array[BUD_CNT]
#declare FLOWER_STALK_BUD_MAX = BUD_CNT;

#declare SMOOTHNESS = SMOOTHNESS;
#declare FILTER = FILTER;
#declare LUSTER = LUSTER;
#declare AMBIENT = AMBIENT;


//create random, but similar looking buds
#local i = 0;
#while(i < BUD_CNT)
  #local Height = HEIGHT * (rand(R1)*0.3 + 0.7);
  #local Radius = RADIUS * (rand(R1)*0.3 + 0.7);
  #local Color =  rgb<COLOR.red*(rand(R1)*COLOR_VAR + (1-COLOR_VAR)), COLOR.green*(rand(R1)*COLOR_VAR + (1-COLOR_VAR)), COLOR.blue*(rand(R1)*COLOR_VAR + (1-COLOR_VAR))>;
  
  #declare FLOWER_STALK_BUD_ARRAY[i] = Make_FS_Bud(Height, Radius, Color)
                                  
  #local i = i + 1;
#end 


#end //Populate_FS_Bud_Array() 













  //                                                            \\
 //----------------------Make Flower Stalk-----------------------\\
//                                                                \\

// This macro is called by the Populate_Flower_Stalk_Array() macro.  It is used to create unique flower stalks for the global FLOWER_STALK_ARRAY.


#macro Make_Flower_Stalk(HEIGHT, RADIUS, BEND, SEED)                                                            

/**********   VARIABLES   **************\
HEIGHT -      Float -   height of the plant's stem
RADIUS -      Float -   radius of the stem  
BEND -        Radian -  bend of the stem around the z-axis
SEED -        Integer - random seed
\***************************************/      



#if(BEND = 0) //catch a bend of zero radians and fix it
  #local BEND = 0.0001;
#end


#local circumference = 2*pi*RADIUS; //the circular length around the stem
#local step = (1-SMOOTHNESS)/2; //distance between points in the mesh
#local points = ceil(circumference/step); //number of points per cross section

#local lower = array[points];                                                                          
#local upper = array[points];

#local x_bend_rad = HEIGHT/BEND; //the radius of the greater circle of which the bent stem is but an arc-length of

#local circlet_cnt = 4; //number of flowers in each circlet around the stem



union
{
  
  //CREATE THE STEM OF THE PLANT
  mesh
  {
    //initialize lower array
    #local i = 0;
    #while(i < points)
      #local theta = (i/points)*(2*pi);
      #local lower[i] = <RADIUS*cos(theta), 0, RADIUS*sin(theta)>;
      #local i = i + 1;
    #end
    
    //begin builing the stem
    #local h = step;
    #while(h <= HEIGHT)
      
      #local h_ratio = h/HEIGHT;
      
      #local rad = RADIUS*((1-(h/HEIGHT))*0.5 + 0.5);
      //generate new points for upper array
      #local i = 0;
      #while(i < points)
        #local theta = (i/points)*(2*pi);
        #local bend = (h/HEIGHT)*BEND;
        #local upper[i] = <rad*cos(theta)*cos(bend) + x_bend_rad*cos(bend) - x_bend_rad,    rad*cos(theta)*sin(bend) + x_bend_rad*sin(bend),    rad*sin(theta)>;
        #local i = i + 1;
      #end
      
      //create the triangles for this pass
      #local i = 0;
      #while(i < points)
        #local j = i+1;
        #if(j = points) #local j = 0; #end 
        
       
        
        triangle{upper[i],lower[i],lower[j]}
        triangle{upper[i],upper[j],lower[j]}
        
              
        #local i = i + 1;
      #end
      
      //set lower to upper
      #if(h + step <= HEIGHT)
      #local i = 0;
        #while( i < points)
          
          #local lower[i] = upper[i];
          
          #local i = i + 1;
        #end
      #end 
      
      
      #local h = h + step;
    #end
    
    pigment{STEM_COLOR filter FILTER}
    finish{ambient AMBIENT diffuse 0.85 specular LUSTER}
       
    }
    
  
  
  
  //ADD FLOWERS AND BUDS TO THE STEM
  #local bend_max = degrees(BEND); //the degree version of the radian value, BEND
  #local min_flower_height = HEIGHT*(1-FLOWERINESS); //minimum height along the stem where flowers can begin sprouting (derived from FLOWERINESS)
  #local h = min_flower_height; //current length value along the stem
  #local theta_start = 0; //this value is updated which each iteration of "h", it is used to wind the flowers around the stem so they don't appear to be lined up in vertical columns
  
  #while(h < HEIGHT)
        
    #local h_ratio = h/HEIGHT; //the percentage h is along the total height
    #local h_ratio2 = (h-min_flower_height)/(HEIGHT-min_flower_height); //the percentage h is along the total height that has flowers on it
    
    #local up_bend = (1-h_ratio2)*-70 - 20; //the number of degrees which a given circlet of flowers should bend up toward the vertical stem direction
    
    #local bend = h_ratio*BEND; //the current bend in radians along the stem for any given value of "h" (varies between zero and BEND)
    #local point = <x_bend_rad*cos(bend) - x_bend_rad, x_bend_rad*sin(bend), 0>; //the point on the stem for a given "h"
    
    #local bud_prob = h_ratio2-0.15; //probability that a bud will appear on the plant rather than a flower.  This probability increases as "h" approached the tip of the stem
    #local theta_start = theta_start + 25;
    
    #local j = 0;
    #while(j < circlet_cnt)
    
      #local theta = j/circlet_cnt * 360 + theta_start;
      
      object
      { 
        //randomly make a bud or flower based on bud_prob
        #if(rand(R1) < bud_prob)
          FLOWER_STALK_BUD_ARRAY[floor(rand(R1)*FLOWER_STALK_BUD_MAX)]
        #else
          FLOWER_STALK_FLOWER_ARRAY[floor(rand(R1)*FLOWER_STALK_FLOWER_MAX)]
        #end
        
        rotate y*rand(R1)*360 //randomly rotate the flower around its own y-axis
        
        rotate z*up_bend + rand(R1)*5 //rotate the flower so that it sticks out away from the stem, semi-perpendicularlly
                           
        rotate y*theta + rand(R1)*5 //rotate the flower around the stem so that each flower in the circlet is semi-evenly spaced from one another                  
                           
        rotate z*bend_max*h_ratio2 //rotate the flower so that it sticks out away from the stem, semi-perpendicularlly a second time
        
        #if(h_ratio2 < 0.10) scale 0.75 #end //make flowers at the bottom of the flower area slightly smaller (to give it a rounded look, kind of like a cat-tail)
        
        scale (1-h_ratio2)*0.25 + 0.75 //scale the flowers smaller as they approach the top
        
        translate <point.x+rand(R1)*0.01,  point.y+rand(R1)*(FLOWER_DIST/2),  point.z+rand(R1)*0.01> //put the flower where it needs to be
        
      }
     
      
      #local j = j + 1;
    #end
    
    #local h = h + FLOWER_DIST;
  #end
  
  //place one last bud at the very top of the stem
  object
  {     
    FLOWER_STALK_BUD_ARRAY[floor(rand(R1)*FLOWER_STALK_BUD_MAX)]
    
    scale 1.5                                                                        
    rotate z*degrees(BEND)        
    scale 0.75
    translate <x_bend_rad*cos(bend) - x_bend_rad, x_bend_rad*sin(bend), 0>      
  }
  
  
  }

    
#end //Make_Flower_Stalk()

          
          
          
          
          
          
                
          
          


//PETAL WRINKLE PATTERN
#declare LS_PETAL_PATTERN = 
function
{ 
  pattern
  { 
    agate
     
    lambda 0.8
    octaves 3
    omega 5  
    turbulence 0.1

     
    scale 11
    translate y*674
  } 
} 

