

/*_______________________________________________________________________________________________________*\

FILE NAME: Make_Grass.inc
CREATED BY: Isaac Kulka, Copyright 12-28-2006
NOTES: Created for the Internet Raytracing Competition's Nov-Dec 2006 Contest: 'Complexity' (www.irtc.org)

DESCRIPTION: The macros included in this file can be used to create individual blades of grass, as well as circular clusters of grass blades.

\*_______________________________________________________________________________________________________*/





                                                                                                                       //                                                            \\
 //----------------------MAKE GRASS BLADE------------------------\\
//                                                                \\


// This macro is called by the Populate_Grass_Array() macro to create blades of grass for various "grass arrays".

#macro Make_Grass_Blade(HEIGHT, WIDTH, SEED, RAD_MIN, RAD_MAX, X_OFFSET, Z_OFFSET, X_BEND, TWIST) 

/**********   VARIABLES   **************\
HEIGHT -      Float -   length of the blade of grass from base to tip
WIDTH -       Float -   length of the blad of grass from left to right at is thickest point
SEED -        Integer - random seed

RAD_MIN -     Float -   radius of the curled blade of grass at its base
RAD_MAX -     Float -   radius of the curled blade everywhere beyond its base

X_OFFSET -    Float -   amount which the blade skews in the x-direction
Z_OFFSET -    Float -   amount which the blade skews in the z-direction

X_BEND -      Radian -  determines how much the blade of grass bends around the z-axis (it "bends" in the x-direction)
\***************************************/ 


#if(X_BEND = 0) //X_BEND cannot be zero (this causes problems with the mesh)
  #local X_BEND = 0.001;
#end

#if(TWIST = 0) //TWIST cannot be zero
  #local TWIST = 0.001;
#end 

#ifndef (GRASS_TRIANGLE_CNT)
  #declare GRASS_TRIANGLE_CNT = 0;
#end



//CONSTANTS
#local R1 = seed(SEED);
#local CURL = 0.025; //controls how the grass blade curls into itself (around the y-axis) near the base of the stem.  This makes it so that the radius of the curl spirals, avoiding a completely tubular blade of grass
#local X_Offset = HEIGHT*X_OFFSET; //amount by which the blade tilts in the x-direction
#local Z_Offset = HEIGHT*Z_OFFSET; //amount by which the blade tilts in the z-direction
#local RAD_DIF = RAD_MAX - RAD_MIN; //difference between RAD_MAX and RAD_MIN

#local STEM_TOP = STEM*HEIGHT; //portion of the blade dedicated to the stem
#local CENTER_TOP = CENTER*HEIGHT; //portion of the blade dedicated to the changing-radius center
#local MIDDLE_TOP = MIDDLE*HEIGHT; //portion of the blade dedicated to the fixed-radius middle
#local TOP_TOP = HEIGHT; //portion of the blade dedicated to the tip


 
//GRASS WRINKLE PATTERN
#local grass_pattern = 
function
{ 
  pattern
  { 
    waves
    
    turbulence 0.2
     
    scale HEIGHT*0.1
    translate y*rand(R1)*1000
  } 
}



//CALCULATIONS
#local h_step = (1 - SMOOTHNESS)/8; //Progressive steps of the height
#local points = max(3,ceil(WIDTH/h_step*2));  //numbers of points in width arrays

#local lower = array[points];
#local upper = array[points]; 



//GRASS COLORS
#local c_cnt = 10;
#local color_array = array[c_cnt]

#local i = 0;
#while(i < c_cnt)
  
  #local c_dif = rand(R1)*C_VAR;
  
  #local color_array[i] = 
  #texture
  {
    pigment{rgb<COLOR_1.red*(1-c_dif) + COLOR_2.red*c_dif, COLOR_1.green*(1-c_dif) + COLOR_2.green*c_dif, COLOR_1.blue*(1-c_dif) + COLOR_2.blue*c_dif>   filter FILTER} 
    
    finish{ambient AMBIENT diffuse 0.80 specular LUSTER*(rand(R1)*C_VAR*0.4 + 0.6) roughness LUSTER*(rand(R1)*C_VAR*0.3 + 0.2)}
  }

  #local i = i + 1;
#end



  
//distribution of colors horizontally along width of grass blade
#local color_distribution = array[points];

//set stips of color for the grass in groups of three triangles at a time
#local i = 0;
#while(i < points)
  #local color_i = color_array[floor(rand(R1)*c_cnt)]
  #local color_distribution[i] = color_i;
  #local color_distribution[min(i+1,points-1)] = color_i;
  #local color_distribution[min(i+2,points-1)] = color_i;
  
  #local i = i + 1;
#end
  

mesh
{
 
  //initialize first set of lower points
   
  #local h = 0; //current height along the curving length of the blade
  #local phi_rad = (HEIGHT-STEM_TOP)/X_BEND; //radius of the circle described by X_BEND

     
  #while(h < HEIGHT)
   
    
    #local stem_h_ratio = h/HEIGHT; //ranges between 0 and 1 starting at the bottom of the blade and ending at the top
    #local center_h_ratio = max(0, (h-STEM_TOP)/(HEIGHT-STEM_TOP));  //ranges between 0 and 1 starting at STEM_TOP and ending at TOP_TOP
    #local center_middle_h_ratio = min(max(0, (h-STEM_TOP)/(CENTER_TOP-STEM_TOP)),1); //ranges between 0 and 1 starting at STEM_TOP and ending at CENTER_TOP
    #local top_h_ratio = min(1, max(0, (h-MIDDLE_TOP)/(TOP_TOP-MIDDLE_TOP))); //ranges between 0 and 1 starting at MIDDLE_TOP and ending at TOP_TOP
   
   
                    
    //Generate the new points
    #local i=0;
    #while(i < points)
      
   
      #local i_ratio = i/points;
      
      
      #if(h < STEM_TOP) //Variables concerning the stem
        #local rad = RAD_MIN + CURL*i_ratio;
        #local center_offset = 0;
        #local arc_length = WIDTH;  
        #local x_off = 0;
        #local z_off = 0;
      #end
      
      #if(h > STEM_TOP & h < CENTER_TOP) //Variables concerning the center    
        #local rad = RAD_MIN + RAD_DIF * (cos(center_middle_h_ratio*pi+pi)*0.5+0.5) + CURL*i_ratio;        
        #local center_offset = STEM_TOP;
        #local arc_length = WIDTH;
        #local x_off = X_Offset * center_h_ratio * (cos(center_middle_h_ratio*pi+pi)*0.5+0.5);
        #local z_off = Z_Offset * center_h_ratio * (cos(center_middle_h_ratio*pi+pi)*0.5+0.5);
      #end
      
      #if(h > CENTER_TOP & h < MIDDLE_TOP) //Variables concerning the middle
        #local rad = RAD_MAX + CURL*i_ratio;
        #local center_offset = STEM_TOP;
        #local arc_length = WIDTH;
        #local x_off = X_Offset * center_h_ratio;
        #local z_off = Z_Offset * center_h_ratio;
      #end
      
      #if(h > MIDDLE_TOP) //Variables concerning the top 
        #local rad = RAD_MAX + CURL*i_ratio;
        #local center_offset = STEM_TOP;
        #local arc_length = max(0.01,WIDTH*(cos(top_h_ratio*pi/2+0.01)));
        #local x_off = X_Offset * center_h_ratio;
        #local z_off = Z_Offset * center_h_ratio;
      #end
      
      #local theta_max = arc_length/rad; //maximum number of degrees which the blade should rotate around its own curve
      #local theta_start = TWIST*center_h_ratio - theta_max*0.5 + pi*0.4; //where the blade starts twisting
      #local theta = theta_start + theta_max*i_ratio;
      
      #local phi = X_BEND*center_h_ratio;
      
      #local pp =
      //     X_BEND                                    CURL                OFFSETS
        <                               (         rad*cos(theta)      ) + x_off         ,//  + cos(phi)*rad*sin(TWIST*center_h_ratio), 
         (phi_rad*sin(phi)          ) + (sin(phi)*rad*sin(theta)      ) + center_offset ,//  + sin(phi)*rad*sin(TWIST*center_h_ratio),
         (phi_rad*cos(phi) - phi_rad) + (cos(phi)*rad*sin(theta) - rad) + z_off         >;//  - rad*cos(TWIST*center_h_ratio)>;
      
      
      #local wrink = grass_pattern(pp.x,pp.y,pp.z)*WRINKLINESS;
            
      #local upper[i] = pp+wrink;
           
      #local i = i + 1;
      
      
    #end
         
    
    //make the triangles (don't make the triangles on the first pass)
    #if(h > 0)
      #local i = 0;   
      #while(i < points - 1)   
        
        #local j = i+1;
        
        triangle{lower[i] lower[j] upper[i] texture{color_distribution[i]}}
        triangle{lower[j] upper[j] upper[i] texture{color_distribution[i]}}
        
        #local i = i + 1;
      #end
    #end
    
    
    //copy upper points into lower points
    #local i = 0;
    #while(i < points)
      
      #local lower[i] = upper[i];
      
      #local i = i + 1;
    #end
    
    #local h = h + h_step;
  
  #end
}    
  

#end //Make_Grass()
 
 
 
 
 
 
 





  //                                                            \\
 //---------------------POPULATE GRASS ARRAY---------------------\\
//                                                                \\

// Use this macro to generate unique blades of grass for one of several grass arrays.


#macro Populate_Grass_Array(GRASS_CNT, VARIETY, HEIGHT, THICKNESS, SMOOTHNESS, SEED, RAD_MIN, RAD_MAX, X_OFFSET, Z_OFFSET, X_BEND, TWIST, STEM,CENTER,MIDDLE, COLOR_1, COLOR_2, C_VAR, LUSTER, AMBIENT, FILTER, WRINKLINESS, GRASS_NUM)

/**********   VARIABLES   **************\
GRASS_CNT -   Integer - determines the number of unique blades of grass to be passed to the GRASS_ARRAY
VARIETY -     0-1 -     determines how much certain values can deviate from the maximum values passed to this macro

HEIGHT -      Float -   maximum length of the blade of grass from base to tip
THICKNESS -   Float -   width of each blade of grass in relation to its randomly determined height
SMOOTHNESS -  0-1 -     determines the number of triangles in the mesh
SEED -        Integer - random seed

RAD_MIN -     Float -   radius of the curled blade of grass at its base
RAD_MAX -     Float -   radius of the curled blade beyond its base

X_OFFSET -    Float -   maximum amount which the blade skews in the x-direction
Z_OFFSET -    Float -   maximum amount which the blade skews in the z-direction

X_BEND -      Radian -  maximum determines how much the blade of grass bends around the z-axis (it "bends" in the x-direction)
TWIST -       Radian -  maximum determines how much the blade twists along the y-axis

//these three variables work as one.  
STEM -        0-1 -     determines the percentage of the grass devoted to the stem 
CENTER -      0-1 -     determines the percentage of the grass devoted to the transition between the stem and the main body of the grass.  This value must be greater than STEM
MIDDLE -      0-1 -     determines the percentage of the grass devoted to the main body of the grass.  This value must be greater than CENTER

COLOR_1 -     rgb -     primary color of the grass
COLOR_2 -     rgb -     secondary color of the grass
C_VAR -       0-1 -     determines how much the color of the grass may deviate from the primary to the secondary color (this is a percentage)

LUSTER -      0-1 -     specular highlight
AMBIENT -     0-1 -     ambient value
FILTER -      0-1 -     filter light value

WRINKLINESS - 0-1 -     wrinkliness of the blades of grass

ARRAY_NUM -   0,1,2 -   determines which of the five grass arrays to populate.  0: green grass, 1: brown grass, 2: green cluster grass, 3: brown cluster grass, 4: wide cluster grass
\***************************************/       


#local R1 = seed(SEED);
#local INV_VAR = 1-VARIETY;


#declare SMOOTHNESS = SMOOTHNESS;
#declare STEM = STEM;
#declare CENTER = CENTER;
#declare MIDDLE = MIDDLE;

#declare COLOR_1 = COLOR_1;
#declare COLOR_2 = COLOR_2;
#declare C_VAR = C_VAR;

#declare LUSTER = LUSTER;
#declare AMBIENT = AMBIENT;
#declare FILTER = FILTER;
#declare WRINKLINESS = WRINKLINESS;


//global variables
//below are declared five different grass arrays, for the five different types of grass I use in Grassland_Foreground.pov

#if(ARRAY_NUM = 0) //green grass array
  #declare GRASS_ARRAY_0 = array[GRASS_CNT];
  #declare GRASS_MAX_0 = GRASS_CNT;
#end

#if(ARRAY_NUM = 1) //brown grass array
  #declare GRASS_ARRAY_1 = array[GRASS_CNT];
  #declare GRASS_MAX_1 = GRASS_CNT;
#end

#if(ARRAY_NUM = 2) //cluster green grass array
  #declare GRASS_ARRAY_2 = array[GRASS_CNT];
  #declare GRASS_MAX_2 = GRASS_CNT;
#end

#if(ARRAY_NUM = 3) //cluster brown grass array
  #declare GRASS_ARRAY_3 = array[GRASS_CNT];
  #declare GRASS_MAX_3 = GRASS_CNT;
#end

#if(ARRAY_NUM = 4) //cluster thick grass array
  #declare GRASS_ARRAY_4 = array[GRASS_CNT];
  #declare GRASS_MAX_4 = GRASS_CNT;
#end



#local i = 0;

//create random blades based on the variety passed to this macro
#while(i < GRASS_CNT)
  
  #local Height = HEIGHT*(rand(R1)*VARIETY + INV_VAR);
  #local Width = Height*THICKNESS;
  #local Seed = floor(rand(R1)*10000);
  #local Rad_Max = RAD_MAX*(rand(R1)*VARIETY + INV_VAR);
  #local X_Offset = X_OFFSET*(rand(R1)*VARIETY + INV_VAR);
  #local Z_Offset = Z_OFFSET*(rand(R1)*VARIETY + INV_VAR);
  #local X_Bend = X_BEND*(rand(R1)*VARIETY*2 + (1-VARIETY*2));
  #local Twist = TWIST*(rand(R1)*VARIETY*2 + (1-VARIETY*2));
  
  #if(ARRAY_NUM = 0)
    #declare GRASS_ARRAY_0[i] = Make_Grass_Blade(Height, Width, Seed, RAD_MIN, Rad_Max, X_Offset, Z_Offset, X_Bend, Twist);
  #end
  
  #if(ARRAY_NUM = 1)
    #declare GRASS_ARRAY_1[i] = Make_Grass_Blade(Height, Width, Seed, RAD_MIN, Rad_Max, X_Offset, Z_Offset, X_Bend, Twist);
  #end
  
  #if(ARRAY_NUM = 2)
    #declare GRASS_ARRAY_2[i] = Make_Grass_Blade(Height, Width, Seed, RAD_MIN, Rad_Max, X_Offset, Z_Offset, X_Bend, Twist);
  #end
  
  #if(ARRAY_NUM = 3)
    #declare GRASS_ARRAY_3[i] = Make_Grass_Blade(Height, Width, Seed, RAD_MIN, Rad_Max, X_Offset, Z_Offset, X_Bend, Twist);
  #end
  
  #if(ARRAY_NUM = 4)
    #declare GRASS_ARRAY_4[i] = Make_Grass_Blade(Height, Width, Seed, RAD_MIN, Rad_Max, X_Offset, Z_Offset, X_Bend, Twist);
  #end
  
  #local i = i + 1;
#end 





#end //Populate_Grass_Array()




            
            
            
            
            
            
            
            





  //                                                            \\
 //----------------------MAKE GRASS CLUSTER----------------------\\
//                                                                \\

// Use this macro to create several different grass clusters.  NOTE: You must run the Populate_Grass_Array() macro before running this one.

  
#macro Make_Grass_Cluster(THIN_RADIUS, WIDE_RADIUS, THIN_GRASS_CNT, WIDE_GRASS_CNT, MAX_BEND, SEED)
  
/**********   VARIABLES   **************\
THIN_RADIUS - Float -   determines the radius of the smaller clusters of grass
Wide_RADIUS - Float -   determines the radius of the larger cluster of grass

THIN_GRASS_CNT - Interger - number of blades of grass in the smaller clusters of grass
WIDE_GRASS_CNT - Interger - number of blades of grass in the larger cluster of grass
     
MAX_BEND -    Degree -  maximum ammount by which each blade of grass may bend outwards from the cluster
SEED -        Integer - random seed
\***************************************/ 


  #local R1 = seed(SEED);
      
  //each element in the following array will be a distinct grass cluster, using grass blades from different GRASS_ARRAY's    
  #declare GRASS_CLUSTER_MAX = 3;
  #declare GRASS_CLUSTER_ARRAY = array[GRASS_CLUSTER_MAX];
  
  
  #local i = 0;
  

  
  //Thin Green Grass Cluster
  #declare GRASS_CLUSTER_ARRAY[0] =
  union
  {
    #while(i < THIN_GRASS_CNT)
      
      #local rad = min(THIN_RADIUS, rand(R1)*THIN_RADIUS + THIN_RADIUS*0.00);
      #local theta = rand(R1)*360;
      #local bend = MAX_BEND*(rad/THIN_RADIUS);
      
      object
      {
        GRASS_ARRAY_2[floor(rand(R1)*GRASS_MAX_2)]
        
        rotate y*90
        rotate z*bend
        translate<-rad,0,0>
        rotate y*theta
      }
    
      
      #local i = i + 1;
      
    #end 
  }
  
  
  
  
  //Thin Brown Grass Cluster
  #local i = 0;
  #declare GRASS_CLUSTER_ARRAY[1] =
  union
  {
    #while(i < THIN_GRASS_CNT)
      
      #local rad = min(THIN_RADIUS, rand(R1)*THIN_RADIUS + THIN_RADIUS*0.00);
      #local theta = rand(R1)*360;
      #local bend = MAX_BEND*(rad/THIN_RADIUS);
      
      object
      {
        GRASS_ARRAY_3[floor(rand(R1)*GRASS_MAX_3)]
        
        rotate y*90
        rotate z*bend
        translate<-rad,0,0>
        rotate y*theta
      }
    
      
      #local i = i + 1;
      
    #end 
  }
  
  
  
  
  
       
  //Wide Grass Cluster
  #local i = 0;
  
  #declare GRASS_CLUSTER_ARRAY[2] =
  union
  {
    #while(i < WIDE_GRASS_CNT)
      
      #local rad = min(WIDE_RADIUS, rand(R1)*WIDE_RADIUS + WIDE_RADIUS*0.00);
      #local theta = rand(R1)*360;
      #local bend = MAX_BEND*(rad/WIDE_RADIUS);
      
      object
      {
        GRASS_ARRAY_4[floor(rand(R1)*GRASS_MAX_4)]
        
        rotate y*90
        rotate z*bend
        translate<-rad,0,0>
        rotate y*theta
      }
    
      
      #local i = i + 1;
      
    #end 
  }


#end //Make_Grass_Cluster()









