

/*_______________________________________________________________________________________________________*\

FILE NAME: Make_Seeded_Grass.inc
CREATED BY: Isaac Kulka, Copyright 12-28-2006
NOTES: Created for the Internet Raytracing Competition's Nov-Dec 2006 Contest: 'Complexity' (www.irtc.org)

DESCRIPTION: The macros included in this file can be used to create realistic looking stalks of seeded grass, as one might encounter on a prairie

\*_______________________________________________________________________________________________________*/

 


  //                                                            \\
 //----------------------MAKE SEED-------------------------------\\
//                                                                \\

// This macro is called by the Populate_Seed_Array() macro to create seeds for the SEEDED_GRASS_SEED_ARRAY.


#macro Make_Seed(HEIGHT, RADIUS, COLOR) 
 
 
/**********   VARIABLES   **************\
HEIGHT -      Float -   height of the seed
RADIUS -      Float -   radius of the seed at its widest point
COLOR -       <r,g,b> - color of the seed
\***************************************/ 


#local step = (1-max(SMOOTHNESS,0.7))/2;
#local rad = 0; //radius of the current cross section in the mesh
#local h = step; //since all the points at h=0 are equal to <0,0,0> we can move h one step above that
#local theta = 0; //rotation around the radius of the current cross section of the mesh
#local points = ceil(HEIGHT/step); //number of points in each circular cross section

#local upper = array[points];
#local lower = array[points];
    
    
//initialize lower array at <0,0,0>
#local i = 0;
#while(i < points)
  #local lower[i] = <0,0,0>;
  #local i = i + 1;
#end

mesh
{
  #while(h <= HEIGHT)
    
    #if(h + step <= HEIGHT)
      #local rad = sin(h*(pi/HEIGHT))*RADIUS;
    #else
      #local rad = 0;
    #end
    
    //define upper points  
    #local i = 0;
    #while(i < points)
      
      #local theta = (i/points)*(2*pi);
      
      #local upper[i] = <rad*cos(theta), h, rad*sin(theta)>;
    
      #local i = i + 1;
    #end
    
    //make triangles
    #local i = 0;
    #while(i < points)
      #local j = i+1;
      #if(j = points) #local j = 0; #end 
      
      triangle{upper[i],lower[i],lower[j]}
      triangle{upper[i],upper[j],lower[j]}
      
      
      #local i = i + 1;
    #end
    
    //set lower to upper
    #local i = 0;
    #while( i < points)
      
      #local lower[i] = upper[i];
      
      #local i = i + 1;
    #end
    
    #local h = h + step;
  #end   
                           
  //make one last tall narrow triangle. This gives the seeds a fuzzy appearance at their tips
  #local dd = RADIUS * 0.1; //distance to offset this triangle by
  triangle{<upper[0].x-dd,upper[0].y-dd,upper[0].z>, <upper[0].x,upper[0].y+HEIGHT*0.5,upper[0].z>, <upper[0].x+dd, upper[0].y-dd,upper[0].z>}
                           
  pigment
    {COLOR filter FILTER}
  finish
    {ambient AMBIENT diffuse 1.00 specular LUSTER roughness 0.5}
}

#end //end Make_Seed macro












  //                                                            \\
 //----------------------POPULATE SEED ARRAY---------------------\\
//                                                                \\

// Use this macro to create an array of unique seeds.  This macro must be run before the Populate_Seeded_Grass_Array() macro.


#macro Populate_Seed_Array(SEED_CNT, HEIGHT, RADIUS, VARIETY, SMOOTHNESS, R_SEED, COLOR, COLOR_VAR, FILTER, LUSTER, AMBIENT)


/**********   VARIABLES   **************\
SEED_CNT -    Integer - number of unique seeds in the SEED_ARRAY
HEIGHT -      Float -   maximum height of any one seed
RADIUS -      Float -   maximum radius of any one seed
VARIETY -     0-1 -     maximum variation between different seeds in this array
SMOOTHNESS -  0-1 -     determines the number of triangles in the mesh
R_SEED -      Integer - random seed
COLOR -       rgb -     primary color of the seed
COLOR_VAR     0-1 -     percentage by which the color of any one seed may deviate from the primary color
FILTER -      0-1 -     filter light value  
LUSTER -      0-1 -     specular highlight
AMBIENT -     0-1 -     ambient value
\***************************************/  




#local R1 = seed(R_SEED);

#declare SEED_ARRAY = array[SEED_CNT]
#declare SEED_MAX = SEED_CNT;

#declare SMOOTHNESS = SMOOTHNESS;
#declare FILTER = FILTER;
#declare LUSTER = LUSTER;
#declare AMBIENT = AMBIENT;



//create random, but similar looking seeds for the SEED_ARRAY
#local i = 0;
#while(i < SEED_CNT)
  #local Height = HEIGHT * (rand(R1)*VARIETY + (1-VARIETY));
  #local Radius = RADIUS * (rand(R1)*VARIETY + (1-VARIETY));
  #local Color =  rgb<COLOR.red*(rand(R1)*COLOR_VAR + (1-COLOR_VAR)), COLOR.green*(rand(R1)*COLOR_VAR + (1-COLOR_VAR)), COLOR.blue*(rand(R1)*COLOR_VAR + (1-COLOR_VAR))>;
  
  #declare SEED_ARRAY[i] = Make_Seed(Height, Radius, Color)
                                  
  #local i = i + 1;
#end 

#end //end Generate_Seeds macro

 
 
 
 
 
 
 








  //                                                            \\
 //----------------------MAKE SEEDED GRASS-----------------------\\
//                                                                \\

//This macro is called by the Populate_Seeded_Grass_Array() macro to create petals for the SEEDED_GRASS_ARRAY.


#macro Make_Seeded_Grass(HEIGHT, RADIUS, R_SEED, X_BEND, COLOR, SEEDINESS)


/**********   VARIABLES   **************\
HEIGHT -      Float -   height of the stalk
RADIUS -      Float -   thickness of the stalk
R_SEED -      Integer - random seed
X_BEND -      radian -  amount by which the stalk rotates around the z-axis
COLOR -       <r,g,b> - color of the stalk
SEEDINESS -   0-1 -     maximum percentage of the stalk that is populated with seeds starting at the top of the stalk and moving down
\***************************************/ 


union //creates a union between the stalk mesh and all of the seeds attached to the stalk
{
  
  #local R1 = seed(R_SEED);
  
  #local circumference = 2*pi*RADIUS; //distance around the stalk as wrapped around the y-axis
  #local step = (1-SMOOTHNESS)/2; //distance between points on the stalk
  #local points = ceil(circumference/step); //number of points per circular cross section
  
  #local lower = array[points];
  #local upper = array[points];
  
  #local x_bend_rad = HEIGHT/X_BEND; //the radius of the large circle which the bent stalk forms but a small arc-length of
  
  mesh
  {
    //initialize lower array
    #local i = 0;
    #while(i < points)
      #local theta = (i/points)*(2*pi);
      #local lower[i] = <RADIUS*cos(theta), 0, RADIUS*sin(theta)>;
      #local i = i + 1;
    #end
    
    //begin builing the stem
    #local h = step;
    #while(h <= HEIGHT)
      
      #local h_ratio = h/HEIGHT;
      
      #local rad = RADIUS*((1-(h/HEIGHT))*0.75 + 0.25);
      //generate new points for upper array
      #local i = 0;
      #while(i < points)
        #local theta = (i/points)*(2*pi);
        #local bend = (h/HEIGHT)*X_BEND;
        #local upper[i] = <rad*cos(theta)*cos(bend) + x_bend_rad*cos(bend) - x_bend_rad,    rad*cos(theta)*sin(bend) + x_bend_rad*sin(bend),    RADIUS*sin(theta)>;
        #local i = i + 1;
      #end
      
      //create the triangles for this pass
      #local i = 0;
      #while(i < points)
        #local j = i+1;
        #if(j = points) #local j = 0; #end 
        
       
        
        triangle{upper[i],lower[i],lower[j]}
        triangle{upper[i],upper[j],lower[j]}
        
              
        #local i = i + 1;
      #end
      
      //set lower to upper
      #local i = 0;
      #while( i < points)
        
        #local lower[i] = upper[i];
        
        #local i = i + 1;
      #end 
      
      
      #local h = h + step;
    #end
    
    pigment
      {COLOR filter FILTER}
    finish
      {ambient AMBIENT diffuse 0.85 specular LUSTER}
  
  }
  
  //POPULATE STALK WITH READY-MADE SEEDS
  
  #local seed_start = (1-SEEDINESS)*HEIGHT; //the lowest point on the stalk where seeds should be placed
  #local h = seed_start; //current location along the stalk vertically
  #local seed_step = SEED_HEIGHT/4; //distance between seeds
  
  #while(h < HEIGHT)
  
    #local seed_ratio = (h-seed_start)/(HEIGHT-seed_start);
    #local h_ratio = h/HEIGHT;
   
      
    //MAKE SEEDS IF h IS HIGH ENOUGH.  Scatter the seeds around each cross section of the stalk radially
  
    #local bend_i =  h_ratio*X_BEND;
    #local point_i = <x_bend_rad*cos(bend_i) - x_bend_rad, x_bend_rad*sin(bend_i), 0>;
    #local bend_f =  ((h+seed_step*2)/HEIGHT)*X_BEND;
    #local point_f = <x_bend_rad*cos(bend_f) - x_bend_rad, x_bend_rad*sin(bend_f), 0>;
    
    #local dx = point_f.x - point_i.x;
    #local dy = point_f.y - point_i.y;
    #local hyp = sqrt(dx*dx + dy*dy);
    
    #local theta_rad = -asin(dx/hyp);
    #local theta_deg = (theta_rad/(2*pi)) * 360;
    
    #local rot_offset = ((sin(seed_ratio*pi)*0.7)/(2*pi)) * 360;
    #local xz_ratio = rand(R1);
    #local x_rot = rot_offset*xz_ratio;
    #local z_rot = rot_offset*(1-xz_ratio);
    
    #if(rand(R1) > 0.50)
      #local x_rot = -x_rot;
    #end
    
    #if(rand(R1) > 0.50)
      #local z_rot = -z_rot;
    #end    
    
    object{SEED_ARRAY[floor(rand(R1)*SEED_MAX)] rotate z*(theta_deg+10) rotate<x_rot,0,z_rot> translate point_i}
    
    #local h = h + seed_step;
    
  #end
  
  
  //Make additional seed bunches curving out from the main stem
  
  
  #local seed_bunch_cnt = 0;
  #local seed_bunch_max = 4;
   
  #local seed_rot_max = 30;
  
  //make the seed bunch
  #while(seed_bunch_cnt < seed_bunch_max)
  
    #local start = SEEDINESS*rand(R1)*0.5 + (1-SEEDINESS);
    #local cluster_rot = X_BEND*start;
    #local base_point = <x_bend_rad*cos(cluster_rot) - x_bend_rad, x_bend_rad*sin(cluster_rot), 0>;
  
    
    #local h = 0;
    #local h_max = HEIGHT*(1-start)*0.80;
    #local h_step = 0.04;
    
    #local bunch_bend = rand(R1)*(pi/6);
    
    union
    {
      #while(h < h_max)
        
        #local h_ratio = h/h_max;
        #local bend = -bunch_bend*h_ratio;
        
        #if(h_ratio < 0.30)
          #local seed_rot = seed_rot_max * (h_ratio/0.30);
        #else
          #local seed_rot = (1-(h_ratio-0.30)/0.70)*seed_rot_max;
        #end
        
        object
        {
          SEED_ARRAY[floor(rand(R1)*SEED_MAX)] 
          rotate z*seed_rot
          rotate y*rand(R1)*360
          rotate -z*degrees(bend*2)
          translate <h*sin(bend),h*cos(bend),0>
        }
        
        #local h = h + h_step;
      #end
      
      #local x_rot = rand(R1)*10;
      
      rotate x*x_rot
      //rotate y*rand(R1)*360
      rotate y*(rand(R1)*60 - 30)
      rotate z*degrees(cluster_rot)
      translate base_point  
    }
    
    #local seed_bunch_cnt = seed_bunch_cnt + 1;
  #end
}

#end //Make_Seeded_Grass()














  //                                                            \\
 //-----------------POPULATE SEEDED GRASS ARRAY------------------\\
//                                                                \\

// Use this macro to create an array of unique seeded grass stalks.  NOTE: you must run the Populate_Seed_Array() macro before running this one.


#macro Populate_Seeded_Grass_Array(GRASS_CNT, VARIETY, HEIGHT, THICKNESS, SMOOTHNESS, R_SEED, X_BEND, COLOR, COLOR_VAR, FILTER, LUSTER, AMBIENT, SEEDINESS, SEED_HEIGHT)

/**********   VARIABLES   **************\
GRASS_CNT -   Integer - number of unique seeded grass instances
VARIETY -     0-1 -     percent by which certain values may deviate from their provided maximums in each seeded grass stalk
HEIGHT -      Float -   maximum height of the stalk
THICKNESS -   0-1 -     thickness of eachs talk (a percentage of the height)
SMOOTHNESS -  0-1 -     determines the number of triangles in the mesh
R_SEED -      Integer - random seed
X_BEND -      Radian -  maximum amount by which the stalk rotates around the z-axis (in the x-direction)
COLOR -       <r,g,b> - primary color of the stalk (does not apply to seeds)
FILTER -      0-1 -     filter light value  
LUSTER -      0-1 -     specular highlight
AMBIENT -     0-1 -     ambient value
SEEDINESS -   0-1 -     maximum percentage of the stalk that is populated with seeds starting at the top of the stalk and moving down
SEED_HEIGHT - Float -   height of the seeds, used to determine how far apart to string the seeds from one another
\***************************************/  


#local R1 = seed(R_SEED);
#local INV_VAR = 1-VARIETY;


//global variables
#declare SEEDED_GRASS_ARRAY = array[GRASS_CNT];
#declare SEEDED_GRASS_MAX = GRASS_CNT;

#declare SMOOTHNESS = SMOOTHNESS;
#declare FILTER = FILTER;
#declare LUSTER= LUSTER;
#declare AMBIENT = AMBIENT; 
#declare SEED_HEIGHT = SEED_HEIGHT;

#local i = 0;
#while(i < GRASS_CNT)
  #local Height = HEIGHT * (rand(R1)*VARIETY + INV_VAR);
  #local Radius = Height * THICKNESS*0.1;
  
  #local Color = COLOR*(rand(R1)*COLOR_VAR + (1-COLOR_VAR));
  #local X_Bend = X_BEND * (rand(R1)*VARIETY + INV_VAR);
  #local Seediness = SEEDINESS*(rand(R1)*VARIETY+INV_VAR);
  #local R_Seed = rand(R1)*10000;
  
  #declare SEEDED_GRASS_ARRAY[i] = Make_Seeded_Grass(Height, Radius, R_Seed, X_Bend, Color, Seediness)
  
  #local i = i + 1;
#end  

#end //Populate_Seeded_Grass_Array()