                                             

/*_______________________________________________________________________________________________________*\

FILE NAME: Make_Sumac.inc
CREATED BY: Isaac Kulka, Copyright 12-28-2006
NOTES: Created for the Internet Raytracing Competition's Nov-Dec 2006 Contest: 'Complexity' (www.irtc.org)

DESCRIPTION: The macros in this file can be used to create Sumac shrubs complete with stems, leaves and berry clusters.

\*_______________________________________________________________________________________________________*/




       
  //                                                     \\
 //----------------------MAKE SUMAC-----------------------\\
//                                                         \\       

// This macro is called by the Populate_Sumac_Array() macro.  It is used to create the main stem of the plant and spawn off branches from that stem.


#macro Make_Sumac(HEIGHT, BEND, SEED)

/**********   VARIABLES   **************\
HEIGHT -       Float -   The length of the trunk of the shrub (that is, the ur-branch)
BEND -         Radian -  The ammount by which stem bends
SEED -         Integer - random seed
\***************************************/  
  
  #local R1 = seed(SEED);
  #local Base_Radius = HEIGHT*THICKNESS;
  #local Top_Radius = Base_Radius * 0.40;
  
  
  //MAKE SPLINE
  #local spline_step = HEIGHT/10;
  #local i = spline_step;
  
  #local theta = 0;
  #local theta_max = BEND; //ammount by which the main stem of the sumac plant bends
  
  #local bend_start = rand(R1)*0.40 + 0.40;
  
  #local x_offset = 0;
  #local z_offset = 0;
  #local curve = CURVINESS*HEIGHT/10;
  
  //this spline will define the shape of the branch
  #local branch_spline = 
  
  spline
  {
    natural_spline
    
    //initial points
    -spline_step, <0,-spline_step*HEIGHT,0>
    0.00, <0,0,0>
    
    #while(i <= 1.20)
      
      #local up_i = min(i, bend_start);
      #local bend_i = max(0, i-bend_start);
      
      #if(i > bend_start)
        #local theta = theta_max*(bend_i/(1-bend_start));
      #end
      
      #local x_offset = x_offset + rand(R1)*curve - curve*0.50;
      #local z_offset = z_offset + rand(R1)*curve - curve*0.50;
      
      i, <sin(theta)*bend_i*HEIGHT + x_offset, up_i*HEIGHT + cos(theta)*bend_i*HEIGHT, z_offset>
      
    
      #local i = i + spline_step;
    
    #end
    
    
  
  }
   
     
             
             
  //MAKE MESH of the stem
  
  mesh
  { 
    #local h = 0;
    #local h_step = (1-SMOOTHNESS) * 0.50; //distance between points in the branch mesh
    
    #local rad = Base_Radius; //radius of the current cross-section
    #local rad_diff = Base_Radius - Top_Radius;
    
    #local points = max(3,ceil(Base_Radius/(1-SMOOTHNESS)*5)); //number of points per circular cross-section
    #local lower = array[points];
    #local upper = array[points];
    
    #local l_norm = array[points]; //norms of the points of the lower array (used for smooth triangles in the mesh)
    #local u_norm = array[points]; //norms of the points of the upper array
    
    #local spline_norm = branch_spline(h_step/HEIGHT)-branch_spline(0);
    
    #local norm = <1,0,0>;
    #if(spline_norm.x != 0 | spline_norm.y != 0 | spline_norm.z != 0)
      #local norm = vnormalize(spline_norm);
    #end
    
    #local norm2 = vnormalize(<norm.x+0.1,norm.y,norm.z>);
    #local norm_perp = vnormalize(vcross(norm, norm2)); //normal vector perpendicular to branch (used to rotate points around norm in order to give your branch some thickness)
    
    //set initial lower points
    #local i=0;
    #while(i < points)
      #local theta = (i/points)*360; //current rotation around the current cross-section
      #local point = norm_perp*rad; //a single point sticking out a distance of rad from the normal vector
      #local lower[i] = vaxis_rotate(point, norm, theta) + branch_spline(h/HEIGHT); //to dermine all the points in the cross-section, start rotating point around norm
      
      #local l_norm[i] = vnormalize(lower[i] - (branch_spline(h_step/HEIGHT))); //determine the normal of each point (to be used with smooth triangles)
      
      #local i = i + 1;
    #end
    
    #local h = h + h_step;
  
  
  
    #while(h < HEIGHT)
      
      #local h_ratio = h/HEIGHT;
      #local h_ratio_f = (h+h_step)/HEIGHT;
      #local rad = rad_diff*(1-h_ratio) + Top_Radius;
      
      
      #local spline_norm = branch_spline(h_ratio_f) - branch_spline(h_ratio); //direction of bottom-most cross-section
      
      #local norm = <0,1,0>;
      #if(spline_norm.x != 0 | spline_norm.y != 0 | spline_norm.z != 0)
        #local norm = vnormalize(spline_norm);
      #end
      
      #local norm2 = vnormalize(<norm.x,norm.y,norm.z+0.1>);
      #local norm_perp = vnormalize(vcross(norm, norm2));
      
      //set upper points
      #local i=0;
      #while(i < points)
        #local theta = (i/points)*360;
        #local point = norm_perp*rad;
        #local upper[i] = vaxis_rotate(point, norm, theta) + branch_spline(h_ratio); //<rad*cos(theta), 0, rad*sin(theta)>;
        
        //smooth normal
        #local u_norm[i] = vnormalize(upper[i] - (branch_spline(h_ratio)));
       
        
        #local i = i + 1;
      #end
        
      //create triangles
      #local i = 0;
      #while(i < points)
        #local j = i+1;
        #if(j >= points) #local j=0; #end
        
        smooth_triangle{upper[i], u_norm[i], lower[i], l_norm[i], lower[j], l_norm[j]}
        smooth_triangle{upper[j], u_norm[j], lower[j], l_norm[j], upper[i], u_norm[i]}
        
        
        #local i = i + 1;
      #end
      
      //set lower equal to upper
      #local i = 0;
      #while(i < points)
        #local lower[i] = upper[i];
        #local l_norm[i] = u_norm[i];
        
        #local i = i + 1;
    #end
      
     
        
      
      #local h = h + h_step;
    #end
  
    texture{SUMAC_STEM_TEXTURE}                           
  } 
                      
   
   
   
   
   
  //MAKE BERRY BUNCH at top of stem
   
            
  #local B_START = branch_spline(0.96);
  #local B_END = branch_spline(1.05);
  #local B_HEIGHT = HEIGHT*0.18; //0.16
  #local B_BASE_RADIUS = B_HEIGHT*0.22;
  #local B_BERRY_RADIUS = BERRY_RADIUS; //see Make_Sumac_Berry above
  #local B_COLOR_1 = rgb<0.8,0.3,0.2>;
  #local B_COLOR_2 = rgb<0.6,0.1,0.0>;
  #local B_C_VAR = 1.00;
  #local B_FILTER = 0.00;
  #local B_SEED = rand(R1)*1000; 
        
  #if(rand(R1) < 1.00)
    Make_Sumac_Berry_Bunch(B_START, B_END, B_HEIGHT, B_BASE_RADIUS, B_BERRY_RADIUS, B_COLOR_1, B_COLOR_2, B_C_VAR, B_FILTER, B_SEED)                       
  #end                   
  
  
   
              
              
              
   
   
  //ADD BRANCHES TO STEM
  
  #local h = HEIGHT*0.33;
  #local h_step = HEIGHT*0.10;
  
  #local theta = 0;
  #local theta_step = 105;
  
  #while(h < HEIGHT)
    
    #local h_ratio = h/HEIGHT;
    #local h_ratio_f = (h+h_step)/HEIGHT;
    
    #local tip_rot = h_ratio*20 + 15;
    
    #local Pi = branch_spline(h_ratio);
    #local Pf = branch_spline(h_ratio_f);
    
    #local dx = Pi.x - Pf.x;
    #local dy = Pi.y - Pf.y;
    #local hyp = sqrt(dx*dx + dy*dy);
    
    #if(hyp = 0)
      #local z_rot = 0;
    #else
      #local z_rot = degrees(asin(dy/hyp))*-1 - 90;
    #end
    
    #local Height = HEIGHT * 0.33 + rand(R1)*HEIGHT*0.10;
    #local Bend = -h_ratio*pi/8 - rand(R1)*h_ratio*pi/16;
    #local Seed = ceil(rand(R1)*10000);
    
    union
    {
      Make_Sumac_Branch(Height, Bend, Seed)
      
      rotate z*(tip_rot + rand(R1)*tip_rot*0.25)
      rotate y*(theta + rand(R1)*theta_step*0.5)
      rotate z*z_rot
      
      translate branch_spline(h_ratio + rand(R1)*0.05)
    }
  
    #local h = h + h_step;
    #local theta = theta + theta_step;
    
  #end
  
  
  //add the last two branches at the top
  
  #local Height = HEIGHT * 0.33 + rand(R1)*HEIGHT*0.10;
  #local Bend = -h_ratio*pi/8 - rand(R1)*h_ratio*pi/16;
  #local Seed = ceil(rand(R1)*10000);
  
  union
  {
    Make_Sumac_Branch(Height, Bend, Seed)
    
    rotate z*(tip_rot + rand(R1)*tip_rot*0.25)
    rotate y*(theta + rand(R1)*theta_step*0.5)
    rotate z*z_rot
    
    translate branch_spline(h_ratio + rand(R1)*0.05)
  }
  
  #local theta = theta +  theta_step;
  
  #local Height = HEIGHT * 0.33 + rand(R1)*HEIGHT*0.10;
  #local Bend = -h_ratio*pi/8 - rand(R1)*h_ratio*pi/16;
  #local Seed = ceil(rand(R1)*10000);
  
  union
  {
    Make_Sumac_Branch(Height, Bend, Seed)
    
    rotate z*(tip_rot + rand(R1)*tip_rot*0.25)
    rotate y*(theta + rand(R1)*theta_step*0.5)
    rotate z*z_rot
    
    translate branch_spline(h_ratio + rand(R1)*0.05)
  }

                      
#end // Make_Sumac()

        
        
        
        
        
        
        
        
        
        
        
        
        
        


  //                                                     \\
 //----------------------MAKE SUMAC BRANCH----------------\\
//                                                         \\       

// This macro is called by the Make_Sumac() macro.  It creates the branches for the sumac and populates them with leaves.

#macro Make_Sumac_Branch(HEIGHT, BEND, SEED)


/**********   VARIABLES   **************\
HEIGHT -       Float -   The length of the trunk of the shrub (that is, the ur-branch)
BEND -         Radian -  amount by which the branch bends
SEED -         Integer - random seed
\***************************************/  

  #declare BRANCH_CNT = 0;
  #declare LEAVES_CNT = 0;
  #declare BRANCH_TRIANGLE_CNT = 0;
  
  #local R1 = seed(SEED);
  #local Base_Radius = HEIGHT*THICKNESS;
  #local Top_Radius = Base_Radius * 0.40;
  
  
  
  
  //MAKE SPLINE
  #local spline_step = HEIGHT/10;
  #local i = spline_step;
  
  #local theta = 0;
  #local theta_max = BEND;
  
  #local bend_start = rand(R1)*0.40 + 0.20;
  
  #local y_offset = 0;
  #local z_offset = 0;
  #local curve = CURVINESS*HEIGHT/10;
  
  //this spline will define the shape of the branch
  #local branch_spline = 
  
  spline
  {
    natural_spline
    
    //initial points
    -spline_step, <-spline_step*HEIGHT,0,0>
    0.00, <0,0,0>
    
    #while(i <= 1.20)
      
      #local up_i = min(i, bend_start);
      #local bend_i = max(0, i-bend_start);
      
      #if(i > bend_start)
        #local theta = theta_max*(bend_i/(1-bend_start));
      #end
      
      #local y_offset = y_offset + rand(R1)*curve - curve*0.50;
      #local z_offset = z_offset + rand(R1)*curve - curve*0.50;
      
      i, <up_i*HEIGHT + cos(theta)*bend_i*HEIGHT, sin(theta)*bend_i*HEIGHT + y_offset, z_offset>
      
    
      #local i = i + spline_step;
    
    #end    
  
  }
  
             
             
        
             
             
  //MAKE MESH
  
  mesh
  { 
    #local h = 0;
    #local h_step = (1-SMOOTHNESS) * 0.50; //distance between points in the branch mesh
    
    #local rad = Base_Radius; //radius of the current cross-section
    #local rad_diff = Base_Radius - Top_Radius;
    
    #local points = max(3,ceil(Base_Radius/(1-SMOOTHNESS)*5)); //number of points per circular cross-section
    #local lower = array[points];
    #local upper = array[points];
    
    #local l_norm = array[points]; //norms of the points of the lower array (used for smooth triangles in the mesh)
    #local u_norm = array[points]; //norms of the points of the upper array
    
    #local norm = vnormalize(branch_spline(0)-branch_spline(-spline_step));
    #local norm2 = vnormalize(<norm.x,norm.y+0.1,norm.z>);
    #local norm_perp = vnormalize(vcross(norm, norm2)); //normal vector perpendicular to branch (used to rotate points around norm in order to give your branch some thickness)
    
    //set initial lower points
    #local i=0;
    #while(i < points)
      #local theta = (i/points)*360; //current rotation around the current cross-section
      #local point = norm_perp*rad; //a single point sticking out a distance of rad from the normal vector
      #local lower[i] = vaxis_rotate(point, norm, theta) + branch_spline(h/HEIGHT); //to dermine all the points in the cross-section, start rotating point around norm
      
      #local l_norm[i] = vnormalize(lower[i] - (branch_spline(h/HEIGHT))); //determine the normal of each point (to be used with smooth triangles)
      
      #local i = i + 1;
    #end
    
    #local h = h + h_step;
  
  
  
    #while(h < HEIGHT)
      
      #local h_ratio = h/HEIGHT;
      #local h_ratio_f = (h+h_step)/HEIGHT;
      #local rad = rad_diff*(1-h_ratio) + Top_Radius;
      
      
      #local spline_norm = branch_spline(h_ratio_f) - branch_spline(h_ratio); //direction of bottom-most cross-section
      
      #local norm = <1,0,0>;
      #if(spline_norm.x != 0 | spline_norm.y != 0 | spline_norm.z != 0)
        #local norm = vnormalize(spline_norm);
      #end
      
      #local norm2 = vnormalize(<norm.x,norm.y+0.1,norm.z>);
      #local norm_perp = vnormalize(vcross(norm, norm2));
      
      //set upper points
      #local i=0;
      #while(i < points)
        #local theta = (i/points)*360;
        #local point = norm_perp*rad;
        #local upper[i] = vaxis_rotate(point, norm, theta) + branch_spline(h_ratio); //<rad*cos(theta), 0, rad*sin(theta)>;
        
        //smooth normal
        #local u_norm[i] = vnormalize(upper[i] - (branch_spline(h_ratio)));
       
        
        #local i = i + 1;
      #end
        
      //create triangles
      #local i = 0;
      #while(i < points)
        #local j = i+1;
        #if(j >= points) #local j=0; #end

        smooth_triangle{upper[i], u_norm[i], lower[i], l_norm[i], lower[j], l_norm[j]}
        smooth_triangle{upper[j], u_norm[j], lower[j], l_norm[j], upper[i], u_norm[i]}
        
        
        #local i = i + 1;
      #end
      
      //set lower equal to upper
      #local i = 0;
      #while(i < points)
        #local lower[i] = upper[i];
        #local l_norm[i] = u_norm[i];
        
        #local i = i + 1;
    #end
      
     
        
      
      #local h = h + h_step;
    #end
  
    texture{SUMAC_BRANCH_TEXTURE}                           
  }
  
  
  
  
  //ADD LEAVES TO BRANCH
  
  #local h = HEIGHT*0.25;
  #local h_step = LEAF_WIDTH * 1.25; //space in between leaves
  
  #while(h < HEIGHT)
    
    #local h_ratio = h/HEIGHT;
    #local h_ratio_f = (h+h_step)/HEIGHT;
    
    #local tip_rot = h_ratio*45;
    
    #local Pi = branch_spline(h_ratio);
    #local Pf = branch_spline(h_ratio_f);
    
    #local dx = Pi.x - Pf.x;
    #local dy = Pi.y - Pf.y;
    #local hyp = sqrt(dx*dx + dy*dy);
    
    #if(hyp = 0)
      #local z_rot = 0;
    #else
      #local z_rot = degrees(asin(dy/hyp))*-1;
    #end
    
    #local x_rot = 5*((90+z_rot)/90); //this controls the droopiness of the leaves
    
    object
    { 
      //randomly select another leaf from the SHRUB_LEAF_ARRAY
      SUMAC_LEAF_ARRAY[floor(rand(R1)*SUMAC_LEAF_MAX)]  
      
      scale ((1-h_ratio)*0.25 + 0.75 + rand(R1)*0.15)
      
      rotate x*rand(R1)*20 - 10 
      rotate x*90
      rotate y*tip_rot + tip_rot*rand(R1)*0.20
      rotate z*z_rot
      rotate x*x_rot + x_rot*rand(R1)*0.20
      translate branch_spline(h_ratio)              
    }
    
    object
    { 
      //randomly select another leaf from the SHRUB_LEAF_ARRAY
      SUMAC_LEAF_ARRAY[floor(rand(R1)*SUMAC_LEAF_MAX)]  
      
      scale ((1-h_ratio)*0.25 + 0.75 + rand(R1)*0.15)
      
      rotate x*rand(R1)*20 - 10
      rotate y*180
      rotate x*-90
      rotate y*-tip_rot + tip_rot*rand(R1)*0.20
      rotate z*z_rot
      rotate x*-x_rot - x_rot*rand(R1)*0.20
      translate branch_spline(h_ratio)              
    }
    
    
    #local h = h + h_step;
  #end
                                         
                                         
  //add the leaf at the top of the branch 
  object
    { 
      //randomly select another leaf from the SHRUB_LEAF_ARRAY
      SUMAC_LEAF_ARRAY[floor(rand(R1)*SUMAC_LEAF_MAX)]  
      
      scale ((1-h_ratio)*0.25 + 0.75 + rand(R1)*0.15)
      
      rotate x*90 + rand(R1)*10
      rotate y*90 + rand(R1)*10
      rotate z*z_rot
      translate branch_spline(1.00)              
    }
  
   
#end






           




  //                                                         \\
 //--------------------POPULATE SUMAC ARRAY-------------------\\
//                                                             \\   

// Use this macro to create an array of unique sumac plants.  NOTE: you must run Make_Sumac_Berry() and Populate_Sumac_Leaf_Array() before running this macro.


#macro Populate_Sumac_Array(SUMAC_CNT, VARIETY, HEIGHT, THICKNESS, MAX_BEND, SMOOTHNESS, CURVINESS, BERRY_RAD, LEAF_WIDTH, SEED)


/**********   VARIABLES   **************\
SUMAC_CNT -    Integer - total number of individual sumac plants to generate
VARIETY -      0-1 -     maximum percentage variation between individual sumac plants and the set default values
HEIGHT -       Float -   The length of the trunk of the shrub (that is, the ur-branch)
THICKNESS -    0-1 -     radius of base of the sumac plant, as a percentage of the height
SMOOTHNESS -   0-1 -     determines the number of trianges in the mesh, higher value means more triangles
CURVINESS -    0-1 -     determines how much the branches curve and juggle
BERRY_RAD -    Float -   radius of the berries on the shrub
SEED -         Integer - random seed
\***************************************/   



#declare THICKNESS = THICKNESS;
#declare SMOOTHNESS = SMOOTHNESS;
#declare CURVINESS = CURVINESS;
#declare BERRY_RAD = BERRY_RAD;
#declare LEAF_WIDTH = LEAF_WIDTH;

#local R1 = seed(SEED);
#local INV_VAR = 1-VARIETY;

//global values
#declare SUMAC_ARRAY = array[SUMAC_CNT];
#declare SUMAC_MAX = SUMAC_CNT;

//populate the array with unique sumac plants
#local i = 0;
#while(i < SUMAC_CNT)
  #local Height = HEIGHT * (rand(R1)*VARIETY + INV_VAR);
  #local Bend = MAX_BEND*INV_VAR + MAX_BEND*(rand(R1)*VARIETY);
  #local Seed = ceil(rand(R1)*10000);
 
  #declare SUMAC_ARRAY[i] = union{Make_Sumac(Height, Bend, Seed)}
                                                
  #local i = i + 1;
#end
  

#end //end Populate_SHRUB_LEAF_Array           
           
           
           
           
           
           
           
           
           
           
                               

  //                                                          \\
 //----------------------MAKE SUMAC LEAF-----------------------\\
//                                                              \\

// This macro is called by the Populate_Sumac_Leaf_Array() macro.  It is responsible for generating the individual sumac leaf meshes.


#macro Make_Sumac_Leaf(HEIGHT, WIDTH, BEND, RUFFLE, SMOOTHNESS, INNER_COLOR, OUTER_COLOR, FILTER, LUSTER, AMBIENT, SEED)


/**********   VARIABLES   **************\
HEIGHT -      Float -   length of the leaf from stem to tip
WIDTH -       Float -   width of the leaf from side to side at its widest point
BEND -        Radian -  curl of the leaf from stem to tip
RUFFLE -      Float -   amount by which the leaves balloon out around the central stem
SMOOTHNESS -  0-1 -     determines the number of triangles in the mesh
INNER_COLOR - rgb -     color value of the center of the leaf
OUTER_COLOR - rgb -     color value of the edges of the leaf
FILTER -      0-1 -     filtered light value
LUSTER -      0-1 -     specular highlight of the leaf
AMBIENT -     0-1 -     ambient value of the leaf
\***************************************/   

#local R1 = seed(SEED);
 
  mesh
  {
    #ifndef(LEAF_TRIANGLES)
      #declare LEAF_TRIANGLES = 0;
    #end
    
    //various textures of the leaf's areas
    #local SUMAC_LEAF_STEM_TEXTURE =  texture{pigment{rgbft<0.34,0.29,0.17, 0.2, 0>} finish{ambient 0 diffuse 0.6 specular 0.1}};
    #local SUMAC_LEAF_TEXTURE =       texture{pigment{INNER_COLOR filter FILTER} finish{ambient AMBIENT diffuse 0.8 specular LUSTER roughness 0.3}};
    #local SUMAC_OUTER_LEAF_TEXTURE = texture{pigment{OUTER_COLOR filter FILTER} finish{ambient AMBIENT diffuse 0.8 specular LUSTER roughness 0.3}};
    
    #local RUFFLE = max(RUFFLE, 0.05); //ruffle cannot be below 0.05 (no flat leaves)
    
    #local h_step = HEIGHT*(1-SMOOTHNESS)*0.1; //distance between points on the mesh
    #local h = 0; //current value along the stem-to-tip length of the leaf
    
    #local points = max(3, ceil(WIDTH/h_step)); //number of points per horizontal cross-section of the leaf
    
    #local lower = array[points];
    #local upper = array[points];
    
    #local leaf_wrinkle = 0.02;
    
    #local i = 0;
    #while(i < points)
      
      #local lower[i] = <0,0,0>;
      
      #local i = i + 1;
    #end
    
    #local rpx = rand(R1)*1000;
    #local rpy = rand(R1)*1000;
    #local rpz = rand(R1)*1000;
    
    //Leaf Stem
    #local sp1 = <-WIDTH*0.025,0,0>;
    #local sp2 = < WIDTH*0.025,0,0>;
    #local sp3 = <-WIDTH*0.020,HEIGHT*0.1,0>;
    #local sp4 = < WIDTH*0.020,HEIGHT*0.1,0>;
    
    triangle{sp1, sp3, sp2 texture{SUMAC_LEAF_STEM_TEXTURE}}
    triangle{sp3, sp4, sp2 texture{SUMAC_LEAF_STEM_TEXTURE}}
  
    
    #local width_switch = 0.20;
    
    //the leaf mesh
    #while(h < HEIGHT)
      
      #local h_ratio = h/HEIGHT;
      
      #if(h_ratio < width_switch)
        #local rad = sin(h_ratio/width_switch*pi/2)*WIDTH;
      #else
        #local rad = sin((h_ratio-width_switch)/(1-width_switch)*pi/2 + pi/2)*WIDTH;
      #end
      
      #if(h+h_step >= HEIGHT) #local rad = 0; #end
      
      #local rad_ratio = #if(rad != 0) (WIDTH*0.05)/rad; #else points; #end //the if statement here avoids divide by zero exceptions
      #local yellow_point = ceil(rad_ratio*points); //this value determines where to stop coloring the triagles with the OUTER_COLOR
      
      #local i = 0;
      #while(i < points)
        #local i_ratio = i/points;
        
        #local z_bend = BEND * h_ratio;
        
        #local pp = <rad*i_ratio - rad/2,      cos(z_bend)*(h + HEIGHT*0.03) + cos(z_bend)*((1-h_ratio) * rad*RUFFLE*(1-h_ratio)*abs(sin(i_ratio*pi*2))),      sin(z_bend)*(h + HEIGHT*0.03)> + sin(z_bend)*((1-h_ratio) * rad*RUFFLE*(1-h_ratio)*abs(sin(i_ratio*pi*2)));
        
        #local upper[i] = <pp.x,      pp.y + SUMAC_LEAF_PATTERN(pp.x,pp.y,pp.z)*leaf_wrinkle*HEIGHT,     pp.z + SUMAC_LEAF_PATTERN(pp.x+rpx,pp.y+rpy,pp.z+rpz)*leaf_wrinkle*HEIGHT>;
        
        
        #local i = i + 1;
      #end
      
      
      #local i = 0;
      #while(i < points-1)
        
        #local i_ratio = i/(points-1);
        
        #if(i < yellow_point-1 | i >= (points - yellow_point - 1))
          
          triangle{upper[i], lower[i], lower[i+1] texture{SUMAC_OUTER_LEAF_TEXTURE}}
          triangle{upper[i+1], upper[i], lower[i+1] texture{SUMAC_OUTER_LEAF_TEXTURE}}  
        #else
          triangle{upper[i], lower[i], lower[i+1] texture{SUMAC_LEAF_TEXTURE}}
          triangle{upper[i+1], upper[i], lower[i+1] texture{SUMAC_LEAF_TEXTURE}}
        #end
        
        
        #local i = i + 1;
      #end
      
      
      #local i = 0;
      #while(i < points)
        
        #local lower[i] = upper[i];
        
        #local i = i + 1;
      #end
      
      #local h = h + h_step;
    #end
    
   
   double_illuminate //this value makes it so that the leaves are never completely dark no matter which side your looking at (sort of like ambient)
   
  }

#end //Make_Sumac_Leaf()













  //                                                              \\
 //--------------------POPULATE SHRUB LEAF ARRAY-------------------\\
//                                                                  \\   

// Use this macro to create an array of unique leaves for your sumac plants.  This macro must be run before the Populate_Sumac_Array() macro.


#macro Populate_Sumac_Leaf_Array(LEAF_CNT, VARIETY, HEIGHT, THICKNESS, BEND, RUFFLE, SMOOTHNESS, INNER_COLOR, OUTER_COLOR, C_VAR, FILTER, LUSTER, AMBIENT, SEED)

/**********   VARIABLES   **************\
LEAF_CNT -    Integer - Number of unique leaves to generate
VAREITY -     0-1 -     determines how much certain values may vary from their provided maximums
HEIGHT -      Float -   maximum length of the leaf from stem to tip
THICKNESS -   Float -   determines width of a given leaf based on its HEIGHT
BEND -        Radian -  maximum curl of the leaf from stem to tip
RUFFLE -      Float -   maximum amount by which the leaves balloon out around the central stem
SMOOTHNESS -  0-1 -     determines the number of triangles in the mesh
INNER_COLOR - rgb -     color value of the center of the leaf
OUTER_COLOR - rgb -     color value of the edges of the leaf
C_VAR -       0-1 -     amount by wihich the colors can vary from their root colors
FILTER -      0-1 -     filtered light value
LUSTER -      0-1 -     specular highlight of the leaf
AMBIENT -     0-1 -     ambient value of the leaf
SEED -        Integer - random seed 
\***************************************/   


#local R1 = seed(SEED);
#local INV_VAR = 1-VARIETY;

//global values
#declare SUMAC_LEAF_ARRAY = array[LEAF_CNT];
#declare SUMAC_LEAF_MAX = LEAF_CNT;

//populate the array with unique leaves
#local i = 0;
#while(i < LEAF_CNT)
  #local Height = HEIGHT * (rand(R1)*VARIETY + INV_VAR);
  #local Width = Height * THICKNESS;
  
  #local Bend = BEND * (rand(R1)*VARIETY + INV_VAR);
  #local Ruffle = RUFFLE * (rand(R1)*VARIETY + INV_VAR);
  
  #local C_Var = (1 - rand(R1)*C_VAR);
  
  #local Inner_Color = rgb<INNER_COLOR.red*C_Var, INNER_COLOR.green*C_Var, INNER_COLOR.blue*C_Var>;
  #local Outer_Color = rgb<OUTER_COLOR.red*C_Var, OUTER_COLOR.green*C_Var, OUTER_COLOR.blue*C_Var>;
  
  #local Seed = rand(R1)*10000;
  
  #declare SUMAC_LEAF_ARRAY[i] = Make_Sumac_Leaf(Height, Width, Bend, Ruffle, SMOOTHNESS, Inner_Color, Outer_Color, FILTER, LUSTER, AMBIENT, Seed)
                                                
  #local i = i + 1;
#end
  

#end //end Populate_SHRUB_LEAF_Array











  //                                                          \\
 //----------------------MAKE SUMAC BERRY----------------------\\
//                                                              \\

// Use this macro to create the berry which will be used with your sumac plants' berry bunches.  This macro must be run before the Populate_Sumac_Array() macro.


#macro Make_Sumac_Berry(RADIUS, SMOOTHNESS, AMBIENT, LUSTER) 

/**********   VARIABLES   **************\
RADIUS -      Float -   radius of the spherical berry
SMOOTHNESS -  0-1 -     determines the number of triangles in the berry
AMBIENT -     0-1 -     ambient Value of the berry   
LUSTER -      0-1 -     specular highlight of the berry 
\***************************************/    

//please note: this macro does not determine color or filter values for the berry, those are determined in the Make_Berry_Bunch macro located below


#local step = (1-max(SMOOTHNESS,0.70))/16; //distance between points in mesh
#local rad = 0; //radius of the current circular cross-section in the mesh
#local theta = 0; //current rotation around the current cross-section
#local Height = RADIUS; //height of the berry
#local h = -Height; //current height-value, which determines the current cross-section (ranges from -HEIGHT to +HEIGHT)

#local points = ceil(Height/step);

#local upper = array[points];
#local lower = array[points];

//initialize lower array at <0,0,0>
#local i = 0;
#while(i < points)
  #local lower[i] = <0,h,0>;
  #local i = i + 1;
#end

#local h = h + step;

#declare SUMAC_BERRY = 
mesh
{
  #while(h <= Height)
    
    #local h_ratio = h/Height;
    
    #if(h + step <= Height)
      #local rad = sqrt(pow(RADIUS,2)-pow(h,2));
    #else
      #local rad = 0;
    #end
    
    //define upper points  
    #local i = 0;
    #while(i < points)
      
      #local theta = (i/points)*(2*pi);
      
      #local upper[i] = <rad*cos(theta), h, rad*sin(theta)>;
    
      #local i = i + 1;
    #end
    
    //make triangles
    #local i = 0;
    #while(i < points)
      #local j = i+1;
      #if(j = points) #local j = 0; #end 

      
      triangle{upper[i],lower[i],lower[j]}
      triangle{upper[i],upper[j],lower[j]}
      
      
      #local i = i + 1;
    #end
    
    //set lower to upper
    #local i = 0;
    #while( i < points)
      
      #local lower[i] = upper[i];
      
      #local i = i + 1;
    #end
    
    #local h = h + step;
  #end
  
  finish{ambient AMBIENT diffuse 1.0 specular LUSTER roughness 0.05}                                                  
}

#end //Make_Sumac_Berry()
                           
                           
                           
                           
                           
                           
                           




 
 
  //                                                          \\
 //----------------------MAKE SUMAC BERRY BUNCH----------------\\
//                                                              \\

// This macro is called by the Make_Sumac() macro.  It is used to create large clusters of sumac berries.


#macro Make_Sumac_Berry_Bunch(START, END, HEIGHT, BASE_RADIUS, BERRY_RADIUS, COLOR_1, COLOR_2, C_VAR, FILTER, SEED)

/**********   VARIABLES   **************\
START -        <x,y,z> - base point of the berries
END -          <x,y,z> - point used to determine the direction in which the bunch should slant, as compared with START
HEIGHT -       Float -   height of the berry bunch
BASE_RADIUS -  Float -   radius of the berry bunch at its base
BERRY_RADIUS - Float -   radius of the berries (used to determine how far to space the berries from one another)
COLOR_1 -      rgb -     primary color of the berries (usually red)
COLOR_2 -      rgb -     secondary color of the berries (usually green or possibly yellow)
C_VAR -        0-1 -     determines how much the color of an individual berry in the bunch may diviate from the primary color toward the secondary color
FILTER -       0-1 -     filtered light value of all the berries in the bunch
SEED -         Integer - random seed
\***************************************/  


#local R1 = seed(SEED);

#local d = sqrt(pow(END.x-START.x,2) + pow(END.y-START.y,2) + pow(END.z-START.z,2));

#local rx = rand(R1)*1000;
#local ry = rand(R1)*1000;
#local rz = rand(R1)*1000;

#local noise_func =
function
{
  pattern
  {
    granite
    turbulence 0.1
    
    scale 0.50
  }
}

#local h = 0;
#local h_step = BERRY_RADIUS * 1.20;

#local x_bend = -1*(rand(R1)*15 + 25);
       

#if(d = 0)
  #local x_weight = 0;
  #local y_weight = 1;
  #local z_weight = 0;
#else
  #local x_weight = (END.x-START.x)/d;
  #local y_weight = (END.y-START.y)/d;
  #local z_weight = (END.z-START.z)/d;
#end

union
{
  #while(h < HEIGHT)
   
    #local h_ratio = h/HEIGHT;
    
    #local rad = (1-h_ratio)*BASE_RADIUS;
    #if(h_ratio < 0.20)
      #local rad = (h_ratio+h_step)/0.20 * BASE_RADIUS*0.80;// + rand(R1)*BERRY_RADIUS*4;
    #end
 
    
    #local theta = 0;
    #local theta_step = BERRY_RADIUS/rad * pi * 0.35;
    
    #while(theta < 2*pi)
      
      #if(h_ratio < 0.90)
        #local x_val = h*x_weight*(rand(R1)*0.10 + 0.90) + sin(theta)*rad*(rand(R1)*0.10+0.90)*(noise_func(rx+h_ratio*1,ry,rz) + 0.5) + rand(R1)*BERRY_RADIUS*1;
        #local z_val = h*z_weight*(rand(R1)*0.10 + 0.90) + cos(theta)*rad*(rand(R1)*0.10+0.90)*(noise_func(rx+h_ratio*1,ry,rz) + 0.5) + rand(R1)*BERRY_RADIUS*1;
        #local y_val = h*y_weight*(rand(R1)*0.10 + 0.90) + rand(R1)*BERRY_RADIUS*1;
      #else
        #local x_val = h*x_weight + sin(theta)*rad;
        #local z_val = h*z_weight + cos(theta)*rad;
        #local y_val = h*y_weight;
      #end  
             
      #local C_Var = rand(R1)*C_VAR;
      #local Color = rgb<COLOR_1.red*(1-C_Var) + COLOR_2.red*C_Var,  COLOR_1.green*(1-C_Var) + COLOR_2.green*C_Var, COLOR_1.blue*(1-C_Var) + COLOR_2.blue*C_Var>;
      
      object
      {
        SUMAC_BERRY    
        
        //randomly color and scale each berry in the bunch
        pigment{color Color filter FILTER}
        
        scale rand(R1)*0.50 + 0.50
        
        translate <x_val, y_val, z_val>
        rotate z*x_bend*h_ratio
        translate START
      }
      
      #local theta = theta + theta_step;
      
    #end
    
    #local h = h + h_step;
  
  #end
}

#undef noise_func


#end //Make_Sumac_Berry_Bunch()










#declare SUMAC_STEM_TEXTURE =
texture
{
  pigment{rgb<0.5,0.3,0.0>}


  
  finish
  {
    ambient 0.00 
    diffuse 0.8 
    specular 0.4 
    roughness 0.4 
  } 
}


#declare SUMAC_BRANCH_TEXTURE =
texture
{
  pigment{rgb<0.6,0.3,0.3>}
  
  finish
  {
    ambient 0.00 
    diffuse 0.8 
    specular 0.4 
    roughness 0.4 
  } 
}



//LEAF WRINKLE PATTERN
#declare SUMAC_LEAF_PATTERN = 
function
{ 
  pattern
  { 
    agate
     
    lambda 0.8
    octaves 3
    omega 5  
    turbulence 0.1

     
    scale 12
    translate y*45477
  } 
}